import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F
import numpy as np
import random


def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True)


def conv_init(m):
    classname = m.__class__.__name__
    if classname.find('Conv') != -1:
        init.xavier_uniform_(m.weight, gain=np.sqrt(2))
        init.constant_(m.bias, 0)
    elif classname.find('BatchNorm') != -1:
        init.constant_(m.weight, 1)
        init.constant_(m.bias, 0)


class WideBasic(nn.Module):
    def __init__(self, in_planes, planes, dropout_rate, stride=1):
        super(WideBasic, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes, momentum=0.1)
        self.conv1 = nn.Conv2d(
            in_planes, planes, kernel_size=3, padding=1, bias=True)
        self.dropout = nn.Dropout(p=dropout_rate)
        self.bn2 = nn.BatchNorm2d(planes, momentum=0.1)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3,
                               stride=stride, padding=1, bias=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1,
                          stride=stride, bias=True),
            )

    def forward(self, x):
        out = self.dropout(self.conv1(F.relu(self.bn1(x))))
        out = self.conv2(F.relu(self.bn2(out)))
        out += self.shortcut(x)

        return out


class WideResNet(nn.Module):
    def __init__(self, depth, widen_factor, dropout_rate, num_classes, datamixer):
        super(WideResNet, self).__init__()
        self.in_planes = 16

        assert ((depth - 4) % 6 == 0), 'Wide-resnet depth should be 6n+4'
        n = int((depth - 4) / 6)
        k = widen_factor

        nStages = [16, 16*k, 32*k, 64*k]

        self.conv1 = conv3x3(3, nStages[0])
        self.layer1 = self._wide_layer(
            WideBasic, nStages[1], n, dropout_rate, stride=1)
        self.layer2 = self._wide_layer(
            WideBasic, nStages[2], n, dropout_rate, stride=2)
        self.layer3 = self._wide_layer(
            WideBasic, nStages[3], n, dropout_rate, stride=2)
        self.bn1 = nn.BatchNorm2d(nStages[3], momentum=0.1)
        self.fc = nn.Linear(nStages[3], num_classes)

        self.datamixer = datamixer
        self.apply(conv_init)

    def _wide_layer(self, block, planes, num_blocks, dropout_rate, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, dropout_rate, stride))
            self.in_planes = planes

        return nn.Sequential(*layers)

    # def extract_multi_layer_feature(self, x):
    #     out = self.layer1(x)
    #     out = self.layer2(out)
    #     out2 = F.adaptive_avg_pool2d(out, (1, 1))
    #     out2 = out2.view(out2.size(0), -1)

    #     out = self.layer3(out)
    #     out = F.relu(self.bn1(out))
    #     out = F.adaptive_avg_pool2d(out, (1, 1))
    #     out = out.view(out.size(0), -1)

    def forward(self, x, target=None, mixing=False, mix_mode=None,  mixup_alpha=0.1, layer_mix=None):

        if mixing:
            assert mix_mode is not None
            if mix_mode == 'input':
                layer_mix = 0
            elif mix_mode == 'manifold':
                if layer_mix is None:
                    layer_mix = random.randint(0, 2)
            elif mix_mode == 'camix':
                pass
            else:
                pass
                print('Unknown mix_mode')

            out = x

            if layer_mix == 0:
                out, y_a, y_b, lam = self.datamixer.mixup_data(
                    out, target, mixup_alpha, img=x, layer_mix=layer_mix)
                # print (out)

            out = self.conv1(out)
            out = self.layer1(out)

            if layer_mix == 1:
                out, y_a, y_b, lam = self.datamixer.mixup_data(
                    out, target, mixup_alpha, img=x, layer_mix=layer_mix)
            # print (out)

            out = self.layer2(out)

            if layer_mix == 2:
                out, y_a, y_b, lam = self.datamixer.mixup_data(
                    out, target, mixup_alpha, img=x, layer_mix=layer_mix)
            # print (out)

            out = self.layer3(out)

            if layer_mix == 3:
                out, y_a, y_b, lam = self.datamixer.mixup_data(
                    out, target, mixup_alpha, img=x, layer_mix=layer_mix)
            # print (out)

            out = F.relu(self.bn1(out))
            # out = F.avg_pool2d(out, 8)
            if layer_mix == 4:
                out, y_a, y_b, lam = self.datamixer.mixup_data(
                    out, target, mixup_alpha, img=x, layer_mix=layer_mix)

            out = F.adaptive_avg_pool2d(out, (1, 1))
            out = out.view(out.size(0), -1)
            out = self.fc(out)

            return out, y_a, y_b, lam
        else:
            out = self.conv1(x)
            out = self.layer1(out)
            out = self.layer2(out)
            out = self.layer3(out)
            out = F.relu(self.bn1(out))
            out = F.adaptive_avg_pool2d(out, (1, 1))
            out = out.view(out.size(0), -1)
            out = self.fc(out)

            return out

    def premix(self, x, out_layer=3):
        out = x
        out = self.conv1(out)
        out = self.layer1(out)
        if out_layer == 1:
            return out
        out = self.layer2(out)
        if out_layer == 2:
            return out
        out = self.layer3(out)
        out = F.relu(self.bn1(out))
        return out


def wrn40_2(dropout_rate, num_class, datamixer):
    model = WideResNet(depth=40,
                       widen_factor=2,
                       dropout_rate=dropout_rate,
                       num_classes=num_class,
                       datamixer=datamixer)
    return model


def wrn28_10(dropout_rate, num_class, datamixer):
    model = WideResNet(depth=28,
                       widen_factor=10,
                       dropout_rate=dropout_rate,
                       num_classes=num_class,
                       datamixer=datamixer)
    return model
