% Authors: Oscar Lopez
% Institute: Harbor Branch Oceanographic Institute
% Date: March 23 2023

%% Poisson Tensor Completion: Varying SG
% This script sets up tensor completion problems with varying spectral gap
% in the distribution of the observed entries. In the experiments, we 
% randomly generate a low-rank parameter tensor X and a random tensor Y 
% whose entries are i.i.d. Poisson random variables with means given by the 
% respective entries of X. We then generate a set of observed tensor 
% entries "Omega" in a periodic manner. A percentage of the periodic 
% samples are "swapped" uniformly at random with the unobserved entries.
% Swapping larger percentages generally produces decresing second eigenvalues. 
% Given X and Y, we achieve tensor completion for varying percentages of swapped
% entries. We then repeat this whole proceedure with a newly generated Y 
% for a given number of "repetition" times (X remains fixed throughout).

%% Add required paths to run GCP
addpath(genpath('tensor_toolbox'));

% Experiment parameters and output initializations
n = 100;            % tensor dimensions
rank = 3;          % actual rank of data tensor
r = 3;              % rank to run cp_apr
perc = .05; % percentage of missing entries
percr = .1:.02:.99; % varying percentages of swapped entries from periodic grid
rep = 2;           % number of times to repeat experiment
% initialize matrices to store relative errors and second eigenvalues
E_ideal = zeros(rep,length(percr));
E_gap = zeros(rep,length(percr));

%% Generate random tensor with low-rank structure and entries in [alpha,alpha+1]
alpha = 1;
X = create_low_rank_param_tensor([n n n], rank, [alpha alpha+5]); % local function, see end of script

%% GCP optimization parameters
clear opts_gcp
opts_gcp.type = 'count';                             % Poisson negative log likelihood objective function
opts_gcp.opt = 'lbfgsb';                             % Limited-memory bound-constrained quasi-Newton optimizer
opts_gcp.maxiters = 3000;
opts_gcp.printitn = 1;
opts_gcp.pgtol = 1e-15;                              % stopping tolerance - gradient
opts_gcp.factr = 1e-15;                              % stopping tolerance - function value reduction

for k1 = 1:rep
    
    % Generate Poisson observations
    Yobs = tensor(poissrnd(double(full(X)))); % Poisson observations
    
    for k2 = 1:length(percr)

        % Generate periodic set of missing entries with desired sampling percentage
        st = round(1/perc); % step size according to sampling percentage
        Om = 1:st:n^3; % set of observed entries
        OmC = setdiff(1:n^3,Om); % unobserved entries
        
        % Swap desired percentage of observed entries uniformly at random
        % with unobserved entries
        IndP = randperm(length(Om));
        IndP = IndP(1:round(length(Om)*percr(k2))); % observed entries to swap
        IndPC = randperm(length(OmC));
        IndPC = IndPC(1:round(length(Om)*percr(k2))); % unobserved entries to swap       
        Om(IndP) = OmC(IndPC); % swap entries
        OmC = setdiff(1:n^3,Om); % re-define unobserved entries according to swap

        % Poisson tensor completion
        W2 = ones([n n n]);    % create an indicator tensor for observed entries, all ones
        W2(OmC) = 0;           % remove unobserved entries
        W2 = tensor(W2);
        M_ideal = gcp_opt(Yobs,r,opts_gcp,'mask',W2); % run Poisson tensor completion
        E_ideal(k1,k2) = norm(X-M_ideal)/norm(X);  % compute relative error

        % compute second eigenvalue
        R1 = nnz(W2)*ones(size(W2))*n^(-3); % scaled all ones tensor
        R2 = double(cp_als(W2-R1,1)); % rank-1 approx of adjecency tensor-R1
        E_gap(k1,k2) = norm(R2(:)); % second eigenvalue is Frob norm of R2

    end
end

% Save important outputs as .mat file
save('results_rank3.mat','perc','E_ideal','E_gap')

% plot results and save figure
E_ideal = E_ideal(:);
E_gap = E_gap(:);
p = scatter(E_gap,E_ideal);
xlabel('\lambda_2') 
ylabel('Generalization Error')
title(['Poisson Tensor Completion: rank = ' num2str(rank)])
p(1).LineWidth = 2;
ax = gca;
ax.FontSize = 14;
saveas(gcf,'PTC_varyingSG','png')

%% Local Function, to generate low-rank parameter tensor
function X = create_low_rank_param_tensor(sz,r,paramRange)
    nd = length(sz);
    factorRange = paramRange.^(1/nd)/r^(1/nd);   
    info = create_problem('Size', sz, ...
        'Num_Factors', r, ...
        'Factor_Generator', @(m,n)(factorRange(1)+(rand(m,n))*(factorRange(2)-factorRange(1))), ...
        'Lambda_Generator', @(m,n)ones(m,1), ...
        'Noise', 0);

    X = normalize(arrange(info.Soln));
end