import numpy as np

class _DataStream:
    def __init__(self, batch_size = 1):
        self.batch_size = batch_size

    def pop(self):
        return self._generator()

    def _generator(self):
        raise NotImplementedError


class RandomWalk(_DataStream):
    def __init__(self, batch_size = 1, poly_order = 1, params = None):
        super().__init__(batch_size = batch_size)
        self.current_state = 0.0

        if params is None:
            # compute parameters of Pareto distribution when `params` is not provided
            assert isinstance(poly_order, int)
            self.params = {"alpha": poly_order+2, "x_m": 1.0}
        else:
            self.params = params

    def _generator(self):
        # generate a mini-batch of data
        data = []
        for _ in range(self.batch_size):
            w = (np.random.pareto(self.params["alpha"] ) - (self.params["alpha"]+1.0)/(self.params["alpha"]-1.0)  ) * self.params["x_m"]
            self.current_state = np.max([0.0, self.current_state + w])
            data.append(self.current_state)
        return np.array(data)


class MCMC(_DataStream):
    def __init__(self, batch_size = 1, dim = 1, r=1):
        super().__init__(batch_size = batch_size)
        self.dim = dim
        self.current_state = np.random.uniform(0,1,dim)
        self.params = {"r": r}

    def _generator(self):
        r = self.params["r"]
        dim = self.dim

        def q(x):
            return (r + 1) * x**r

        # generate a mini-batch of data
        data = []
        for _ in range(self.batch_size):
            #  sample next state
            y = np.random.beta(r+1, 1, dim)

            # compute acceptance probability
            alpha = q(self.current_state)/q(y)
            alpha[alpha > 1] = 1.0

            # determine if y[i] is accepted or not
            prob = np.random.uniform(0,1,dim)
            next_state = np.zeros(dim)
            next_state[prob < alpha] = y[prob < alpha]
            next_state[prob >= alpha] = self.current_state[prob >= alpha]

            data.append(next_state)
            self.current_state = next_state
        return np.array(data)

class DataStream(_DataStream):
    def __init__(self, dataset, batch_size = 1):
        super().__init__(batch_size = batch_size)
        self.current_index = 0
        self.dataset = dataset
        self.dim = len(dataset[0])

    def _generator(self):
        next_index = self.current_index + self.batch_size
        if next_index <= len(self.dataset):
            data = self.dataset[self.current_index: next_index]
            self.current_index = next_index
            return np.array(data)
        else:
            return None





