from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
import xml.etree.ElementTree as ET
import os

import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env


class CartpoleEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    PENDULUM_LENGTH = 0.6
    def __init__(self, random_init=True, PENDULUM_LENGTH=0.6):
        CartpoleEnv.PENDULUM_LENGTH = PENDULUM_LENGTH
        utils.EzPickle.__init__(self)
        dir_path = os.path.dirname(os.path.realpath(__file__))
        self.set_pole_length()
        mujoco_env.MujocoEnv.__init__(self, '%s/assets/cartpole.xml' % dir_path, 2)
        self.random_init = random_init

    def step(self, a):
        self.do_simulation(a, self.frame_skip)
        ob = self._get_obs()

        cost_lscale = CartpoleEnv.PENDULUM_LENGTH
        reward = np.exp(
            -np.sum(np.square(self._get_ee_pos(ob) - np.array([0.0, CartpoleEnv.PENDULUM_LENGTH]))) / (cost_lscale ** 2)
        )
        reward -= 0.01 * np.sum(np.square(a))

        done = False
        return ob, reward, done, {}

    def reset_model(self):
        if self.random_init:
            qpos = self.init_qpos + np.random.normal(0, 0.1, np.shape(self.init_qpos))
            qvel = self.init_qvel + np.random.normal(0, 0.1, np.shape(self.init_qvel))
        else:
            qpos = self.init_qpos
            qvel = self.init_qvel
        self.set_state(qpos, qvel)
        return self._get_obs()

    def _get_obs(self):
        return np.concatenate([self.sim.data.qpos, self.sim.data.qvel]).ravel()

    @staticmethod
    def _get_ee_pos(x):
        x0, theta = x[0], x[1]
        return np.array([
            x0 - CartpoleEnv.PENDULUM_LENGTH * np.sin(theta),
            -CartpoleEnv.PENDULUM_LENGTH * np.cos(theta)
        ])
    @staticmethod
    def set_pole_length():
        dir_path = os.path.dirname(os.path.realpath(__file__))
        tree = ET.parse("%s/assets/cartpole.xml" % dir_path)
        world = tree.findall("worldbody")[0]
        cart = world.findall("body")[0]
        pole = cart.findall("body")[0]
        pole_geometry = pole.findall("geom")[0]

        pole_geometry.set("fromto", f"0 0 0 0.001 0 -{CartpoleEnv.PENDULUM_LENGTH}")
        tree.write("%s/assets/cartpole.xml" % dir_path)

    def viewer_setup(self):
        v = self.viewer
        v.cam.trackbodyid = 0
        #v.cam.distance = v.model.stat.extent
        v.cam.distance = self.model.stat.extent


    # def viewer_setup(self):
    #     self.viewer.cam.distance = self.model.stat.extent * 0.25
    #     self.viewer.cam.elevation = -55
