import pickle as pkl
import pandas
import numpy as np
import io
import os
import matplotlib.pyplot as plt
import rosbag
from PIL import Image
import argparse

from notebooks.process_data_utils import process_images, process_odom, process_img_func

robotnames = ["tartan", "uw"]
imtopics = [
    "/multisense/left/image_rect_color",
    "/uwarthog/camera/image_color/compressed",
]
odomtopics = ["", "/uwarthog/odometry/filtered"]

# # Parse arguments
parser = argparse.ArgumentParser()
parser.add_argument("--robot", type=str, default="spot", help="Robot name")
parser.add_argument("--idx", type=int, default=0, help="Bag ID")
args = parser.parse_args()

robotname = args.robot
idx = args.idx
assert robotname in robotnames, f"Robot name {robotname} not recognized"

datadir = f"/media/barry/multirobot/{robotname}"
csvfile = f"{robotname}.csv"
imtopic = imtopics[robotnames.index(robotname)]
odomtopic = odomtopics[robotnames.index(robotname)]

df = pandas.read_csv(os.path.join(datadir, csvfile), header=None)
filenames = df.values
filenames = [name[0] for name in filenames]
bagname = filenames[idx]

# open bag
bag = rosbag.Bag(os.path.join(datadir, bagname))

# check if bag has both topics
if not bag.get_message_count(imtopic) or not bag.get_message_count(odomtopic):
    print("bag does not have both topics")
    exit()
# get image and odom data from bags
synced_imdata = []
synced_odomdata = []

rate = 4.0

# get start time of bag in seconds
currtime = bag.get_start_time()
starttime = currtime

curr_imdata = None
curr_odomdata = None
times = []

for topic, msg, t in bag.read_messages(topics=[imtopic, odomtopic]):
    if topic == imtopic:
        curr_imdata = msg
    elif topic == odomtopic:
        curr_odomdata = msg
    if (t.to_sec() - currtime) >= 1.0 / rate:
        if curr_imdata is not None and curr_odomdata is not None:
            synced_imdata.append(curr_imdata)
            synced_odomdata.append(curr_odomdata)
        else:
            print("missing data")
        currtime = t.to_sec()
        times.append(currtime - starttime)



# decompress images
synced_images = process_images(synced_imdata, process_img_func[robotname])

# resize image to X by 120, preserving aspect ratio
synced_images_resized = [
    im.resize((int(120 * im.size[0] / im.size[1]), 120)) for im in synced_images
]

# process odom data
trajdata = process_odom(synced_odomdata)
savedir = f"multirobot/test_process/{robotname}"
bagdir = robotname + "_" + bagname.split(".")[0]
if os.path.exists(os.path.join(savedir, bagdir)):
    print("bag already processed")
    # exit()
else:
    os.makedirs(os.path.join(savedir, bagdir))

    # save images as jpegs
    for i, im in enumerate(synced_images):
        im.save(os.path.join(savedir, bagdir, str(i) + ".jpg"))
    print("saved images")
    # save trajdata as a pickle file
    with open(os.path.join(savedir, bagdir, "traj_data.pkl"), "wb") as f:
        pkl.dump(trajdata, f)
    print("saved trajdata")

