import argparse
from transformers import AutoModelForCausalLM, AutoTokenizer
from data_process.data_factory import data_provider
from utils.tools import EarlyStopping, adjust_learning_rate, visual
from utils.metrics import metric
import torch
import torch.nn as nn
from torch import optim
import os
import time
import warnings
import numpy as np
from model import get_model_class
import random
warnings.filterwarnings('ignore')
import json
from math import sqrt
import matplotlib.pyplot as plt
from models.model import TITSP

global_predict_len = 96

llm_model = AutoModelForCausalLM.from_pretrained(
    "Qwen/Qwen2-1.5B-Instruct",
    torch_dtype="auto",
    device_map="auto"
)
tokenizer = AutoTokenizer.from_pretrained("Qwen/Qwen2-1.5B-Instruct")
word_embeds = llm_model.get_input_embeddings().weight.to(llm_model.dtype) 
class Exp_Long_Term_Forecast():
    def __init__(self, args):
        self.args = args
        self.device = "cuda:0" if torch.cuda.is_available() else "cpu"
        self.top_k = 5
        self.seq_len = args.seq_len
        self.pred_len = args.pred_len
        self._build_model(args)
        
    def _build_model(self, args):

        with open(args.config_path, 'r') as f:
            vqvae_config = json.load(f)["vqvae_config"]
        if 'general_seed' not in vqvae_config:
            vqvae_config['seed'] = random.randint(0, 9999)
        model_class = get_model_class(vqvae_config['model_name'].lower())
        vqvae = model_class(vqvae_config, word_embeds)  # Initialize model
        vqvae_model_path = args.vqvae_model_path
        vqvae = torch.load(vqvae_model_path)
        vqvae.eval()
        self.vqvae = vqvae
        self.model = TITSP(args).to(self.device)  
        print(self.model)
    def calcute_lags(self, x_enc):
        q_fft = torch.fft.rfft(x_enc.permute(0, 2, 1).contiguous(), dim=-1)
        k_fft = torch.fft.rfft(x_enc.permute(0, 2, 1).contiguous(), dim=-1)
        res = q_fft * torch.conj(k_fft)
        corr = torch.fft.irfft(res, dim=-1)
        mean_value = torch.mean(corr, dim=1)
        _, lags = torch.topk(mean_value, self.top_k, dim=-1)
        return lags
    def _get_data(self, flag):
        data_set, data_loader = data_provider(self.args, flag)
        return data_set, data_loader

    def _select_optimizer(self):
        model_optim = optim.Adam(self.model.parameters(), lr=self.args.learning_rate)
        return model_optim

    def _select_criterion(self):
        #rion = nn.MSELoss()
        criterion = nn.L1Loss()
        return criterion

    def vali(self, vali_data, vali_loader, criterion):
        total_loss = []
        self.model.eval()
        with torch.no_grad():
            iter_count = 0
            for i, (batch_x, batch_y, order, _) in enumerate(vali_loader):
                try:
                    bs, n_vars = batch_x.size(0),batch_x.size(1)
                    batch_x = torch.reshape(batch_x, (bs*n_vars, args.seq_len, -1))
                    batch_y = torch.reshape(batch_y, (bs*n_vars, self.args.pred_len+self.args.label_len, -1))     
                    order = np.reshape(order,(bs*n_vars,-1))
                except Exception as e:
                    print('e', e)
                    bs, n_vars = batch_x.size(0), 1
                    batch_x = batch_x.unsqueeze(2)
                    batch_y = batch_y.unsqueeze(2)
                #continue
                iter_count += 1
                batch_x = batch_x.float().to(self.device)
                batch_y = batch_y.float().to(self.device)
                loss, vq_loss, recon_error, x_recon, perplexity, embedding_weight, encoding_indices, encodings, embeds = \
                                self.vqvae.shared_eval(batch_x, None, 'val', comet_logger=None)
                # encoder - decoder   torch.Size([128, 128, 24])
                embeds = embeds.to(llm_model.dtype)  
                # torch.Size([7, 1536, 24])
                x_enc = batch_x
                prompt = []
                for b in range(x_enc.shape[0]):
                    prompt_ = (
                        f"Task description: forecast the next {str(self.pred_len)} steps"
                        f"Follow the order of {order[b][0]}"
                    )
                    prompt.append(prompt_)

                prompt = tokenizer(prompt, return_tensors="pt", padding=True, truncation=True, max_length=2048).input_ids # (batch, prompt_token)
                llm_model.to(x_enc.device)
                prompt_embeddings = llm_model.get_input_embeddings()(prompt.to(x_enc.device))  # (batch, prompt_token, dim)   # torch.Size([56, 256, 1536])

                outputs = self.model(prompt_embeddings,embeds)
                outputs = outputs.unsqueeze(2)
                outputs = self.vqvae.revin_layer(outputs, 'denorm')  
                f_dim = -1 if self.args.features == 'MS' else 0
                
                outputs = outputs[:, -self.args.pred_len:, f_dim:].detach().cpu()
                batch_y = batch_y[:, -self.args.pred_len:, f_dim:].to(self.device).detach().cpu()      
                loss = criterion(outputs, batch_y)
                total_loss.append(loss)
        total_loss = np.average(total_loss)
        self.model.train()
        return total_loss

    def train(self, setting):
        total_params = sum(p.numel() for p in self.model.parameters() if p.requires_grad)
        print(f"Total Trainable Parameters: {total_params}")
        train_data, train_loader = self._get_data(flag='train')
        vali_data, vali_loader = self._get_data(flag='val')
        test_data, test_loader = self._get_data(flag='test')

        path = os.path.join(self.args.checkpoints, setting)
        if not os.path.exists(path):
            os.makedirs(path)

        time_now = time.time()
        train_steps = len(train_loader)
        early_stopping = EarlyStopping(patience=self.args.patience, verbose=True)

        model_optim = self._select_optimizer()
        criterion = self._select_criterion()

        for epoch in range(self.args.train_epochs):
            iter_count = 0
            train_loss = []

            self.model.train()
            epoch_time = time.time()
            for i, (batch_x, batch_y, order, _) in enumerate(train_loader):
                try:
                    bs, n_vars = batch_x.size(0),batch_x.size(1)
                    #print('atch_x.shape:',batch_x.shape)
                    batch_x = torch.reshape(batch_x, (bs*n_vars, args.seq_len, -1))
                    #print(2)
                    #print('atch_y.shape:',batch_y.shape)
                    batch_y = torch.reshape(batch_y, (bs*n_vars, self.args.pred_len+self.args.label_len, -1))     
                    order = np.reshape(order,(bs*n_vars,-1 ))
                except Exception as e:
                    print('e', e)
                    bs, n_vars = batch_x.size(0), 1
                    batch_x = batch_x.unsqueeze(2)
                    batch_y = batch_y.unsqueeze(2)
                #continue
                iter_count += 1
                model_optim.zero_grad()
                batch_x = batch_x.float().to(self.device)
                batch_y = batch_y.float().to(self.device)
                loss, vq_loss, recon_error, x_recon, perplexity, embedding_weight, encoding_indices, encodings, embeds = \
                                self.vqvae.shared_eval(batch_x, None, 'val', comet_logger=None)
                # encoder - decoder   torch.Size([128, 128, 24])
                embeds = embeds.to(llm_model.dtype)  
                # torch.Size([7, 1536, 24])
                x_enc = batch_x
                prompt = []
                for b in range(x_enc.shape[0]):
                    prompt_ = (
                        f"Task description: forecast the next {str(self.pred_len)} steps"
                        f"Follow the order of {order[b][0]}"
                    )
                    prompt.append(prompt_)
                    #print(order[b][0])
                prompt = tokenizer(prompt, return_tensors="pt", padding=True, truncation=True, max_length=2048).input_ids # (batch, prompt_token)
                llm_model.to(x_enc.device)
                # prompt->语言的tokenizer直接获取embedding
                prompt_embeddings = llm_model.get_input_embeddings()(prompt.to(x_enc.device))  # (batch, prompt_token, dim)   # torch.Size([56, 256, 1536])

                outputs = self.model(prompt_embeddings,embeds)

                outputs = outputs.unsqueeze(2)
                outputs = self.vqvae.revin_layer(outputs, 'denorm') 
                f_dim = -1 if self.args.features == 'MS' else 0
                outputs = outputs[:, -self.args.pred_len:, f_dim:]
                batch_y = batch_y[:, -self.args.pred_len:, f_dim:]
                loss = criterion(outputs, batch_y)
                train_loss.append(loss.item())

                if (i + 1) % 100 == 0:
                    print("\titers: {0}, epoch: {1} | loss: {2:.7f}".format(i + 1, epoch + 1, loss.item()))
                    speed = (time.time() - time_now) / iter_count
                    left_time = speed * ((self.args.train_epochs - epoch) * train_steps - i)
                    print('\tspeed: {:.4f}s/iter; left time: {:.4f}s'.format(speed, left_time))
                    iter_count = 0
                    time_now = time.time()

                loss.backward(retain_graph=True)
                model_optim.step()

            print("Epoch: {} cost time: {}".format(epoch + 1, time.time() - epoch_time))
            train_loss = np.average(train_loss)
            vali_loss = self.vali(vali_data, vali_loader, criterion)
            test_loss = self.vali(test_data, test_loader, criterion)

            print("Epoch: {0}, Steps: {1} | Train Loss: {2:.7f} Vali Loss: {3:.7f} Test Loss: {4:.7f}".format(
                epoch + 1, train_steps, train_loss, vali_loss, test_loss))
            early_stopping(vali_loss, self.model, path)
            if early_stopping.early_stop:
                print("Early stopping")
                break

            adjust_learning_rate(model_optim, epoch + 1, self.args)

        best_model_path = path + '/' + 'checkpoint.pth'
        self.model.load_state_dict(torch.load(best_model_path))

        return self.model

    def test(self, setting, test=0):
        test_data, test_loader = self._get_data(flag='test')
        if test:
            print('loading model')
            self.model.load_state_dict(torch.load(os.path.join('./model_ts/' + setting, 'checkpoint.pth')))

        preds = []
        trues = []
        folder_path = './test_results/' + setting + '/'
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

        self.model.eval()
        with torch.no_grad():
            for i, (batch_x, batch_y, order, _) in enumerate(test_loader):
                try:
                    bs, n_vars = batch_x.size(0),batch_x.size(1)
                    batch_x = torch.reshape(batch_x, (bs*n_vars, args.seq_len, -1))
                    batch_y = torch.reshape(batch_y, (bs*n_vars, self.args.pred_len+self.args.label_len, -1))     
                    order = np.reshape(order,(bs*n_vars,-1))
                except Exception as e:
                    print('e', e)
                    bs, n_vars = batch_x.size(0), 1
                    batch_x = batch_x.unsqueeze(2)
                    batch_y = batch_y.unsqueeze(2)
                batch_x = batch_x.float().to(self.device)
                batch_y = batch_y.float().to(self.device)
                loss, vq_loss, recon_error, x_recon, perplexity, embedding_weight, encoding_indices, encodings, embeds = \
                                self.vqvae.shared_eval(batch_x, None, 'val', comet_logger=None)
                # encoder - decoder   torch.Size([128, 128, 24])
                embeds = embeds.to(llm_model.dtype)   
                # torch.Size([7, 1536, 24])
                x_enc = batch_x
                prompt = []
                for b in range(x_enc.shape[0]):
                    prompt_ = (
                        f"Task description: forecast the next {str(self.pred_len)} steps"
                        f"Follow the order of {order[b][0]}"
                    )
                    prompt.append(prompt_)
                        
                prompt = tokenizer(prompt, return_tensors="pt", padding=True, truncation=True, max_length=2048).input_ids # (batch, prompt_token)
                llm_model.to(x_enc.device)
                # prompt->语言的tokenizer直接获取embedding
                prompt_embeddings = llm_model.get_input_embeddings()(prompt.to(x_enc.device))  # (batch, prompt_token, dim)   # torch.Size([56, 256, 1536])

                outputs = self.model(prompt_embeddings,embeds)
                outputs = outputs.unsqueeze(2)
                outputs = self.vqvae.revin_layer(outputs, 'denorm')   
                # 这个denorm中的数据，均值方差是由batch_x来的
                f_dim = -1 if self.args.features == 'MS' else 0
                outputs = outputs[:, -self.args.pred_len:, :]
                batch_y = batch_y[:, -self.args.pred_len:, :].to(self.device)
                outputs = outputs.detach().cpu()
                batch_y = batch_y.detach().cpu()
                if test_data.scale and self.args.inverse:
                    shape = outputs.shape
                    outputs = test_data.inverse_transform(outputs.squeeze(0)).reshape(shape)
                    batch_y = test_data.inverse_transform(batch_y.squeeze(0)).reshape(shape)
        
                outputs = outputs[:, :, f_dim:]
                batch_y = batch_y[:, :, f_dim:]

                pred = outputs
                true = batch_y
                # print('pred.shape', outputs.shape)
                # print('trues.shape', batch_y.shape)
                
                preds.append(pred)
                trues.append(true)
                if i % 2 == 0:
                    input = batch_x.detach().cpu().numpy()
                    transposed_arr = np.reshape(input, (bs, n_vars, -1))
                    input = np.transpose(transposed_arr, axes=(0, 2, 1))

                    truei = true.detach().cpu().numpy()
                    transposed_arr = np.reshape(truei, (bs, n_vars, -1))
                    truei = np.transpose(transposed_arr, axes=(0, 2, 1))
                    
                    predi = pred.detach().cpu().numpy()
                    transposed_arr = np.reshape(predi, (bs, n_vars, -1))
                    predi = np.transpose(transposed_arr, axes=(0, 2, 1))
                    if test_data.scale and self.args.inverse:
                        shape = input.shape
                        input = test_data.inverse_transform(input.squeeze(0)).reshape(shape)
                    gt = np.concatenate((input[0, :, -1], truei[0, :, -1]), axis=0)
                    pd = np.concatenate((input[0, :, -1], predi[0, :, -1]), axis=0)
                    visual(gt, pd, os.path.join(folder_path, str(i) +'_'+ order[0][0]+'_'+'.pdf'))

        preds = torch.cat(preds, dim=0).numpy()#np.array(preds)
        trues = torch.cat(trues, dim=0).numpy()#np.array(trues)
        
        preds = preds.reshape(-1, preds.shape[-2], preds.shape[-1])
        trues = trues.reshape(-1, trues.shape[-2], trues.shape[-1])
        print('test shape:', preds.shape, trues.shape)

        # result save
        folder_path = './test_results/'+'./results/' + setting + '/'
        if not os.path.exists(folder_path):
            os.makedirs(folder_path)

        mae, mse, rmse, mape, mspe = metric(preds, trues)
        print('mse:{}, mae:{}'.format(mse, mae))
        f = open(folder_path +"result_long_term_forecast.txt", 'a')
        f.write(setting + "  \n")
        f.write('mse:{}, mae:{}'.format(mse, mae))
        f.write('\n')
        f.write('\n')
        f.close()

        np.save(folder_path + 'metrics.npy', np.array([mae, mse, rmse, mape, mspe]))
        np.save(folder_path + 'pred.npy', preds)
        np.save(folder_path + 'true.npy', trues)
        return 