import pandas as pd
import numpy as np
import torch


def scale_dataframe(df: pd.DataFrame, seperate: bool = True) -> pd.DataFrame:
    """
    Scale a pandas DataFrame to range [-1, 1] using log scaling.
    If seperate=True, positive and negative values are scaled seperately.
    """
    # Convert to float as log1p fails for other dtypes, e.g. int
    df = df.astype(float)
    if seperate:
        df_pos = df[df > 0]
        df_neg = df[df < 0]
        df_zero = df[df == 0]

        # Apply log scaling for positive values
        df_pos_log = np.log1p(df_pos)
        df_pos_scaled = df_pos_log / df_pos_log.max(axis=None)

        # Apply log scaling for negative values
        df_neg_log = -np.log1p(df_neg.abs())
        df_neg_scaled = -(df_neg_log / df_neg_log.min(axis=None))
        return df_pos_scaled.combine_first(df_neg_scaled).combine_first(df_zero)
    else:
        df_normalized = np.sign(df) * np.log1p(np.abs(df))
        return df_normalized / (np.max(np.abs(df_normalized)) + 1e-12)


def flatten_image(image: torch.Tensor) -> torch.Tensor:
    "Flatten image to 1D Tensor."

    return image[0].view(-1)


def flatten_image_maintain_dim(image: torch.Tensor) -> torch.Tensor:
    "Flatten image to Tensor, but maintain dimensionality."

    return image.view([1, 1, 32 * 32])


def get_device(device_id: int):
    # Check if CUDA is available
    if torch.cuda.is_available():
        if device_id == None:
            print("WARNING: No device ID given!")
            device = torch.device("cuda")
        else:
            # Use the provided device ID to select the GPU
            device = torch.device(f"cuda:{device_id}")
    else:
        # Fall back to CPU
        device = torch.device("cpu")
        if device_id:
            device = torch.device(f"cpu:{device_id}")
    # Prevent memory overflow
    torch.cuda.empty_cache()
    return device
