#!/usr/bin/env fish

# Define the base directory containing the programs
set base_dir plp/programs

# Loop through each directory in the base directory
for dir in (ls -d $base_dir/*/)
    # Get the program name from the directory name
    set program_name (basename $dir)

    # Define a list of prefixes to check
    #set valid_prefixes coloring_ irn irl pin queens
    set valid_prefixes coloring_

    # If the program name does not start with any of the valid prefixes, skip
    set match_prefix 0
    for prefix in $valid_prefixes
        if string match -q "$prefix*" $program_name
            set match_prefix 1
            break
        end
    end

    if test $match_prefix -eq 0
        continue
    end

    # Print the name of the program
    echo "$program_name"

    # Define the input file path
    set input_file "$dir/$program_name.pasp"

    # Check if the input file exists
    if test -f $input_file
        # Parse the program and generate the CNF and JSON files
        uv run rule_parser.py $input_file

        # Define the CNF file path
        set cnf_file "$dir/$program_name.cnf"

        # Check if the CNF file was generated
        if test -f $cnf_file
            # Convert the CNF file to be compatible with SharpSAT-TD
            uv run convert_sharpsat_td.py $cnf_file
        else
            echo "CNF file not found: $cnf_file"
        end

        # Define the JSON file path
        set json_file "$dir/$program_name.json"

        # Check if the JSON file was generated
        if test -f $json_file
            # Split the rules into disjoint groups based on non-shared atoms
            # over the program without initialization heuristic
            uv run non_incremental_heuristic.py $json_file

            # Reassign the atom IDs based on the dependency graph
            uv run initialization_heuristic.py $json_file
            uv run minfill.py $json_file
            uv run mindegree.py $json_file

            # Define the initialized JSON file path
            set init_json_file (string replace -r ".json" "_init.json" $json_file)
            set minfill_json_file (string replace -r ".json" "_minfill.json" $json_file)
            set mindegree_json_file (string replace -r ".json" "_mindegree.json" $json_file)

            # Loop over init, minfill, and mindegree JSON files
            for heuristic in init minfill mindegree
                set heuristic_json_file (string replace -r ".json" "_$heuristic.json" $json_file)

                # Check if the heuristic JSON file was generated
                if test -f $heuristic_json_file
                    # Split the rules into disjoint groups based on non-shared atoms
                    # over the program with the current heuristic
                    uv run non_incremental_heuristic.py $heuristic_json_file
                else
                    echo "Heuristic JSON file not found: $heuristic_json_file"
                end
            end
        else
            echo "JSON file not found: $json_file"
        end
    else
        echo "Input file not found: $input_file"
    end
end
