import logging
from .agent_abs import Agent
from .team import CollaborationContext
import json
from utils.format_json import extract
from prompts import (
    generate_tom_response_template,
    generate_code_response_template,
    generate_score_template,
    CODE_COMPLETE_TEMPLATE
)
import ast
from tenacity import retry, stop_after_attempt, wait_fixed, retry_if_exception_type
from typing import Dict, Any, List, Optional
from dataclasses import dataclass


class ProjectManager(Agent):
    def __init__(self, id, tom_k, task):
        super().__init__(name=f"ProjectManager{id}", tom_k=tom_k, \
            default_prompts=[
                {
                "role": "system",
                "content": f"Give the programming task {task}. You are the expert project manager, you need to assign tasks to engineers to cooperate and complete the project.",
                }
            ]
        )
        self.task = task
        self.action = ""

    def generate_belief_model(self):
        teammates_names = [teammate.name for teammate in self.get_teammates()]
        tom_prompts = (
            f"Using your {self.tom_k}-Level Theory of Mind, infer the mental states of your teammates and provide task outlines as `action`:\n\n"
            "For ToM_Level0, provide initial task outlines for each engineer.\n\n"
        )
        if self.tom_k > 0:
            tom_prompts += (
                "For subsequent ToM levels (ToM_Level1 and above):\n"
                "   - (belief) Infer each engineer's possible actions based on your belief (the previous ToM level's task outlines).\n"
                "   - (explanation) Explain your thought process for inferring actions.\n"
                "   - (action) Provide updated task outlines for each engineer.\n\n"
            )
        tom_prompts += "ONLY Response in json format as follows:\n"
        tom_prompts += generate_tom_response_template(self.tom_k, teammates_names)

        return tom_prompts

    async def check_belief_alignment(self, implementations):
        if self.tom_k == 0:
            return {}
        teammate_names = [teammate.name for teammate in self.get_teammates()]

        prompts = f"You will provide your belief alignment scores for each engineer's implementation based on your belief model.\n"
        prompts += f"Engineer's implementation: {json.dumps(implementations)}\n"
        prompts += f"Your belief model: {json.dumps(self.belief_models)}\n"

        prompts += f"Instruction:\n For each engineer, provide a belief alignment score between -1 and 1. \n Respond in the following JSON format:\n```json"
        score_template = generate_score_template(teammate_names)
        prompts += score_template
        prompts += "```"

        try:
            output = await self.aask(
                prompts, use_memory=False, response_format="json_object"
            )
            output = extract(output, template=score_template)
        except Exception as e:
            self.logger.error(f"Error: {e}")
            self.logger.error(f"output: {output}")
            raise ValueError("Failed to update belief model")

        return output

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(2))
    async def run(self) -> Dict[str, str]:
        # Generate belief model
        belief_model_prompt = self.generate_belief_model()
        # print(belief_model_prompt)

        output = await self.aask(belief_model_prompt, "json_object")
        # Update belief model
        output = extract(output, template=generate_tom_response_template(self.tom_k))[
            f"ToM_level{self.tom_k}"
        ]
        self.belief_models = self.update_belief_model(output["belief"])
        self.action = output["action"]

        # log belief model, action
        self.logger.info(f"Belief Model: {self.belief_models}")
        self.logger.info(f"Action: {self.action}")

        return self.action

    async def get_final_solution(self, final_solution: Dict[str, str]) -> str:
        self.update_default_prompts(["You are the project manager, the final solution from the engineers is provided to you.", \
            "Your goal is to provide a final solution."])
        prompts = f"## Final solution from engineers: {json.dumps(final_solution['implementation'])}\n"
        prompts += CODE_COMPLETE_TEMPLATE.format(idea=final_solution["idea"])
        
        return await self.aask(prompts)

# Add new exception
class CollaborationError(Exception):
    """Raised when team collaboration issues occur"""
    pass

class TaskImplementationError(Exception):
    """Raised when task implementation fails"""
    pass

# Add to Engineer class
class Engineer(Agent):
    def __init__(self, id, task, tom_k: int = 0):
        super().__init__(name=f"Engineer{id}", tom_k=tom_k, \
            default_prompts=[
                {
                "role": "system",
                "content": (
                    f"You are expert software engineer Engineer{id}.\n"
                    f"Implement assigned tasks in Python, focusing on:\n"
                    f"- Clean, efficient code\n"
                    f"- Proper error handling\n"
                    f"- Team coordination\n"
                    f"- Best practices\n"
                    f"- Interface compatibility with teammates"
                ),
            }
        ]
        )
        self.task = task
        self.codebase = ""
        self.action = ""

    def generate_belief_model(self, task_outline: str) -> str:
        """Generate belief model prompt with team awareness"""
        teammates = self.get_teammates()
        teammates_info = []
        
        # Collect teammate information including PM
        for teammate in teammates:
            if isinstance(teammate, ProjectManager):
                teammates_info.append(f"ProjectManager: {teammate.name}")
            else:
                teammates_info.append(f"Engineer: {teammate.name}")

        observation_prompts = [
            f"As Engineer {self.name}, analyzing team dynamics:",
            f"Current task outline: {task_outline}",
            f"Team members: {', '.join(teammates_info)}",
            f"\nUsing {self.tom_k}-Level Theory of Mind, analyze:",
        ]

        if self.tom_k == 0:
            observation_prompts.extend([
                "\nToM_Level0:",
                "- Understand task requirements",
                "- Plan implementation approach",
                "- Identify dependencies"
            ])
        else:
            observation_prompts.extend([
                f"\nFor ToM levels 0 to {self.tom_k}:",
                "- Analyze teammate capabilities and likely approaches",
                "- Identify potential collaboration points",
                "- Predict project manager's expectations",
                "- Plan implementation considering team dynamics"
            ])

        if self.team.collaboration_context:
            observation_prompts.extend([
                "\nTeam Context:",
                f"Current implementations: {json.dumps(self.team.collaboration_context.teammate_implementations)}",
                f"Dependencies: {self.team.collaboration_context.dependencies}",
                f"Interfaces: {json.dumps(self.team.collaboration_context.interfaces)}"
            ])

        observation_prompts.extend([
            "\nProvide analysis in JSON format:",
            "```json",
            self._generate_engineer_tom_template(teammates),
            "```"
        ])

        return "\n".join(observation_prompts)

    async def analyze_task(self, task_outline: str) -> Optional[Dict[str, Any]]:
        """Analyze task using Theory of Mind"""
        if not task_outline:
            self.logger.warning(f"No task outline provided for {self.name}")
            return None

        try:
            belief_model_prompt = self.generate_belief_model(task_outline)
            tom_output = await self.aask(belief_model_prompt, "json_object")
            tom_output = extract(tom_output, template=self._generate_engineer_tom_template(self.get_teammates()))
            self.belief_models = tom_output[f"ToM_level{self.tom_k}"]["belief"]
            self.action = tom_output[f"ToM_level{self.tom_k}"]["action"]
            
            self.logger.info({
                "engineer": self.name,
                "belief_models": self.belief_models,
                "action_plan": self.action,
                "explanation": tom_output[f"ToM_level{self.tom_k}"]["explanation"]
            })
            
            return self.belief_models
        except Exception as e:
            self.logger.error(f"Task analysis failed for {self.name}: {str(e)}")
            return None
        
    def _generate_engineer_tom_template(self, teammates: List[Agent]) -> str:
        """Generate ToM response template for engineer"""
        template = {}
        teammate_names = [t.name for t in teammates]

        for level in range(self.tom_k + 1):
            if level == 0:
                template[f"ToM_level{level}"] = {
                    "belief": {
                        "task_understanding": "{{Your understanding of the task requirements, 10 words}}",
                        "implementation_approach": "{{Your planned implementation approach, 10 words}}",
                        "dependencies": ["{{List of identified dependencies, 10 words}}"]
                    },
                    "explanation": "{{Explain your initial analysis and planning, 10 words}}",
                    "action": "{{Your implementation steps, 10 words}}"
                }
            else:
                template[f"ToM_level{level}"] = {
                    "belief": {
                        teammate: {
                            "expected_approach": "{{What you think they will do, 10 words}}",
                            "potential_challenges": "{{Challenges they might face, 10 words}}",
                            "collaboration_points": "{{How you can collaborate, 10 words}}"
                        }
                        for teammate in teammate_names
                    },
                    "explanation": "{{Explain your reasoning about teammates' actions and collaboration opportunities}}",
                    "action": "{{Your implementation steps, 10 words}}"
                }

        return json.dumps(template, indent=2)

    def generate_implementation_prompt(self, task_outline: str) -> str:
        """Generate implementation prompt with team context"""
        prompts = [f"Implementation Task: {task_outline}"]
        
        if self.tom_k > 0:
            prompts.extend([
                f"\nBelief Model Context: {json.dumps(self.belief_models)}",
                f"Planned Approach: {self.action}"
            ])
            
        if self.team.collaboration_context:
            prompts.extend([
                "\nTeam Collaboration Context:",
                "Required Interfaces:",
                json.dumps(self.team.collaboration_context.interfaces, indent=2),
                "\nDependencies:",
                json.dumps(self.team.collaboration_context.dependencies, indent=2)
            ])
            
        prompts.extend([
            "\nImplementation Requirements:",
            "1. Ensure interface compatibility with teammates",
            "2. Handle dependencies appropriately",
            "3. Include necessary error handling",
            "4. Add interface documentation",
            "\nProvide implementation in JSON format:",
            "```json",
            generate_code_response_template(),
            "```"
        ])
        
        return "\n".join(prompts)

    async def analyze_teammate_implementations(self, implementations: Dict[str, str]) -> None:
        """Analyze teammate implementations and update collaboration context"""
        try:
            analysis_prompt = [
                f"As Engineer {self.name}, analyze teammate implementations:",
                "Consider:",
                "- Interface compatibility",
                "- Dependency relationships",
                "- Integration points",
                "- Potential challenges",
            ]
            
            analysis_prompt.append(
                "##Your teammates' implementations:\n{json.dumps(implementations)}"
            )            
            analysis_prompt.append(
                "##Your belief model:\n{json.dumps(self.belief_models)}"
            )
            
            analysis_prompt.append(
                "##Provide analysis in JSON format: " + generate_score_template(list(implementations.keys()))
            )
            
            analysis = await self.aask(
                "\n".join(analysis_prompt),
                response_format="json_object"
            )
            
            parsed_analysis = extract(analysis, template=generate_score_template(list(implementations.keys())))

            
            dependencies = []
            interfaces = {}
            for impl in implementations.values():
                dependencies.extend(impl["dependencies"])
                interfaces.update(impl["interfaces"])
                
            self.collaboration_context = CollaborationContext(
                teammate_implementations=implementations,
                teammate_beliefs=self.belief_models,
                dependencies=dependencies,
                interfaces=interfaces
            )
            
            return parsed_analysis

        except Exception as e:
            self.logger.error(f"Failed to analyze teammate implementations: {str(e)}")
            raise CollaborationError(f"Implementation analysis failed: {str(e)}")
    @retry(
        stop=stop_after_attempt(3),
        wait=wait_fixed(2),
        retry=retry_if_exception_type((TaskImplementationError, CollaborationError))
    )
    async def run(self, task_outline_dict: Dict[str, str], pm_name: str) -> str:
        """Execute engineer's implementation cycle with team awareness"""
        task_outline = task_outline_dict.get(self.name, "")
        if not task_outline:
            self.logger.warning(f"No task outline found for {self.name}")
            return ""

        try:            
            # Generate and execute implementation
            implementation_prompt = self.generate_implementation_prompt(task_outline)
            output = await self.aask(
                implementation_prompt,
                response_format="json_object",
                for_who=pm_name
            )
            
            # Extract and validate implementation code
            impl = extract(output, template=generate_code_response_template())
        
            # Update codebase
            self.codebase = impl["code"]
            
            self.update_default_prompts([
                {
                    "role": "system",
                    "content": f"Your previous codebase: \n```\n{self.codebase}\n```"
                }
            ])
            
            return impl
            
        except Exception as e:
            self.logger.error(f"Implementation failed: {str(e)}")
            raise TaskImplementationError(f"Failed to implement task: {str(e)}")


