import asyncio
import logging
import json, os
from pathlib import Path
from typing import List, Dict
from agent.memory import Message
from agent.software import Engineer, ProjectManager
from concurrent.futures import ThreadPoolExecutor, as_completed
from utils.save_files import save_outline, save_implementation
from config.const import CONFIGS
# from agent.team import Team
from agent.belief_model import BeliefModel as Team

class Programming:
    def __init__(self, num_rounds: int, task: str, project_name: str, n_engineers: int, n_project_managers: int):
        self.num_rounds = num_rounds
        self.task = task
        self.belief_scores: Dict[str, Dict] = {}
        self.model_name = CONFIGS["llm"]["model"]
        self.project_name = self._generate_project_name(project_name)
        
        self.engineers = self._hire_agents("engineer", n_engineers)
        self.project_managers = self._hire_agents("project_manager", n_project_managers)
        
        self._prepare_folder()
        self.teams = [self._setup_team(i) for i in range(len(self.project_managers))]

    def _generate_project_name(self, project_name: str) -> str:
        cur_dir = os.path.dirname(os.path.abspath(__file__))
        task_name = CONFIGS.get("task", "")
        return f"{cur_dir}/tom_{task_name}_{self.model_name}/{project_name}"

    def _hire_agents(self, role: str, n: int) -> List[Engineer] | List[ProjectManager]:
        if role == "project_manager":
            return [ProjectManager(id=i, tom_k=i, task=self.task) for i in range(n)]
        elif role == "engineer":
            return [Engineer(id=i, task=self.task) for i in range(n)]
        raise ValueError("Invalid role")

    def _prepare_folder(self):
        logging.info(f"Preparing folder for project: {self.project_name}")
        Path(self.project_name).mkdir(parents=True, exist_ok=True)
        
        self.project_manager_dirs = [
            Path(f"{self.project_name}/ProjectManager{pm.id}_ToM{pm.tom_k}")
            for pm in self.project_managers
        ]
        for dir_path in self.project_manager_dirs:
            dir_path.mkdir(parents=True, exist_ok=True)

    def _setup_team(self, team_id: int) -> Team:
        team = Team(f"Team{team_id}")
        team.add_agent(self.project_managers[team_id])
        for engineer in self.engineers:
            team.add_agent(engineer)
        team.initialize_scores()
        return team

    async def run_programming_process(self):
        logging.info(f"Running programming process for {self.num_rounds} rounds.")

        for round in range(self.num_rounds):
            logging.info(f"Round {round + 1}:")
            self.belief_scores[f"Round{round + 1}"] = {}

            self.outlines = await self._get_task_outlines(round)
            implementations = await self._get_implementations(round)
            await self._check_belief_alignment(round, implementations)
            self._update_teams()

        self._save_belief_scores()

    async def _get_task_outlines(self, round: int) -> List[Dict]:
        with ThreadPoolExecutor() as executor:
            futures = [
                executor.submit(
                    lambda pm=pm: asyncio.run(
                        save_outline(pm, round, self.project_manager_dirs[pm.id])
                    )
                ) for pm in self.project_managers
            ]
            results = [future.result() for future in as_completed(futures)]

        outlines = [None] * len(self.project_managers)
        for outline, pm_id in results:
            outlines[pm_id] = outline
            self._add_memory_to_project_manager(pm_id, round, outline)

        return outlines

    async def _get_implementations(self, round: int) -> Dict[str, str]:
        implementations = {}
        with ThreadPoolExecutor() as executor:
            futures = [
                executor.submit(
                    lambda engineer=engineer, outline=outline, proj_id=proj_id: asyncio.run(
                        save_implementation(
                            engineer, outline, round, proj_id, self.project_manager_dirs
                        )
                    )
                ) for engineer in self.engineers
                for proj_id, outline in enumerate(self.outlines)
            ]

            for future in as_completed(futures):
                result = future.result()
                if result:
                    engineer, implementation, proj_id = result
                    implementations[engineer.name] = implementation
                    self._add_memory_to_agents(engineer, proj_id, round, implementation)

        return implementations

    async def _check_belief_alignment(self, round: int, implementations: Dict[str, str]):
        for id, project_manager in enumerate(self.project_managers):
            scores = await project_manager.check_belief_alignment(implementations)
            self.belief_scores[f"Round{round + 1}"][f"{project_manager.name}"] = {
                "scores": scores,
                "tom_k": project_manager.tom_k,
            }
            for teammate_name, score_info in scores.items():
                self.teams[id].update_score(project_manager.name, teammate_name, float(score_info["score"]))

    def _update_teams(self):
        for id, team in enumerate(self.teams):
            stable_coalition = team.find_stable_coalitions(1, 3)
            if not team.rematching_required:
                team.agents = stable_coalition
                team.initialize_scores()
            else:
                self.teams[id] = self._setup_team(id)

    def _save_belief_scores(self):
        belief_scores_path = Path(self.project_name) / "belief_scores.json"
        try:
            with belief_scores_path.open('w') as f:
                json.dump(self.belief_scores, f, indent=4)
            logging.info(f"Belief scores saved to {belief_scores_path}")
        except Exception as e:
            logging.error(f"Failed to save belief scores: {str(e)}")
            raise ValueError(f"Failed to save belief scores: {str(e)}")

    def _add_memory_to_project_manager(self, pm_id: int, round: int, outline: Dict):
        self.project_managers[pm_id].memory.add_memory([
            Message(
                content=f"(Round {round})" + json.dumps(outline, indent=4),
                sender=f"{self.project_managers[pm_id].name}",
            )
        ])

    def _add_memory_to_agents(self, engineer: Engineer, proj_id: int, round: int, implementation: str):
        engineer.memory.add_memory([
            Message(
                content=f"(Round {round})" + json.dumps(self.outlines[proj_id], indent=4),
                sender=f"ProjectManager{proj_id}",
            ),
            Message(
                content=f"(Round {round})" + implementation,
                sender=f"{engineer.name}",
                receiver={f"ProjectManager{proj_id}"},
            )
        ])
        self.project_managers[proj_id].memory.add_memory([
            Message(
                content=f"(Round {round})" + implementation,
                sender=f"{engineer.name}",
            )
        ])
        
        
        

async def run_program(idea, project_name, num_rounds, n_engineers, n_project_managers):
    programming_ = Programming(
        num_rounds=num_rounds,
        task=idea,
        project_name=project_name,
        n_engineers=n_engineers,
        n_project_managers=n_project_managers,
    )
    await programming_.run_programming_process()

