import json, re
import openai
from config.const import CONFIGS as configs
from metagpt.utils.custom_decoder import CustomDecoder
from tenacity import retry, stop_after_attempt, wait_fixed


def custom_json_parser(input_string):
    def parse_value(value):
        # Try to parse as JSON first
        try:
            return json.loads(value)
        except json.JSONDecodeError:
            pass
        
        # If it's not valid JSON, return it as a string
        return value.strip()

    # Remove any leading/trailing whitespace
    input_string = input_string.strip()

    # Check if the input is already valid JSON
    try:
        return json.loads(input_string)
    except json.JSONDecodeError:
        pass

    # If not, try to parse it manually
    result = {}
    
    # Find all key-value pairs
    pairs = re.findall(r'"?(\w+)"?\s*:\s*(.+?)(?=,\s*"?\w+"?\s*:|$)', input_string, re.DOTALL)
    
    for key, value in pairs:
        # Remove quotes from the key if present
        key = key.strip('"')
        
        # Parse the value
        parsed_value = parse_value(value)
        
        result[key] = parsed_value

    return result


@retry(stop=stop_after_attempt(3), wait=wait_fixed(2))
def extract(json_str, template="", using_gpt_formatter=False) -> dict:
    try:
        json_str = json.loads(json_str)
        return json_str
    except json.JSONDecodeError:
       
    
        if using_gpt_formatter: 
            # print(f"Error: {json_str}")
            formatter = GPTformatter()
            return formatter.format_json(json_str, template)
        results = custom_json_parser(json_str)
        return results

class GPTformatter:
    def __init__(self):
        if configs["llm"]["model"].startswith("gpt"):
            self.model = configs["llm"]["model"]
            self.client = openai.OpenAI(
                api_key=configs["llm"]["api_key"], base_url=configs["llm"]["base_url"]
            )
        elif "gpt_model" in configs:
            self.model = configs["gpt_model"]["model"]
            self.client = openai.OpenAI(
                api_key=configs["gpt_model"]["api_key"],
                base_url=configs["gpt_model"]["base_url"],
            )
        else:
            raise ValueError(
                "Invalid model for GPTformater, please check the config file with `gpt_model`"
            )

    def format_json(self, json_str, template="") -> dict:
        response = self.client.chat.completions.create(
            messages=[
                {
                    "role": "system",
                    "content": f"You are the JSON formater. Please format the JSON object as {template}",
                },
                {"role": "user", "content": json_str},
            ],
            response_format={"type": "json_object"},
            model=self.model,
        )
        try:
            response = response.choices[0].message.content
            response= json.loads(response)
        except json.JSONDecodeError:
            raise ValueError(f"Failed to format JSON object: {json_str}")
       
        return response



