import logging
from typing import List
from config.const import CONFIGS as configs
import openai, groq
import google.generativeai as genai
from tenacity import retry, stop_after_attempt, wait_fixed
from utils import Logger


# require openai version 1.6.0
version = openai.__version__
if version != "1.6.0":
    raise Exception("Please install openai version 1.6.0")


class LLM:
    def __init__(
        self,
        name="",
        default_prompts: list = [{"role": "system", "content": ""}],
        client=None,
    ):
        from .memory import ChatHistoryMemory

        self.client = client
        self.configs = configs["llm"]

        self.default_prompts = default_prompts
        self.name = name
        self.use_memory = self.configs.get("use_memory", False)
        self.memory: ChatHistoryMemory = ChatHistoryMemory(my_name=name)

        self.logger = Logger(name)

    def __str__(self) -> str:
        return self.name
    def update_default_prompts(self, new_prompts):
        self.default_prompts = new_prompts

    def init_llm(self):

        if self.configs["api_type"] == "openai":
            self.client = openai.OpenAI(
                api_key=self.configs["api_key"], base_url=self.configs["base_url"]
            )
        elif self.configs["api_type"] == "gemini":
            genai.configure(api_key=self.configs["api_key"])
            model = genai.GenerativeModel(model_name=self.configs["model"])
            self.client = model.start_chat(history=[])

        elif self.configs["api_type"] == "groq":
            self.client = groq.Groq(api_key=self.configs["api_key"])
        
        else:
            raise ValueError(f"Invalid API type: {self.configs['api_type']}")

    def init_allm(self):

        if self.configs["api_type"] == "openai":
            self.client = openai.AsyncOpenAI(
                api_key=self.configs["api_key"], base_url=self.configs["base_url"]
            )
        elif self.configs["api_type"] == "gemini":
            genai.configure(api_key=self.configs["api_key"])

            model = genai.GenerativeModel(model_name=self.configs["model"])
            self.client = model.start_chat(history=[])

        elif self.configs["api_type"] == "groq":
            self.client = groq.AsyncGroq(api_key=self.configs["api_key"])

    def handle_history(self, history: List[str]):  # todo
        return history

    @retry(stop=stop_after_attempt(3), wait=wait_fixed(10))
    def ask(
        self,
        prompt,
        response_format: str = None,
        use_memory: bool = None,
        for_who="all",
    ):
        if self.client is None:
            self.init_llm()

        use_memory = use_memory if use_memory is not None else self.use_memory
        if use_memory:
            history = self.memory.get_memory(for_who=for_who)
        else:
            history = None

        if self.configs["api_type"] == "gemini":
            history = self.memory.to_string() if history is not None else ""
            response = self.client.send_message(
                f"{self.default_prompts} \n {history} \n\n {prompt}"
            )
            return response.text
        elif self.configs["api_type"] == "openai" or self.configs["api_type"] == "groq":
            message = [{"role": "user", "content": prompt}]
            message.extend(self.default_prompts)

            if history is not None:
                message.extend(history)

            if response_format is None:
                response = self.client.chat.completions.create(
                    messages=message, model=self.configs["model"]
                )
            else:
                response = self.client.chat.completions.create(
                    messages=message,
                    response_format={"type": response_format},
                    model=self.configs["model"],
                )
            return response.choices[0].message.content

    @retry(
        stop=stop_after_attempt(3),
        wait=wait_fixed(10),
        retry_error_callback=lambda x: logging.error(x),
    )
    async def aask(
        self,
        prompt,
        response_format: str = None,
        use_memory: bool = None,
        for_who="all",
    ):
        if self.client is None:
            self.init_allm()
        use_memory = use_memory if use_memory is not None else self.use_memory
        if use_memory:
            history = await self.memory.get_memory(for_who=for_who)
        else:
            history = None

        if self.configs["api_type"] == "gemini":
            history = self.memory.to_string() if history is not None else ""
            response = await self.client.send_message_async(
                f"{self.default_prompts} \n {history} \n\n {prompt}"
            )
            return response.text
        elif self.configs["api_type"] == "openai" or self.configs["api_type"] == "groq":
            message = [{"role": "user", "content": prompt}]
            message.extend(self.default_prompts)

            if history is not None:
                message.extend(history)

            if response_format is None:
                response = await self.client.chat.completions.create(
                    messages=message, model=self.configs["model"]
                )
            else:
                response = await self.client.chat.completions.create(
                    messages=message,
                    response_format={"type": response_format},
                    model=self.configs["model"],
                    timeout=30,
                )

            return response.choices[0].message.content

    def send_token_limit(self) -> int:
        send_token_limit_dict = {
            "gpt-3.5-turbo": 4096,
            "gpt-35-turbo": 4096,
            "gpt-3.5-turbo-16k": 16384,
            "gpt-3.5-turbo-0613": 16384,
            "gpt-3.5-turbo-1106": 16384,
            "gpt-3.5-turbo-0125": 16384,
            "gpt-4": 8192,
            "gpt-4-32k": 32768,
            "gpt-4-0613": 32768,
            "gpt-4-1106-preview": 131072,
            "gpt-4-0125-preview": 131072,
            "llama-2-7b-chat-hf": 4096,
        }
        model = self.configs["model"]
        # Default to 4096 tokens if model is not in the dictionary
        return send_token_limit_dict[model] if model in send_token_limit_dict else 4096




# async def main():

#     # Test the LLM class
#     DEFAULT_CLIENT.init_allm()
#     json_template = generate_code_response_template()
#     message = [
#         {
#             "role": "user",
#             "content": f"response in following json format: {json_template}",
#         },
#         {
#             "role": "system",
#             "content": "You are the Software Engineer. You will implement the task based on the provided outline. Implement language: Python",
#         },
#         {
#             "role": "assistant",
#             "content": "[ProjectManager0]: (Round 0)The Engineer should write a Python function called ...ccurrences and return the modified string.'",
#         },
#         {
#             "role": "assistant",
#             "content": "[you]: (Round 0)```python\n# Write a python function to remove first and last occurren...index] + s[last_index+1:]\n return s\n```",
#         },
#         {
#             "role": "assistant",
#             "content": "[ProjectManager0]: (Round 1)The Engineer should write a Python function called game2048.'",
#         },
#     ]

#     response = await DEFAULT_CLIENT.client.chat.completions.create(
#         messages=message,
#         response_format={"type": "json_object"},
#         model=DEFAULT_CLIENT.configs["model"],
#     )
#     print("Response:", response.choices[0].message.content)


# if __name__ == "__main__":
#     import asyncio

#     asyncio.run(main())
