import logging
import sys
from datetime import datetime
import os
from typing import Optional
from pathlib import Path

class CustomFormatter(logging.Formatter):
    """Custom formatter with colors for console output"""
    
    COLORS = {
        'DEBUG': '\033[0;36m',    # Cyan
        'INFO': '\033[0;32m',     # Green
        'WARNING': '\033[0;33m',  # Yellow
        'ERROR': '\033[0;31m',    # Red
        'CRITICAL': '\033[0;35m', # Purple
        'RESET': '\033[0m'        # Reset
    }

    def format(self, record):
        # Save original format
        format_orig = self._style._fmt

        # Add colors for console output
        if hasattr(sys.stdout, 'isatty') and sys.stdout.isatty():
            self._style._fmt = (
                f"{self.COLORS.get(record.levelname, self.COLORS['RESET'])}"
                f"{format_orig}"
                f"{self.COLORS['RESET']}"
            )

        # Call the original format method
        result = super().format(record)

        # Restore the original format
        self._style._fmt = format_orig
        return result


def setup_logger(
    name: str,
    log_dir: str = "logs",
    log_level: int = logging.DEBUG,
    module_path: Optional[str] = None
) -> logging.Logger:
    """
    Set up a logger with both file and console handlers
    
    Args:
        name: Logger name
        log_dir: Directory for log files
        log_level: Logging level
        module_path: Optional module path for more detailed logging
    """
    # Create logs directory if it doesn't exist
    log_dir = Path(log_dir)
    log_dir.mkdir(parents=True, exist_ok=True)

    # Create or get logger
    logger = logging.getLogger(name)
    logger.setLevel(log_level)
    
    # Remove existing handlers
    logger.handlers = []

    # Create formatters
    console_format = (
        '%(asctime)s [%(levelname)s] '
        '%(name)s'
        f'{" - " + module_path if module_path else ""}'
        ' - %(message)s (%(filename)s:%(lineno)d)'
    )
    
    file_format = (
        '%(asctime)s [%(levelname)s] '
        '%(name)s - %(module)s:%(lineno)d - '
        '%(funcName)s - %(message)s'
    )

    # Console handler with colors
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.INFO)
    console_handler.setFormatter(CustomFormatter(console_format))
    logger.addHandler(console_handler)

    # File handler
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    file_handler = logging.FileHandler(
        log_dir / f"{name.lower()}_{timestamp}.log",
        encoding='utf-8'
    )
    file_handler.setLevel(logging.DEBUG)
    file_handler.setFormatter(logging.Formatter(file_format))
    logger.addHandler(file_handler)

    return logger 