import os
import argparse

from pathlib import Path
from utils.logging import get_logger
from utils.io import (
    find_all_directory_children_w_filter,
    write_list_to_txtfile,
    get_full_config,
)

logger = get_logger()


def get_parser():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "-c",
        "--config_path",
        help="path to parent directory that stores analysis configs",
        type=str,
        default=None,
    )
    parser.add_argument("-n", "--n_seeds", help="n random seeds", type=int, default=10)
    parser.add_argument("-s", "--save_name", help="save name", type=str, default=None)
    parser.add_argument(
        "-r",
        "--retrain",
        help="retrain and overwrite previous results. otherwise will exclude commands that have already been run before",
        type=bool,
        default=False,
    )
    return parser


def run(args):
    config_base_path = args.config_path
    n_seeds = args.n_seeds

    # n_seeds = 15
    # config_base_path = "configs/analysis/pulse-base.yaml"

    # config_path = "configs/analysis"
    config_paths = find_all_directory_children_w_filter(config_base_path, ".yaml")

    analysis_data_dir = "data/analysis"
    data_paths = find_all_directory_children_w_filter(analysis_data_dir, "noise")

    commands = []
    for c_path in config_paths:
        logger.info(f"building commands for: {c_path}")
        config = get_full_config(c_path)

        for path in data_paths:
            for seed in range(n_seeds):
                subdir_name = config["model_type"]
                if config["model_name"] is not None:
                    subdir_name = config["model_name"]

                save_dir = (
                    Path("experiments/analysis/")
                    / path.split("/")[2]
                    / f"{Path(path).stem}"
                    / subdir_name
                    / f"seed-{seed}"
                )
                cmd = f"python scripts/run_pretrain.py -c {c_path} -p {path} -s {seed} -sd {save_dir}"
                if (
                    os.path.exists(save_dir / "scores.pkl") and not args.retrain
                ):  # if it exists, and not retrain, skip
                    logger.info(f"skipping {save_dir}. already exists.")
                    continue

                commands.append(cmd)

    save_name = (
        args.save_name if args.save_name else "data/process/analysis/submit-commands.sh"
    )
    logger.info(f"Writing Text file to: {save_name}")

    write_list_to_txtfile(commands, save_name)


if __name__ == "__main__":

    parser = get_parser()
    args = parser.parse_args()

    run(args)
