import argparse
import torch
from datasets import load_dataset
from transformers import AutoTokenizer
from transformers.trainer import get_scheduler
import torch.nn as nn
import torch.optim as optim
from openrlhf.utils import blending_datasets, MockStrategy
from openrlhf.datasets import ProbDataset, RewardDataset
from openrlhf.trainer import DPOPTrainer, DPOPMTrainer, DPOTrainer
from torch.utils.data import DataLoader
import pandas as pd
from openrlhf.models import Actor
# Load your dataset



device="cuda"

parser = argparse.ArgumentParser()
parser.add_argument("--pretrain", type=str, default=None)
parser.add_argument("--max_epochs", type=int, default=5)
parser.add_argument("--l2", type=float, default=0.0, help="weight decay loss")
parser.add_argument("--beta", type=float, default=0.1)
parser.add_argument("--ipo", action="store_true", default=False)  # IPO https://arxiv.org/pdf/2310.12036v2.pdf
parser.add_argument("--label_smoothing", type=float, default=0.0)  # cDPO https://arxiv.org/pdf/2305.18290.pdf
parser.add_argument("--aux_loss_coef", type=float, default=0, help="MoE balancing loss")
parser.add_argument(
    "--nll_loss_coef", type=float, default=0, help="Regularization with NLL loss, see LLama 3.1 tech report."
)
parser.add_argument("--adam_betas", type=float, nargs=2, default=(0.9, 0.95), help="Betas for Adam optimizer")
parser.add_argument("--packing_samples", action="store_true", default=False)
parser.add_argument("--use_wandb", type=str, default=None)
parser.add_argument("--use_tensorboard", type=str, default=None, help="TensorBoard logging path")
parser.add_argument("--save_path", type=str, default="./checkpoint")
parser.add_argument("--result_path", type=str, default="./results")
parser.add_argument("--save_steps", type=int, default=-1)
parser.add_argument("--logging_steps", type=int, default=1)
parser.add_argument("--eval_steps", type=int, default=-1)
parser.add_argument("--ckpt_path", type=str, default="./ckpt/checkpoints_dpo")
parser.add_argument("--max_ckpt_num", type=int, default=3)
parser.add_argument("--max_ckpt_mem", type=int, default=1e8)
parser.add_argument("--seed", type=int, default=42)
# DeepSpeed
# parser.add_argument("--micro_train_batch_size", type=int, default=8, help="batch size per GPU")
parser.add_argument("--train_batch_size", type=int, default=4, help="Global training batch size")
parser.add_argument("--dataset", type=str, default="JSON_Preference")
# parser.add_argument("--dataset", type=str, default="OpenRLHF/preference_dataset_mixture2_and_safe_pku")
parser.add_argument("--dataset_probs", type=str, default="1.0", help="sampling probs for datasets")
parser.add_argument("--train_split", type=str, default="train", help="train split of the HF dataset")
parser.add_argument("--eval_split", type=str, default="test", help="test split of the dataset")

parser.add_argument("--prompt_key", type=str, default=None)
parser.add_argument("--chosen_key", type=str, default="JSON Preference 1")
parser.add_argument("--rejected_key", type=str, default="JSON Preference 2")
parser.add_argument("--input_template", type=str, default=None)
parser.add_argument(
    "--apply_chat_template", action="store_true", default=True, help="Use HF tokenizer chat template"
)
parser.add_argument("--max_samples", type=int, default=1e8, help="Max number of samples")
parser.add_argument("--max_len", type=int, default=512)
parser.add_argument("--load_in_4bit", action="store_true", default=False)
parser.add_argument("--lora_rank", type=int, default=0)
parser.add_argument("--lora_alpha", type=int, default=16)
parser.add_argument("--target_modules", type=str, nargs="*", default="all-linear")
parser.add_argument("--lora_dropout", type=float, default=0)
parser.add_argument("--bf16", action="store_true", default=False, help="Enable bfloat16")
parser.add_argument("--flash_attn", action="store_true", default=False, help="Enable FlashAttention2")
    
args = parser.parse_args()

model_name = args.pretrain
# Load tokenizer for LLaMA
tokenizer = AutoTokenizer.from_pretrained(args.pretrain, use_fast=True, trust_remote_code=True)
# Use eos_token as pad_token
tokenizer.pad_token = tokenizer.eos_token
tokenizer.bos_token = tokenizer.eos_token if tokenizer.bos_token is None else tokenizer.bos_token
tokenizer.padding_side = "right"
template = """{% if not add_generation_prompt is defined %}{% set add_generation_prompt = false %}{% endif %}{% set loop_messages = messages %}{% for message in loop_messages %}{% set content = '<|start_header_id|>' + message['role'] + '<|end_header_id|>\n\n'+ message['content'] | trim + '<|eot_id|>' %}{% if loop.index0 == 0 %}{% set content = bos_token + content %}{% endif %}{{ content }}{% endfor %}{% if add_generation_prompt %}{{ '<|start_header_id|>assistant<|end_header_id|>\n\n' }}{% endif %}"""
tokenizer.chat_template = template

strategy = MockStrategy(args)

train_data, eval_data = blending_datasets(
        args.dataset,
        args.dataset_probs,
        strategy,
        args.seed,
        max_count=args.max_samples,
        stopping_strategy="all_exhausted",
        train_split=args.train_split,
        eval_split=args.eval_split,
    )
train_data = train_data.select(range(min(args.max_samples, len(train_data))))
eval_data = eval_data.select(range(min(args.max_samples, len(eval_data))))
train_dataset = RewardDataset(
    train_data,
    tokenizer,
    args.max_len,
    strategy,
    input_template=args.input_template,
    is_dpo=True,
)
eval_dataset = RewardDataset(
    eval_data,
    tokenizer,
    args.max_len,
    strategy,
    input_template=args.input_template,
    is_dpo=True,
)
    



# Load the LLaMA model
actor_model = Actor(
    model_name,  # Replace with the actual LLaMA model path
    load_in_4bit=args.load_in_4bit,
    bf16=args.bf16,
    use_flash_attention_2=args.flash_attn,
    lora_rank=args.lora_rank,
    lora_alpha=args.lora_alpha,
    lora_dropout=args.lora_dropout,
    target_modules=args.target_modules,                    # Use LoRA fine-tuning for efficiency
).to(device)




ref_model = Actor(
    model_name,  # Replace with the actual LLaMA model path
    use_flash_attention_2=args.flash_attn,
    bf16=args.bf16,
    load_in_4bit=args.load_in_4bit,                    # Use LoRA fine-tuning for efficiency
).to(device)

# Create DataLoaders for train and evaluation datasets
train_dataloader = DataLoader(train_dataset, batch_size=args.train_batch_size, shuffle=True, collate_fn=train_dataset.collate_fn)
eval_dataloader = DataLoader(eval_dataset, batch_size=args.train_batch_size, collate_fn=eval_dataset.collate_fn)

# Define the optimizer
optimizer = torch.optim.AdamW(actor_model.parameters(), lr=1e-4)

scheduler = get_scheduler(
        "cosine_with_min_lr",
        optimizer,
        num_warmup_steps=100,
        num_training_steps=10000,
        scheduler_specific_kwargs={"min_lr": 1e-5},
    )




# Configure the trainer
trainer = DPOTrainer(
    model=actor_model,
    ref_model=ref_model,
    strategy=strategy,
    scheduler=scheduler,
    tokenizer=tokenizer,
    train_dataloader=train_dataloader,
    eval_dataloader=eval_dataloader,
    optim=optimizer,
    max_norm=1.0,         # Gradient clipping
    beta=0.1,             # Hyperparameter for DPO loss
    max_epochs=3,         # Number of epochs to train
)
consumed_samples = 0
num_update_steps_per_epoch = len(train_dataset) // args.train_batch_size
trainer.fit(args, consumed_samples, num_update_steps_per_epoch)

strategy.save_model(actor_model, output_dir=args.save_path)
trainer.evaluate(eval_dataloader)
# data = {'Probability': probs, 'Prediction': probs_}
# df = pd.DataFrame(data)

# # 保存为 CSV 文件
# df.to_csv(args.result_path, index=False)

