
import json
import sys
sys.path.append('..')
from modeling import load_gptj, GPTJWrapper, load_gpt2xl, load_gpt2, GPT2Wrapper, load_bloom, BloomWrapper
from bigbench_tasks import load_bigbench_task, multiple_choice_query, PromptBuilder, DefaultOpenPrompter
from rich.progress import track
from console import console, timer
import numpy as np
from utils import get_probs_and_mrrs, from_layer_logits_to_prob_distros #model, logits, answer ; logits
import random
import gc
import torch


random.seed(42)

class ColoredObjectsPrompter(DefaultOpenPrompter):
    def __init__(self, dataset):
        super(ColoredObjectsPrompter, self).__init__(dataset)

    def build_open_prompt(self, index, include_answer=False):
        """
        nshots: int number of shots. 0 means just return dataset[index] prompt. 1 means 1 example and 1 test prompt
        dataset a bigbench dataset that can be indexed by index. dataset[index]
        index: the index of the test example. This method will use the [index-nshots, index) datapoints as the nshot examples, wrapping if necessary
        returns: a string prompt, the answer as a string
        """
        datapoint = self.dataset[index]
        inputs = datapoint['inputs']#, datapoint['multiple_choice_targets']
        gt = datapoint['targets'][0].title() #captialize the first letter!
        #gt_idx = targets.index(gt)
        prompt = inputs
        if include_answer:
            prompt+=' '+gt
        return prompt, ' '+gt

    def get_mc_targets(self, idx):
        return [' '+t.title() for t in self.dataset[idx]['multiple_choice_targets']]

def generate_ans(model, prompter, idx):
    #making the prompt
    prompt, gt = prompter.nshot_open_prompt(nshots, idx)
    targets = prompter.get_mc_targets(idx)
    gt_idx = targets.index(gt)

    #running prompt thru model
    prompt_ids = model.tokenize(prompt)
    logits = model.get_layers(prompt_ids)

    probs_results = {}
    rrs_results   = {}

    for i in range(len(targets)):
        tgt = targets[i]
        probs, rrs = get_probs_and_mrrs(model, logits, tgt)
        probs_results[i] = probs.tolist()
        rrs_results[i] = rrs.tolist()

    top10_per_layer = model.topk_per_layer(logits, 10)
    prompt_results = {'inputs':prompt, 'targets':targets, 'answer':gt, 'answer_idx':gt_idx, 'probs':probs_results, 'rrs':rrs_results, 'top10_per_layer':top10_per_layer}
    #Lastly, turn the logits into a bunch of prob distributions over the whole vocab. This will be stored separately
    prob_distros = from_layer_logits_to_prob_distros(logits)
    return prompt_results, prob_distros

def get_open_generations(model, dataset):
    output = []
    prompter = ColoredObjectsPrompter(dataset)
    all_prob_distros = []
    with torch.no_grad():
        for i in track(range(len(prompter)), description='iterating...'):
            json_out, prob_distros = generate_ans(model, prompter, i)
            output.append(json_out)
            all_prob_distros.append(prob_distros)

    with open(f'{model_name}_{nshots}_open_cobjs_results.json', 'w') as fp:
        json.dump(output, fp, indent=4)

    all_prob_distros = np.stack(all_prob_distros)
    print(all_prob_distros.shape)
    np.save(f'{model_name}_{nshots}_open_cobjs_vocab_distros.npy', all_prob_distros)

if __name__ == "__main__":

    model_name = sys.argv[1]
    nshots = sys.argv[2]
    console.print(model_name, nshots, 'shot(s)')
    timer_task = timer.add_task("Loading model")
    with timer:
        if 'gpt2' in model_name:#model_name == 'gpt2-xl':
            model, tokenizer = load_gpt2(model_name)
            model = GPT2Wrapper(model, tokenizer)#GPTJWrapper(gptj, tokenizer)
        elif 'gptj' == model_name:
            model, tokenizer = load_gptj()
            model = GPTJWrapper(model, tokenizer)
        elif 'bloom' in model_name:
            model, tokenizer = load_bloom(model_name)
            model = BloomWrapper(model, tokenizer)
    timer.stop_task(timer_task)
    timer.update(timer_task, visible=False)
 

    dataset = load_bigbench_task('reasoning_about_colored_objects')['default']
    #get_mc_generations(model, dataset)
    if ',' in nshots:
        rnshots = [int(n) for n in nshots.split(',')]
        for n in rnshots:
            console.print("NSHOTS", n)
            nshots = n
            get_open_generations(model, dataset)
    else:
        nshots = int(nshots)
        get_open_generations(model, dataset)
