import os
import sys
import traceback

sys.path.append(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
sys.path.append("../..")
from environment.tools import *
from pandas import DataFrame
from typing import Any


class EnvOutput:
    class KeyError(Exception):
        pass

    def __init__(self, success: bool, data: Any):
        self._success = success
        self._data = data
        if isinstance(data, DataFrame):
            self._page_idx = 0
            self._page_total = len(data) // 10 + (1 if len(data) % 10 != 0 else 0)
            self._original_data = data
            self._data = data.head(10)

    def __getitem__(self, key):
        if key == "success":
            return self._success
        if key == "data":
            return self._data
        if key == "whole_data":
            return (
                self._original_data if isinstance(self._data, DataFrame) else self._data
            )
        if key == "str":
            return str(self)
        raise self.KeyError(
            f"Invalid key: {key},only support 'success', 'data', 'whole_data' and 'str'."
        )

    def to_dict(self):
        return {
            "success": self._success,
            "data": self._data,
            "whole_data": (
                self._original_data if isinstance(self._data, DataFrame) else self._data
            ),
            "str": str(self),
        }

    def __str__(self):
        if isinstance(self._data, DataFrame):
            if len(self._data) == 0:
                return "No data."
            header_str = self._data.columns.values
            res = str(header_str) + "\n"
            for i in range(10):
                if i >= len(self._data):
                    break
                res += str(self._data.iloc[i].values) + "\n"
            res += (
                "Page/Total: " + str(self._page_idx + 1) + "/" + str(self._page_total)
            )
            return res
        return str(self._data)

    def next_page(self):
        if not isinstance(self._data, DataFrame):
            return (
                "next_page() is not supported for this data type:"
                + str(type(self._data))
                + "\nonly DataFrame support next_page()."
                + "\nMake sure you are using the correct index. -1 is the lastest result."
            )
        if (self._page_idx - 1) * 10 >= len(self._original_data):
            self._data = "No more data."

        self._page_idx += 1
        self._data = self._original_data.iloc[
            self._page_idx * 10 : (self._page_idx + 1) * 10
        ]
        return self


class WorldEnv:
    """
    World Environment
    Provide APIs to access the virtual world.
    """

    def __init__(self, en_version=False):
        """
        Initialize the world environment.
        """

        self.support_cities = [
            "Shanghai",
            "Beijing",
            "Shenzhen",
            "Guangzhou",
            "Chongqing",
            "Suzhou",
            "Chengdu",
            "Hangzhou",
            "Wuhan",
            "Nanjing",
        ]
        self.attractions = Attractions()
        self.accommodations = Accommodations()
        self.restaurants = Restaurants()
        self.intercitytransport = IntercityTransport()
        self.transportation = Transportation()
        self.poi = Poi()

        self.results = []

    def __call__(self, cmd_str: str):
        """
        Call the API by command string in the format of python function call.
        """
        # init env to execute the command directly
        attractions_keys = self.attractions.keys
        attractions_types = self.attractions.get_type_list
        attractions_select = self.attractions.select
        attractions_id_is_open = self.attractions.id_is_open
        attractions_nearby = self.attractions.nearby

        accommodations_keys = self.accommodations.keys
        accommodations_select = self.accommodations.select
        accommodations_nearby = self.accommodations.nearby

        restaurants_select = self.restaurants.select
        restaurants_keys = self.restaurants.keys
        restaurants_nearby = self.restaurants.nearby
        restaurants_id_is_open = self.restaurants.id_is_open
        restaurants_cuisine = self.restaurants.get_cuisine_list
        restaurants_with_recommended_food = (
            self.restaurants.restaurants_with_recommended_food
        )

        goto = self.transportation.goto
        intercity_transport_select = self.intercitytransport.select
        poi_lat_lon_search = self.poi.search

        next_page = self.next_page
        Results = self.results

        try:
            res = eval(cmd_str)
            if not isinstance(res, EnvOutput):
                res = EnvOutput(True, res)
        except Exception as e:
            res = EnvOutput(False, "Invalid command.\n" + str(e))
            traceback.print_exc()
        self.results.append(res)
        return self.results[-1]

    def next_page(self):
        """
        Go to the next page.
        """
        if len(self.results) == 0:
            return "No results."
        return self.results[-1].next_page()

    def reset(self):
        """
        Reset the environment.
        """
        self.results = []