# Auction-Based Agent System for Overcooked Environment

This module implements a market-based approach where agents bid for tasks using a virtual currency system in the Overcooked environment.

## Overview

The auction-based agent system consists of the following components:

1. **AuctionAgent**: Extends the base Agent class with auction-specific functionality, including bidding on tasks, managing currency, and tracking task performance.

2. **AuctionSystem**: Manages the auction process, including task decomposition, bid collection, and task allocation.

3. **AgentManager**: Updated to support the auction-based agent system.

## How It Works

### Currency System

- Each agent is allocated an initial currency amount (default: 100 units).
- Agents earn currency for completing tasks efficiently and lose currency when they delay or fail a task.
- The currency is used to bid on tasks, with higher bids increasing the chance of winning the task.

### Task Decomposition

When a new dish order arrives, the AuctionSystem breaks it down into subtasks:

1. Getting base ingredients from storage
2. Putting ingredients at the appropriate locations
3. Activating tools to process ingredients
4. Getting processed items
5. Putting the final dish on the serving table

### Bidding Process

For each subtask:

1. Agents calculate a bid based on:
   - Their current location (proximity to the task)
   - Their current workload
   - Their success rate for similar tasks
   - Their available currency

2. The highest bidder wins the task and pays the bid amount from their currency.

### Task Execution

1. Agents receive their assigned tasks through the auction.
2. Each agent generates actions to complete their assigned tasks.
3. The environment executes the actions and provides feedback.

### Reward System

- Agents earn currency for completing tasks successfully.
- The reward amount depends on:
  - The task's difficulty
  - How quickly the task was completed
  - The remaining lifetime of the dish

- Agents lose currency for failing tasks or completing them late.

## Usage

To use the auction-based agent system:

```python
from my_agent.agent_manager import AgentManager
from my_agent.cost_success_matrices import create_cost_matrix_seeded, get_agent_success_percentages
from overcooked.game import World

# Define actions for cost and success rate matrices
actions = ["noop", "goto", "put", "get", "activate"]

# Create cost matrix
cost_matrix = create_cost_matrix_seeded(num_agents, actions, seed)

# Create success rate matrix
success_matrix = get_agent_success_percentages(num_agents, actions, cost_matrix)

# Create the environment
env = World(
    num_agents=num_agents,
    max_steps=max_steps,
    level=level,
    seed=seed,
    success_rates=success_matrix
)

# Create the agent manager with auction agent type
agent_manager = AgentManager(
    env=env,
    model=model,
    total_num_agents=num_agents,
    with_feedback=with_feedback,
    with_notes=with_notes,
    look_ahead_steps=look_ahead_steps,
    agent_type='auction',  # Specify auction agent type
    budget=budget,
    costs=cost_matrix
)

# Reset the environment
obs = env.reset()

# Run the simulation
for step in range(max_steps):
    # Get actions from the agent manager
    actions = agent_manager.step(obs, step, verbose=verbose)
    
    # Execute actions in the environment
    obs, success, info = env.step(actions)
    
    # Check if game is over
    if obs.game_over:
        break
```

## Running the Test Script

A test script is provided to verify the auction-based agent system:

```bash
python -m experiment1.my_agent.test_auction --num_agents 3 --max_steps 60 --level level_1 --seed 42 --with_feedback --with_notes --verbose
```

Command-line arguments:

- `--level`: Game level (default: 'level_1')
- `--num_agents`: Number of agents (default: 3)
- `--max_steps`: Maximum number of steps (default: 60)
- `--seed`: Random seed (default: 42)
- `--model`: Language model to use (default: 'gpt-4')
- `--with_feedback`: Enable feedback
- `--with_notes`: Enable notes
- `--look_ahead_steps`: Number of steps to look ahead (default: 5)
- `--agent_type`: Agent type (default: 'auction')
- `--budget`: Budget for the simulation (default: 1000)
- `--verbose`: Enable verbose output

## Advantages of the Auction-Based System

1. **Efficient Task Allocation**: Tasks are allocated to the most suitable agents based on their location, capabilities, and workload.

2. **Adaptive Behavior**: Agents adapt their bidding strategy based on their success rate and available currency.

3. **Specialization**: Over time, agents develop specializations in certain task types, making them more efficient.

4. **Balanced Workload**: The bidding process naturally balances the workload among agents.

5. **Emergent Cooperation**: Agents learn to cooperate implicitly through the bidding process, with more capable agents taking on more complex tasks.

## Implementation Details

### AuctionAgent

The AuctionAgent extends the base Agent class with auction-specific functionality:

- Currency management
- Bid calculation
- Task performance tracking
- Specialization development

### AuctionSystem

The AuctionSystem manages the auction process:

- Task decomposition
- Bid collection
- Task allocation
- Reward distribution

### AgentManager

The AgentManager is updated to support the auction-based agent system:

- Creates AuctionAgent instances
- Creates the AuctionSystem
- Handles the auction step process 