import random
import numpy as np

def generate_costs_seeded(num_agents, seed=42):
    if num_agents < 1:
        return []

    # Set the seed for reproducibility
    random.seed(seed)

    # Define a fixed range for costs
    max_cost = 10
    min_cost = 1

    # Generate a descending distribution of costs within the fixed range
    costs = np.linspace(max_cost, min_cost, num_agents).astype(int).tolist()

    return costs

def create_cost_matrix_seeded(num_agents, actions, seed=42):
    # Generate the cost brackets for each agent
    agent_costs = generate_costs_seeded(num_agents, seed)
    # print(agent_costs)

    # Create the cost matrix
    cost_matrix = []
    for agent_cost in agent_costs:
        # Assign a slightly varied cost to each action based on the average, except for 'noop'
        action_costs = {action: 0 if action == "noop" else max(1, int(agent_cost * (random.uniform(0.8, 1.2)))) for action in actions}
        cost_matrix.append(action_costs)

    return cost_matrix


def get_agent_success_percentages(num_agents, actions, cost_matrix):
    if num_agents < 1:
        return []

    # The first agent always has 100% success rate for all actions
    success_rates = [100]

    # Calculate a linear decrement for success rates with a minimum threshold
    decrement = 80 / (num_agents - 1)  # Ensures the last agent has a minimum success rate of 20%

    # Generate success rates for the remaining agents
    for i in range(1, num_agents):
        success_rates.append(max(20, success_rates[i-1] - decrement))

    # Create the success rate matrix
    success_matrix = []
    for i, agent_costs in enumerate(cost_matrix):
        action_success_rates = {}
        for action, cost in agent_costs.items():
            if i == 0:
                # First agent always has 100% success rate for all actions
                action_success_rates[action] = 100
            elif action == "noop":
                # Noop always has 100% success rate
                action_success_rates[action] = 100
            else:
                # Success rate with noticeable variation and minimum threshold
                base_success_rate = success_rates[i]
                variation = random.uniform(0.6, 1.4)  # Moderate range of variation
                action_success_rates[action] = max(20, base_success_rate - (cost / max(agent_costs.values()) * variation * 10))
        success_matrix.append(action_success_rates)

    return success_matrix

# Define the actions
actions = ["noop", "goto", "put", "get", "activate"]

# Example usage
num_agents = 5
cost_matrix_seeded = create_cost_matrix_seeded(num_agents, actions)
# Print the cost matrix
print("Costs: ", cost_matrix_seeded)

success_matrix = get_agent_success_percentages(num_agents, actions, cost_matrix_seeded)
print("Successes: ")
for i, success_rates in enumerate(success_matrix):
    print(f"Agent {i+1}: {success_rates}")

#compute the total average success rate
total_average_success_rate = 0
for i, success_rates in enumerate(success_matrix):
    total_average_success_rate += sum(success_rates.values()) / len(success_rates)
total_average_success_rate /= num_agents
print("Total average success rate: ", total_average_success_rate)