from my_agent.agent import Agent
from my_agent.agent_orchestrator import OrchestratorAgent
from my_agent.agent_planner import PlannerAgent, ExecutorAgent
from my_agent.agent import Agent

import copy
import json
class AgentManager: 
    def __init__(self, env, model, total_num_agents, with_feedback, with_notes, look_ahead_steps, agent_type, executor_model=None, ablation=False):
        
        self.agents = []
        self.total_num_agents = total_num_agents
        self.agent_type = agent_type
        self.price = 0
        self.current_plan = ""
        self.all_tokens = {}

        self.predictions = []

        if agent_type == 'orchestrator':
            self.orchestrator = OrchestratorAgent(env, model, -1, total_num_agents, with_feedback, with_notes, look_ahead_steps, ablation=ablation)
        elif agent_type == 'planner':
            self.planner = PlannerAgent(env, model, -1, total_num_agents, with_feedback, with_notes, look_ahead_steps, ablation=ablation)
        else:
            pass

        # Create agents for non-auction agent types
        for agent_id in range(total_num_agents):
            if agent_type == 'orchestrator':
                break
            elif agent_type == 'planner':
                agent = ExecutorAgent(env, executor_model, agent_id, total_num_agents, with_feedback, with_notes, look_ahead_steps, ablation=ablation)
            elif agent_type == 'individual':
                agent = Agent(env, model, agent_id, total_num_agents, with_feedback, with_notes, look_ahead_steps)
            else: 
                raise ValueError("Invalid agent type")
            self.agents.append(agent)
            
            
    def step(self, obs, step, verbose=False): 

        if self.agent_type == 'orchestrator':
            plan, price, tokens = self.orchestrate_step(obs, step, verbose=verbose)
        elif self.agent_type == 'planner':
            plan, price, tokens = self.planner_step(obs, step, verbose=verbose)
        elif self.agent_type == 'individual':
            plan, price, tokens = self.individual_step(obs, step, verbose=verbose)
        else:
            raise ValueError("Invalid agent type")

        self.price += price
        self.all_tokens = tokens

        return plan
    

    def orchestrate_step(self, obs, step, verbose=False):
        """
        The orchestrator agent will generate a plan of actions
        """

        # Get the actions from the orchestrator agent
        actions, price, tokens = self.orchestrator.step(obs, step, verbose=verbose)
        return actions, price, tokens
    

    def individual_step(self, obs, step, verbose=False):
        """
        The individual agent will generate an action for itself
        """
        all_actions = []
        total_price = 0
        all_tokens = {}
        for agent in self.agents:
            actions, price, tokens = agent.step(obs, step, verbose=verbose)
            all_actions.extend(actions)
            total_price += price
            all_tokens[f'agent{agent.agent_id}'] = tokens

        return all_actions, total_price, all_tokens


    def planner_step(self, obs, step, verbose=False):
        """
        Generate a general plan with the planner agent for all the agents when a new dish is added, removed, or completed. 
        Then generate the actions with the executor agents.
        """
        total_price = 0  # Initialize price
        all_tokens = {}
        # Check if there are dish-related events that require planning
        should_plan = (step == 0 or  # First step always needs a plan
                      obs.just_new_task or  # New dish arrived
                      obs.just_failed or  # Dish removed
                      obs.task_just_success)  # Dish completed
        
        # Generate a new plan if needed
        if should_plan:
            if verbose:
                print(f"Step {step}: Generating new plan due to dish state change")
            # Use the planner's plan method rather than step
            self.current_plan, price, tokens = self.planner.plan(obs, step, verbose=verbose)
            print(f"PLAN[step={step}]: ", self.current_plan)
            total_price += price
            all_tokens['planner'] = tokens


        # Get actions from all executor agents based on current plan
        all_actions = []
        
        for agent in self.agents:
            # Each executor gets the current plan and decides its own action
            agent_actions, price, tokens = agent.step(obs, step, self.current_plan, verbose=verbose)
            if agent_actions:
                all_actions.extend(agent_actions)
            total_price += price
            all_tokens[f'agent{agent.agent_id}'] = tokens

        return all_actions, total_price, all_tokens


    def store_history(self, filename):

        history = {}

        if self.agent_type == 'orchestrator':
            history['orchestrator'] = self.orchestrator.history
        elif self.agent_type == 'planner':
            history['planner'] = self.planner.history
        else:
            for agent in self.agents:
                history[agent.agent_id] = agent.history

        with open(filename, 'w') as f:
            json.dump(history, f)


