import pandas as pd
import matplotlib.pyplot as plt
import numpy as np
import textwrap

def plot_doughnut_chart(csv_path):
    # 读取CSV文件
    df = pd.read_csv(csv_path)
    
    # 提取数据并排序
    # 先按记录数排序，然后重新排列使小值分散开
    df_sorted = df.sort_values('Record_Count', ascending=False)
    
    # 创建新的排序方式，将大小值交替排列
    indices = []
    half = len(df_sorted) // 2
    for i in range(half):
        indices.append(i)  # 大值
        indices.append(len(df_sorted) - 1 - i)  # 小值
    # 如果有奇数个元素，添加中间的元素
    if len(df_sorted) % 2 != 0:
        indices.append(half)
    
    # 应用新的排序
    df_reordered = df_sorted.iloc[indices].reset_index(drop=True)
    
    fields = df_reordered['Field']
    record_counts = df_reordered['Record_Count']
    
    # 创建环形图
    fig, ax = plt.subplots(figsize=(16, 14))
    
    # 设置颜色
    colors = plt.cm.Spectral(np.linspace(0, 1, len(record_counts)))
    
    # 绘制饼图
    wedges, texts = ax.pie(
        record_counts, 
        labels=None,  # 不使用默认标签
        colors=colors,
        startangle=90, 
        wedgeprops=dict(width=0.92)  # 增大环的宽度（1.0表示实心，0表示无宽度）
    )
    
    # 绘制中心的白色圆圈
    centre_circle = plt.Circle((0, 0), 0.08, fc='white')  # 减小中心圆圈（从0.25改为0.08）
    fig.gca().add_artist(centre_circle)
    
    # 确保环形图是圆形的
    ax.axis('equal')  
    
    # 添加沿着环形弧度的标签
    for i, wedge in enumerate(wedges):
        # 获取楔形的角度中点
        ang = (wedge.theta2 - wedge.theta1) / 2. + wedge.theta1
        
        # 转换为弧度
        ang_rad = np.deg2rad(ang)
        
        # 计算标签位置（在环形上）- 调整标签位置
        if len(fields[i]) > 20:
            x = 0.55 * np.cos(ang_rad)  # 长文本放得更远一些
            y = 0.55 * np.sin(ang_rad)
            text_size = 24  # 长文本使用更小的字号
        else:
            x = 0.63 * np.cos(ang_rad)
            y = 0.63 * np.sin(ang_rad)
            text_size = 28  # 短文本保持原来的字号
        
        # 计算文本角度
        rotation = ang
        if ang > 90 and ang < 270:
            rotation = ang + 180
        
        # 直接使用原始标签，不进行换行处理
        label = fields[i]
            
        # 添加标签，使用动态字号
        ax.text(x, y, label,
                rotation=rotation, 
                size=text_size,  # 使用动态字号
                ha='center', 
                va='center',
                fontweight='bold',
                rotation_mode='anchor')
        
        # 添加百分比（放到圈外）
        percentage = f'{record_counts[i] / sum(record_counts) * 100:.1f}%'
        
        # 百分比位置放到圈外 - 调整百分比位置
        x_pct = 1.15 * np.cos(ang_rad)  # 从1.075改为1.15
        y_pct = 1.15 * np.sin(ang_rad)
        
        ax.text(x_pct, y_pct, percentage, 
                rotation=rotation, 
                size=28,  # 放大字体
                ha='center', 
                va='center',
                color='black',
                weight='bold',
                rotation_mode='anchor')
    
    # 设置标题
    # plt.title('Record Distribution by Field', fontsize=20)
    
    # 显示图表
    plt.tight_layout()
    plt.savefig('Dataset/statistics/sunburst.png', dpi=300, bbox_inches='tight')
    print("环形图已保存到: Dataset/statistics/sunburst.png")

def main():
    csv_path = 'Dataset/statistics/journal_label_2.csv'
    plot_doughnut_chart(csv_path)

if __name__ == "__main__":
    main()