import pandas as pd
import argparse
import os
import random
from tqdm import tqdm

def merge_and_shuffle_csv(input_files, output_path, seed=None):
    """
    将多个CSV文件拼接起来，打乱所有行的顺序，然后保存到指定路径
    
    Args:
        input_files: 输入CSV文件路径列表
        output_path: 输出CSV文件路径
        seed: 随机种子，用于重现结果
    """
    # 设置随机种子
    if seed is not None:
        random.seed(seed)
    
    # 存储所有数据
    all_data = []
    total_rows = 0
    
    # 读取所有CSV文件
    print("正在读取CSV文件...")
    for file_path in tqdm(input_files):
        if not os.path.exists(file_path):
            print(f"警告: 文件不存在，已跳过: {file_path}")
            continue
            
        try:
            df = pd.read_csv(file_path)
            rows = len(df)
            all_data.append(df)
            total_rows += rows
            print(f"已读取 {file_path}: {rows} 行")
        except Exception as e:
            print(f"错误: 读取文件 {file_path} 时出错: {str(e)}")
    
    if not all_data:
        print("错误: 没有成功读取任何CSV文件")
        return False
    
    # 拼接所有数据
    print(f"正在拼接 {len(all_data)} 个CSV文件，共 {total_rows} 行...")
    merged_df = pd.concat(all_data, ignore_index=True)
    
    # 打乱行顺序
    print("正在打乱行顺序...")
    shuffled_df = merged_df.sample(frac=1.0).reset_index(drop=True)
    
    # 确保输出目录存在
    output_dir = os.path.dirname(output_path)
    if output_dir and not os.path.exists(output_dir):
        os.makedirs(output_dir)
    
    # 保存到输出文件
    print(f"正在保存到 {output_path}...")
    shuffled_df.to_csv(output_path, index=False)
    
    print(f"完成! 已将 {len(shuffled_df)} 行数据保存到 {output_path}")
    return True

def main():
    parser = argparse.ArgumentParser(description="将多个CSV文件拼接起来，打乱所有行的顺序，然后保存到指定路径")
    parser.add_argument("--base_path", type=str, help="输入CSV文件路径列表")
    parser.add_argument("--output", required=True, help="输出CSV文件路径")
    parser.add_argument("--seed", type=int, help="随机种子，用于重现结果")
    
    args = parser.parse_args()
    input_files = [
        f"./Data/Understanding/ACS/{args.base_path}/full_dataset.csv",
        f"./Data/Understanding/Cell/{args.base_path}/full_dataset.csv",
        f"./Data/Understanding/Science/{args.base_path}/full_dataset.csv",
        f"./Data/Understanding/Nature/{args.base_path}/full_dataset.csv",
    ]

    try:
        merge_and_shuffle_csv(input_files, args.output, args.seed)
    except Exception as e:
        print(f"执行出错: {str(e)}")

if __name__ == "__main__":
    main()