from inspect import isfunction
import math
import torch
import torch.nn.functional as F
from torch import nn, einsum
from einops import rearrange, repeat
from typing import Optional, Any
from time import time
from .util import checkpoint, timestep_embedding
import numpy as np

try:
    import xformers
    import xformers.ops
    XFORMERS_IS_AVAILBLE = True
except:
    XFORMERS_IS_AVAILBLE = False

# CrossAttn precision handling
import os
_ATTN_PRECISION = os.environ.get("ATTN_PRECISION", "fp32")

def exists(val):
    return val is not None


def uniq(arr):
    return{el: True for el in arr}.keys()


def default(val, d):
    if exists(val):
        return val
    return d() if isfunction(d) else d


def max_neg_value(t):
    return -torch.finfo(t.dtype).max


def init_(tensor):
    dim = tensor.shape[-1]
    std = 1 / math.sqrt(dim)
    tensor.uniform_(-std, std)
    return tensor


# feedforward
class GEGLU(nn.Module):
    def __init__(self, dim_in, dim_out):
        super().__init__()
        self.proj = nn.Linear(dim_in, dim_out * 2)

    def forward(self, x):
        x, gate = self.proj(x).chunk(2, dim=-1)
        return x * F.gelu(gate)


class FeedForward(nn.Module):
    def __init__(self, dim, dim_out=None, mult=4, glu=False, dropout=0.):
        super().__init__()
        inner_dim = int(dim * mult)
        dim_out = default(dim_out, dim)
        project_in = nn.Sequential(
            nn.Linear(dim, inner_dim),
            nn.GELU()
        ) if not glu else GEGLU(dim, inner_dim)

        self.net = nn.Sequential(
            project_in,
            nn.Dropout(dropout),
            nn.Linear(inner_dim, dim_out)
        )

    def forward(self, x):
        return self.net(x)


def zero_module(module):
    """
    Zero out the parameters of a module and return it.
    """
    for p in module.parameters():
        p.detach().zero_()
    return module


def Normalize(in_channels):
    return torch.nn.GroupNorm(num_groups=32, num_channels=in_channels, eps=1e-6, affine=True)


class CrossAttention(nn.Module):
    def __init__(self, query_dim, context_dim=None, heads=8, dim_head=64, dropout=0.):
        super().__init__()
        inner_dim = dim_head * heads
        context_dim = default(context_dim, query_dim)

        self.scale = dim_head ** -0.5
        self.heads = heads

        self.to_q = nn.Linear(query_dim, inner_dim, bias=False)
        self.to_k = nn.Linear(context_dim, inner_dim, bias=False)
        self.to_v = nn.Linear(context_dim, inner_dim, bias=False)

        self.to_out = nn.Sequential(
            nn.Linear(inner_dim, query_dim),
            nn.Dropout(dropout)
        )

    def forward(self, x, context=None, mask=None):
        h = self.heads

        q = self.to_q(x)
        context = default(context, x)
        k = self.to_k(context)
        v = self.to_v(context)

        q, k, v = map(lambda t: rearrange(t, 'b n (h d) -> (b h) n d', h=h), (q, k, v))

        # force cast to fp32 to avoid overflowing
        if _ATTN_PRECISION =="fp32":
            with torch.autocast(enabled=False, device_type = 'cuda'):
                q, k = q.float(), k.float()
                sim = einsum('b i d, b j d -> b i j', q, k) * self.scale
        else:
            sim = einsum('b i d, b j d -> b i j', q, k) * self.scale
        
        del q, k
    
        if exists(mask):
            mask = rearrange(mask, 'b ... -> b (...)')
            max_neg_value = -torch.finfo(sim.dtype).max
            mask = repeat(mask, 'b j -> (b h) () j', h=h)
            sim.masked_fill_(~mask, max_neg_value)

        # attention, what we cannot get enough of
        sim = sim.softmax(dim=-1)

        out = einsum('b i j, b j d -> b i d', sim, v)
        out = rearrange(out, '(b h) n d -> b n (h d)', h=h)
        return self.to_out(out)


class MemoryEfficientCrossAttention(nn.Module):
    # https://github.com/MatthieuTPHR/diffusers/blob/d80b531ff8060ec1ea982b65a1b8df70f73aa67c/src/diffusers/models/attention.py#L223
    def __init__(self, query_dim, context_dim=None, heads=8, dim_head=64, dropout=0.0):
        super().__init__()
        # print(f"Setting up {self.__class__.__name__}. Query dim is {query_dim}, context_dim is {context_dim} and using "
        #       f"{heads} heads.")
        inner_dim = dim_head * heads
        context_dim = default(context_dim, query_dim)

        self.heads = heads
        self.dim_head = dim_head

        self.to_q = nn.Linear(query_dim, inner_dim, bias=False)
        self.to_k = nn.Linear(context_dim, inner_dim, bias=False)
        self.to_v = nn.Linear(context_dim, inner_dim, bias=False)

        self.to_out = nn.Sequential(nn.Linear(inner_dim, query_dim), nn.Dropout(dropout))
        self.attention_op: Optional[Any] = None

    def forward(self, x, context=None, mask=None):
        q = self.to_q(x)
        context = default(context, x)
        k = self.to_k(context)
        v = self.to_v(context)

        b, _, _ = q.shape
        # print(q.shape, k.shape, v.shape, self.heads)
        q = rearrange(q, 'b l (h c) -> b l h c', h=self.heads)
        k = rearrange(k, 'b l (h c) -> b l h c', h=self.heads)
        v = rearrange(v, 'b l (h c) -> b l h c', h=self.heads)
        # q, k, v = map(
        #     lambda t: t.unsqueeze(3)
        #     .reshape(b, t.shape[1], self.heads, self.dim_head)
        #     .permute(0, 2, 1, 3)
        #     .reshape(b * self.heads, t.shape[1], self.dim_head)
        #     .contiguous(),
        #     (q, k, v),
        # )

        # actually compute the attention, what we cannot get enough of
        # if x.shape == torch.Size([2, 256, 1280]):
        #     qt = q[0]
        #     kt = k[0]
        #     qt = qt.permute(1, 0, 2)
        #     kt = kt.permute(1, 2, 0)
        #     score = qt.matmul(kt)
        #     if score.shape == torch.Size([20, 256, 256]):
        #         # score = score[:, :16, :16]
        #         score_sum = np.zeros([20, 4, 4])
        #         for x in range(4):
        #             for y in range(4):
        #                 if x == 0 or y == 0:
        #                     for i in range(4):
        #                         for j in range(4):
        #                             score_sum[:, i, j] += score[:, x*4 + i * 4: x*4 + i * 4 + 4, y *4 + j * 4: y * 4+j * 4 + 4].mean().cpu().item() 
        #         with open('av/avs', 'r') as file:
        #             number = int(file.read().strip())
        #         if number == 50:
        #             number = 0
        #         np.save(os.path.join('av/s', str(number) + '.npy'), score_sum.astype(np.float32))
        #         number += 1
        #         with open('av/avs', 'w') as file:
        #             file.write(str(number))
        #     else:
        #         # score = score[:, :16, :]
        #         score_sum = np.zeros([20, 4, 1])
        #         for x in range(4):
        #             for i in range(4):
        #                 score_sum[:, i] += score[:, x * 4 + i * 4: x * 4 +i * 4 + 4, :].mean().cpu().item() /100
        #         with open('av/avc', 'r') as file:
        #             number = int(file.read().strip())
        #         if number == 50:
        #             number = 0
        #         np.save(os.path.join('av/c', str(number) + '.npy'), score_sum.astype(np.float32))
        #         number += 1
        #         with open('av/avc', 'w') as file:
        #             file.write(str(number))
        out = xformers.ops.memory_efficient_attention(q, k, v, attn_bias=None, op=self.attention_op)

        if exists(mask):
            raise NotImplementedError
        out = rearrange(out, 'b l h c -> b l (h c)')
        # out = (
        #     out.unsqueeze(0)
        #     .reshape(b, self.heads, out.shape[1], self.dim_head)
        #     .permute(0, 2, 1, 3)
        #     .reshape(b, out.shape[1], self.heads * self.dim_head)
        # )
        return self.to_out(out)





# model = MemoryEfficientCrossAttention(4).cuda().half()
# def cal(shape):
#     inn = torch.rand(shape).cuda().half()
#     t = time()
#     for i in range(1000):
#         model(inn)
#     return time() - t

# data = []
# for i in [1,2,4,8,16,32,64,128,256,512,1024,2048,4096,16384]:
#     data.append([i, cal([i, 16384 // i, 4])])
# print(data)
# import numpy as np
# np.save('a.npy', data)

class BasicTransformerBlock(nn.Module):
    ATTENTION_MODES = {
        "softmax": CrossAttention,  # vanilla attention
        "softmax-xformers": MemoryEfficientCrossAttention
    }
    def __init__(self, dim, n_heads, d_head, dropout=0., context_dim=None, gated_ff=True, checkpoint=True,
                 disable_self_attn=False):
        super().__init__()
        attn_mode = "softmax-xformers" if XFORMERS_IS_AVAILBLE else "softmax"
        assert attn_mode in self.ATTENTION_MODES
        attn_cls = self.ATTENTION_MODES[attn_mode]
        self.disable_self_attn = disable_self_attn
        self.attn1 = attn_cls(query_dim=dim, heads=n_heads, dim_head=d_head, dropout=dropout,
                              context_dim=context_dim if self.disable_self_attn else None)  # is a self-attention if not self.disable_self_attn
        self.ff = FeedForward(dim, dropout=dropout, glu=gated_ff)
        self.attn2 = attn_cls(query_dim=dim, context_dim=context_dim,
                              heads=n_heads, dim_head=d_head, dropout=dropout)  # is self-attn if context is none
        self.norm1 = nn.LayerNorm(dim)
        self.norm2 = nn.LayerNorm(dim)
        self.norm3 = nn.LayerNorm(dim)
        self.checkpoint = checkpoint
    
    # def get_sp_position_encoding(height, width, channels):
    #     pe = torch.zeros(height, width, channels, dtype=torch.float64).cuda().half()
    #     for y in range(height):
    #         for x in range(width):
    #             for c in range(channels):
    #                 pe[y, x, c] = math.sin(x / (10000 ** (c / channels))) + math.cos(y / (10000 ** (c / channels)))
    #     return pe


    # def get_te_position_encoding(height, width, channels):
    #     pe = torch.zeros(height, width, channels, dtype=torch.float64).cuda().half()
    #     for y in range(height):
    #         for x in range(width):
    #             for c in range(channels):
    #                 pe[y, x, c] = math.sin((x * width) + y / (10000 ** (c / channels))) + math.cos((x * width) + y / (10000 ** (c / channels)))
    #     return pe
    
    # def pe(self, x):
    #     big_img = torch.zeros_like(x).cuda().half()
    #     small_img_pe = self.get_sp_position_encoding(x.shape[0] / 4, x.shape[1] / 4, x.shape[2])
    #     for i in range(4):
    #         for j in range(4):
    #             big_img[i*32:(i+1)*32, j*32:(j+1)*32, :2] += small_img_pe
    #     big_img_pe = self.get_te_position_encoding(4, 4, x.shape[2])
    #     for i in range(4):
    #         for j in range(4):
    #             big_img[i*32:(i+1)*32, j*32:(j+1)*32, 2:] += big_img_pe[i, j, :]
    #     return big_img

    def forward(self, x, context=None):
        return checkpoint(self._forward, (x, context), self.parameters(), self.checkpoint)

    def _forward(self, x, context=None):
        # x = x + self.pe(x)
        x = self.attn1(self.norm1(x), context=context if self.disable_self_attn else None) + x
        x = self.attn2(self.norm2(x), context=context) + x
        x = self.ff(self.norm3(x)) + x
        return x


class SpatialTransformer(nn.Module):
    """
    Transformer block for image-like data.
    First, project the input (aka embedding)
    and reshape to b, t, d.
    Then apply standard transformer action.
    Finally, reshape to image
    NEW: use_linear for more efficiency instead of the 1x1 convs
    """
    def __init__(self, in_channels, n_heads, d_head,
                 depth=1, dropout=0., context_dim=None,
                 disable_self_attn=False, use_linear=False,
                 use_checkpoint=True):
        super().__init__()
        self.context_dim = context_dim
        if exists(context_dim) and not isinstance(context_dim, list):
            context_dim = [context_dim]
        self.in_channels = in_channels
        self.n_heads = n_heads
        self.d_head = d_head
        inner_dim = n_heads * d_head
        self.depth = depth
        self.dropout = dropout
        self.disable_self_attn = disable_self_attn
        self.use_checkpoint = use_checkpoint
        self.norm = Normalize(in_channels)
        if not use_linear:
            self.proj_in = nn.Conv2d(in_channels,
                                     inner_dim,
                                     kernel_size=1,
                                     stride=1,
                                     padding=0)
        else:
            self.proj_in = nn.Linear(in_channels, inner_dim)

        self.transformer_blocks = nn.ModuleList(
            [BasicTransformerBlock(inner_dim, n_heads, d_head, dropout=dropout, context_dim=context_dim[d],
                                   disable_self_attn=disable_self_attn, checkpoint=use_checkpoint)
                for d in range(depth)]
        )
        if not use_linear:
            self.proj_out = zero_module(nn.Conv2d(inner_dim,
                                                  in_channels,
                                                  kernel_size=1,
                                                  stride=1,
                                                  padding=0))
        else:
            self.proj_out = zero_module(nn.Linear(in_channels, inner_dim))
        self.use_linear = use_linear

    def forward(self, x, context=None):
        # note: if no context is given, cross-attention defaults to self-attention
        if not isinstance(context, list):
            context = [context]
        b, c, h, w = x.shape
        # x = rearrange(x, 'b c f h w -> (b f) c h w')
        x_in = x
        x = self.norm(x)
        if not self.use_linear:
            x = self.proj_in(x)
        x = rearrange(x, 'b c h w -> b (h w) c').contiguous()
        if self.use_linear:
            x = self.proj_in(x)
        for i, block in enumerate(self.transformer_blocks):
            # ctx = repeat(context[i], 'b l c -> (b f) l c', f=f)
            x = block(x, context=context[i])
        if self.use_linear:
            x = self.proj_out(x)
        x = rearrange(x, 'b (h w) c -> b c h w', h=h, w=w).contiguous()
        if not self.use_linear:
            x = self.proj_out(x)
        out = x + x_in
        # out = rearrange(out, '(b f) c h w -> b c f h w', b=b)
        return out


class TemporalTransformer(nn.Module):
    """
    Transformer block for image-like data.
    First, project the input (aka embedding)
    and reshape to b, t, d.
    Then apply standard transformer action.
    Finally, reshape to image
    NEW: use_linear for more efficiency instead of the 1x1 convs
    """
    def __init__(self, in_channels, n_heads, d_head,
                 depth=1, dropout=0., context_dim=None,
                 disable_self_attn=False, use_linear=False,
                 use_checkpoint=True):
        super().__init__()
        if exists(context_dim) and not isinstance(context_dim, list):
            context_dim = [context_dim]
        self.in_channels = in_channels
        self.n_heads = n_heads
        self.d_head = d_head
        inner_dim = n_heads * d_head
        self.depth = depth
        self.dropout = dropout
        self.disable_self_attn = disable_self_attn
        self.use_checkpoint = use_checkpoint
        self.norm = Normalize(in_channels)
        if not use_linear:
            self.proj_in = nn.Conv1d(in_channels,
                                     inner_dim,
                                     kernel_size=1,
                                     stride=1,
                                     padding=0)
        else:
            self.proj_in = nn.Linear(in_channels, inner_dim)

        self.transformer_blocks = nn.ModuleList(
            [BasicTransformerBlock(inner_dim, n_heads, d_head, dropout=dropout, context_dim=context_dim[d],
                                   disable_self_attn=disable_self_attn, checkpoint=use_checkpoint)
                for d in range(depth)]
        )
        if not use_linear:
            self.proj_out = zero_module(nn.Conv1d(inner_dim,
                                                  in_channels,
                                                  kernel_size=1,
                                                  stride=1,
                                                  padding=0))
        else:
            self.proj_out = zero_module(nn.Linear(in_channels, inner_dim))
        self.use_linear = use_linear

    def forward(self, x, context=None):
        # note: if no context is given, cross-attention defaults to self-attention
        if not isinstance(context, list):
            context = [context]
        b, c, f, h, w = x.shape
        x = rearrange(x, 'b c f h w -> (b h w) c f')
        x_in = x
        x = self.norm(x)
        if not self.use_linear:
            x = self.proj_in(x)
        x = rearrange(x, 'b c f -> b f c').contiguous()
        if self.use_linear:
            x = self.proj_in(x)
        t = torch.arange(0, x.shape[1], 1, dtype=torch.int64, device=x.device)
        t_emb = timestep_embedding(t, x.shape[2], repeat_only=False)
        x = x + t_emb.to(x.dtype).unsqueeze(0)
        for i, block in enumerate(self.transformer_blocks):
            ctx = repeat(context[i], 'b l c -> (b h w) l c', h=h, w=w)
            x = block(x, context=ctx)
        if self.use_linear:
            x = self.proj_out(x)
        x = rearrange(x, 'b f c -> b c f').contiguous()
        if not self.use_linear:
            x = self.proj_out(x)
        out = x + x_in
        out = rearrange(out, '(b h w) c f -> b c f h w', b=b, h=h, w=w)
        return out
