function X = f_prox_TNN_dct(X, rho)
% f_prox_TNN_dct  Apply proximal operator of the tensor nuclear norm (TNN)
%                 in the DCT (Discrete Cosine Transform) domain.
%
%   This function computes the proximal operator of the tensor nuclear norm
%   (TNN) applied slice-wise after DCT transform along the 3rd dimension.
%
%   Input:
%       X   - Input 3D tensor (e.g., height × width × frames)
%       rho - Regularization parameter (threshold level for singular values)
%
%   Output:
%       X   - Tensor after applying proximal TNN operator in DCT domain

% Store original tensor size
sz = size(X);

% Step 1: Apply DCT along the 3rd dimension (tube direction)
X = f_dct_tube(X);  % Assumes f_dct_tube applies DCT along mode-3

% Step 2: Apply soft-thresholded SVD to each frontal slice in DCT domain
for i = 1:sz(3)
    [U, S, V] = svd(X(:,:,i), 'econ');              % SVD on each frontal slice
    S = f_prox_l1(diag(S), rho);                    % Soft-threshold singular values
    X(:,:,i) = U * diag(S) * V';                    % Reconstruct thresholded slice
end

% Step 3: Apply inverse DCT along the 3rd dimension
X = f_idct_tube(X);  % Assumes f_idct_tube inverts the DCT transform
end
