import numpy as np
import pandas as pd

def normalize_yaw(yaw):
    """
    Normalize Yaw to [-180, 180].
    """
    return ((yaw + 180) % 360) - 180

def generate_views(num_points=1000, output_file="views.csv"):
    """
    Generate approximately `num_points` uniformly distributed 3D views
    using spherical coordinates. Adds Roll column with value 0.
    Outputs the result to a CSV file.
    """
    points = []
    golden_ratio = (1 + np.sqrt(5)) / 2  # Fibonacci ratio
    angle_increment = 2 * np.pi / golden_ratio  # Angle increment

    for i in range(num_points):
        # Calculate spherical coordinates
        z = 1 - 2 * (i + 0.5) / num_points  # Map to [-1, 1] for z
        theta = np.arccos(z)  # Polar angle (latitude)
        phi = (i % num_points) * angle_increment  # Azimuthal angle (longitude)

        # Convert spherical to Cartesian coordinates for debugging (optional)
        x = np.sin(theta) * np.cos(phi)
        y = np.sin(theta) * np.sin(phi)
        z = np.cos(theta)

        # Convert to Pitch, Yaw
        pitch = np.degrees(theta - np.pi / 2)  # Convert polar angle to pitch
        yaw = np.degrees(phi) - 180           # Convert azimuth to yaw

        points.append([round(pitch, 1), round(yaw, 1), 0])  # Append Roll = 0

    # Save to DataFrame and CSV
    df = pd.DataFrame(points, columns=["Pitch", "Yaw", "Roll"])
    df.to_csv(output_file, index=False)
    print(f"Generated {num_points} views and saved to {output_file}")

# Generate views and save to a file
generate_views(num_points=1000, output_file="views_with_roll.csv")
