import time
import os

import numpy as np
import torch
import gym

from typing import Optional, Dict, List
from tqdm import tqdm
from collections import deque
from offlinerlkit.buffer import ReplayBuffer
from offlinerlkit.utils.logger import Logger
from offlinerlkit.policy import BasePolicy
import wandb

# model-free policy trainer
class MFPolicyTrainer:
    def __init__(
        self,
        policy: BasePolicy,
        eval_env: gym.Env,
        buffer: ReplayBuffer,
        logger: Logger,
        epoch: int = 1500,
        warmup_epoch: int = -1,
        step_per_epoch: int = 1000,
        batch_size: int = 256,
        osd_batch_size: int = 512,
        eval_episodes: int = 10,
        normalize_obs: bool = False,
        lr_scheduler: Optional[torch.optim.lr_scheduler._LRScheduler] = None,
        use_wandb: int = 0,
        dataset_format: str = '',
        saved_dirs: str = '',
        dataset_type: str = ''
    ) -> None:
        self.policy = policy
        self.eval_env = eval_env
        self.buffer = buffer
        self.logger = logger
        self.saved_dirs = saved_dirs

        self._epoch = epoch
        self._step_per_epoch = step_per_epoch
        self._batch_size = batch_size
        self._eval_episodes = eval_episodes
        self._normalize_obs = normalize_obs
        if normalize_obs:
            self._obs_mean, self._obs_std = self.buffer.normalize_obs()
        self.lr_scheduler = lr_scheduler
        self.use_wandb = use_wandb
        self.warmup_epoch = warmup_epoch
        self._osd_batch_size = osd_batch_size
        self.dataset_type = dataset_type


    def train(self) -> Dict[str, float]:
        start_time = time.time()

        num_timesteps = 0
        last_10_performance = deque(maxlen=10)
        # train loop
        for e in range(1, self._epoch + 1):

            self.policy.train()


            pbar = tqdm(range(self._step_per_epoch), desc=f"Epoch #{e}/{self._epoch}")
            for it in pbar:

                before = time.time()
                if e < self.warmup_epoch:
                    batch = self.buffer.sample(self._osd_batch_size)
                    init_batch = self.buffer.sample_init(self._osd_batch_size)
                    loss = self.policy.learn_osd(init_batch, batch)
                else:
                    batch = self.buffer.sample(self._batch_size)
                    init_batch = self.buffer.sample_init(self._batch_size)
                    loss = self.policy.learn(init_batch, batch)
                # loss = self.policy.learn( batch)
                pbar.set_postfix(**loss)

                for k, v in loss.items():
                    self.logger.logkv_mean(k, v)

                    if self.use_wandb:
                        wandb.log({k: v}, step=e)          
                        # print(k,v) 
                num_timesteps += 1

            if self.lr_scheduler is not None:
                self.lr_scheduler.step()
            
            # evaluate current policy
            eval_info = self._evaluate()
            print(eval_info)
            ep_reward_mean, ep_reward_std = np.mean(eval_info["eval/episode_reward"]), np.std(eval_info["eval/episode_reward"])
            ep_length_mean, ep_length_std = np.mean(eval_info["eval/episode_length"]), np.std(eval_info["eval/episode_length"])
            norm_ep_rew_mean = self.eval_env.get_normalized_score(ep_reward_mean) * 100
            norm_ep_rew_std = self.eval_env.get_normalized_score(ep_reward_std) * 100
            last_10_performance.append(norm_ep_rew_mean)
            self.logger.logkv("eval/normalized_episode_reward", norm_ep_rew_mean)
            self.logger.logkv("eval/normalized_episode_reward_std", norm_ep_rew_std)
            self.logger.logkv("eval/episode_length", ep_length_mean)
            self.logger.logkv("eval/episode_length_std", ep_length_std)
            self.logger.set_timestep(num_timesteps)
            self.logger.dumpkvs()
            if self.use_wandb:
                wandb.log({"eval/normalized_episode_reward":norm_ep_rew_mean}, step=e)
                wandb.log({"eval/normalized_episode_reward_std": norm_ep_rew_std}, step=e)
                wandb.log({"eval/episode_length": ep_length_mean}, step=e)
                wandb.log({"eval/episode_length_std": ep_length_std}, step=e)        
                if e > 10:
                    wandb.log({"eval/last_10_performance": np.mean(last_10_performance)}, step=e)
            # save checkpoint
            torch.save(self.policy.state_dict(), os.path.join(self.logger.checkpoint_dir, "policy.pth"))
            # self._only_optidice = False
            if self.saved_dirs is not '':
                print(self.saved_dirs)
                self.policy.save_models(os.path.join(self.saved_dirs, "osd_dice.pth"))

        self.logger.log("total time: {:.2f}s".format(time.time() - start_time))

        self.logger.close()

        return {"last_10_performance": np.mean(last_10_performance)}

    def _evaluate(self) -> Dict[str, List[float]]:
        self.policy.eval()
        obs = self.eval_env.reset()
        eval_ep_info_buffer = []
        num_episodes = 0
        episode_reward, episode_length = 0, 0
        
        while num_episodes < self._eval_episodes:
            if self._normalize_obs:
                obs = (np.array(obs).reshape(1,-1) - self._obs_mean) / self._obs_std
            # if self.dataset_type == 'dice':
                # obs= np.append((obs - self._obs_mean) / (self._obs_std + 1e-10), 0)
                # obs= np.append(obs, 0)
            action = self.policy.select_action(obs, deterministic=True)
            next_obs, reward, terminal, _ = self.eval_env.step(action.flatten())
            episode_reward += reward
            episode_length += 1

            obs = next_obs

            if terminal:
                eval_ep_info_buffer.append(
                    {"episode_reward": episode_reward, "episode_length": episode_length}
                )
                num_episodes +=1
                episode_reward, episode_length = 0, 0
                obs = self.eval_env.reset()
        
        return {
            "eval/episode_reward": [ep_info["episode_reward"] for ep_info in eval_ep_info_buffer],
            "eval/episode_length": [ep_info["episode_length"] for ep_info in eval_ep_info_buffer]
        }
