import argparse
import random
import os
import gym
import d4rl

import numpy as np
import torch
import wandb
import torch.nn as nn
from typing import Dict, List, Union, Tuple, Optional

from offlinerlkit.nets import MLP
from offlinerlkit.modules import Actor, Critic
from offlinerlkit.utils.noise import GaussianNoise
# from offlinerlkit.utils.load_dataset import qlearning_dataset
from buffer import qlearning_dataset, dice_dataset, ReplayBuffer
from offlinerlkit.utils.scaler import StandardScaler
# from offlinerlkit.buffer import ReplayBuffer
from offlinerlkit.utils.logger import Logger, make_log_dirs
# from offlinerlkit.policy_trainer import MFPolicyTrainer
from mf_policy_trainer import MFPolicyTrainer
from osd_td3bc import OSDTD3BCPolicy



class ValueNetwork(nn.Module):
    def __init__(self, backbone: nn.Module,  device: str = "cpu", is_pos_output: bool=False) -> None:
        super().__init__()

        self.device = torch.device(device)
        self.backbone = backbone.to(device)
        latent_dim = getattr(backbone, "output_dim")
        self.last = nn.Linear(latent_dim, 1).to(device)
        self.is_pos_output = is_pos_output


    def forward(
        self,
        obs: Union[np.ndarray, torch.Tensor],
        actions: Optional[Union[np.ndarray, torch.Tensor]] = None
    ) -> torch.Tensor:
        obs = torch.as_tensor(obs, device=self.device, dtype=torch.float32)
        if actions is not None:
            actions = torch.as_tensor(actions, device=self.device, dtype=torch.float32).flatten(1)
            obs = torch.cat([obs, actions], dim=1)
        # print('obs.shape', obs.shape)
        logits = self.backbone(obs)
        values = self.last(logits)
        if self.is_pos_output:
            values = torch.square(values)
        return values


    def load_params(self, state_dict):
        for key in self.state_dict().keys():
            self.state_dict()[key] = state_dict[key]
        # for name, param in state_dict.named_parameters():
        #     print(name,param.mean())




def get_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--algo-name", type=str, default="osd-td3bc")
    parser.add_argument("--task", type=str, default="antmaze-umaze-v0")
    parser.add_argument("--seed", type=int, default=0)
    parser.add_argument("--actor-lr", type=float, default=3e-4)
    parser.add_argument("--critic-lr", type=float, default=3e-4)
    parser.add_argument("--gamma", type=float, default=0.99)
    parser.add_argument("--tau", type=float, default=0.005)
    parser.add_argument("--exploration-noise", type=float, default=0.1)
    parser.add_argument("--policy-noise", type=float, default=0.2)
    parser.add_argument("--noise-clip", type=float, default=0.5)
    parser.add_argument("--update-actor-freq", type=int, default=2)
    parser.add_argument("--alpha", type=float, default=5.)
    parser.add_argument("--epoch", type=int, default=1500)
    parser.add_argument("--step-per-epoch", type=int, default=1000)
    parser.add_argument("--eval_episodes", type=int, default=10)
    parser.add_argument("--batch-size", type=int, default=256)
    parser.add_argument("--device", type=str, default="cuda:7" if torch.cuda.is_available() else "cpu")

    parser.add_argument("--osd_alpha", type=float, default=0.1)
    parser.add_argument('--osd_hidden_sizes', default=(256, 256))
    parser.add_argument("--osd_beta", type=float, default=1e-3)
    parser.add_argument("--weight_type", type=str, default='median')
    parser.add_argument("--osd_batch_size", type=int, default=512)
    parser.add_argument('--lower', type=float, default=0.1)
    parser.add_argument('--higher', type=float, default=10.)
    parser.add_argument("--osd-lr", type=float, default=3e-4)
    parser.add_argument("--warmup_epoch", type=int, default=500)
    parser.add_argument("--use_wandb", type=int, default=0)
    parser.add_argument("--dataset_type", type=str, default='dice')



    return parser.parse_args()


def train(args=get_args()):
    # create env and dataset
    env = gym.make(args.task)
    if args.dataset_type == 'dice':
        dataset, dataset_statistics = dice_dataset(env,standardize_reward=False)
        args.obs_shape = (env.observation_space.shape[0] + 1,)
    else:
        dataset = qlearning_dataset(env)
        args.obs_shape = env.observation_space.shape 
    
    if 'antmaze' in args.task:
        dataset["rewards"] -= 1.0
    
    
   
    print(args.obs_shape)
    args.action_dim = np.prod(env.action_space.shape)
    args.max_action = env.action_space.high[0]

    # seed
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.cuda.manual_seed_all(args.seed)
    torch.backends.cudnn.deterministic = True
    env.seed(args.seed)




    nu_backbone = MLP(input_dim=np.prod(args.obs_shape), hidden_dims=args.osd_hidden_sizes)
    nu_network= ValueNetwork(nu_backbone, args.device)
    lam_v = torch.zeros(1, requires_grad=True, device=args.device)
    nu_optimizer = torch.optim.Adam(nu_network.parameters(), lr=args.osd_lr)
    lam_v_optimizer = torch.optim.Adam([lam_v], lr=args.osd_lr)
    




    # create buffer
    # buffer = ReplayBuffer(
    #     buffer_size=len(dataset["observations"]),
    #     obs_shape=args.obs_shape,
    #     obs_dtype=np.float32,
    #     action_dim=args.action_dim,
    #     action_dtype=np.float32,
    #     device=args.device
    # )
    
    buffer = ReplayBuffer(
        buffer_size=dataset["observations"].shape[0],
        obs_shape=args.obs_shape,
        obs_dtype=np.float32,
        action_dim=args.action_dim,
        action_dtype=np.float32,
        device=args.device
    )
    buffer.load_dataset(dataset)
    if args.dataset_type == 'dice':
        obs_mean, obs_std = dataset_statistics['observation_mean'], dataset_statistics['observation_std']
    else:
        obs_mean, obs_std = buffer.normalize_obs()



    # create policy model
    actor_backbone = MLP(input_dim=np.prod(args.obs_shape), hidden_dims=[256, 256])
    critic1_backbone = MLP(input_dim=np.prod(args.obs_shape)+args.action_dim, hidden_dims=[256, 256])
    critic2_backbone = MLP(input_dim=np.prod(args.obs_shape)+args.action_dim, hidden_dims=[256, 256])
    actor = Actor(actor_backbone, args.action_dim, device=args.device)

    critic1 = Critic(critic1_backbone, args.device)
    critic2 = Critic(critic2_backbone, args.device)
    actor_optim = torch.optim.Adam(actor.parameters(), lr=args.actor_lr)
    critic1_optim = torch.optim.Adam(critic1.parameters(), lr=args.critic_lr)
    critic2_optim = torch.optim.Adam(critic2.parameters(), lr=args.critic_lr)

    # scaler for normalizing observations
    scaler = StandardScaler(mu=obs_mean, std=obs_std)

    # create policy
    policy = OSDTD3BCPolicy(
        actor,
        critic1,
        critic2,
        actor_optim,
        critic1_optim,
        critic2_optim,
        nu_network=nu_network,
        nu_optimizer = nu_optimizer,
        lam_v = lam_v,
        lam_v_optimizer =lam_v_optimizer,
        osd_alpha=args.osd_alpha,
        osd_beta = args.osd_beta,
        weight_type = args.weight_type,
        lower = args.lower,
        higher = args.higher,
        dataset_type = args.dataset_type,
        tau=args.tau,
        gamma=args.gamma,
        max_action=args.max_action,
        exploration_noise=GaussianNoise(sigma=args.exploration_noise),
        policy_noise=args.policy_noise,
        noise_clip=args.noise_clip,
        update_actor_freq=args.update_actor_freq,
        alpha=args.alpha,
        scaler=scaler
    )

    # log
    log_dirs = make_log_dirs(args.task, args.algo_name, args.seed, vars(args))
    # key: output file name, value: output handler type
    output_config = {
        "consoleout_backup": "stdout",
        "policy_training_progress": "csv",
        "tb": "tensorboard"
    }
    logger = Logger(log_dirs, output_config)
    logger.log_hyperparameters(vars(args))

    # args.use_wandb = 1
    if args.use_wandb:
        wandb.init(project="osd-td3bc", group='osd-td3bc',
        name=args.task, config=args)

    # create policy trainer
    policy_trainer = MFPolicyTrainer(
        policy=policy,
        eval_env=env,
        buffer=buffer,
        logger=logger,
        epoch=args.epoch,
        warmup_epoch = args.warmup_epoch,
        step_per_epoch=args.step_per_epoch,
        batch_size=args.batch_size,
        eval_episodes=args.eval_episodes,
        use_wandb=args.use_wandb,
        osd_batch_size=args.osd_batch_size
    )

    # train
    policy_trainer.train()


if __name__ == "__main__":
    train()