
# Define the International Morse code mapping.
# This dictionary covers the letters A-Z, digits 0-9,
# and a set of punctuation marks that have standard Morse code equivalents.
MORSE_CODE_DICT = {
    'A': '.-',    'B': '-...',  'C': '-.-.',  'D': '-..',
    'E': '.',     'F': '..-.',  'G': '--.',   'H': '....',
    'I': '..',    'J': '.---',  'K': '-.-',   'L': '.-..',
    'M': '--',    'N': '-.',    'O': '---',   'P': '.--.',
    'Q': '--.-',  'R': '.-.',   'S': '...',   'T': '-',
    'U': '..-',   'V': '...-',  'W': '.--',   'X': '-..-',
    'Y': '-.--',  'Z': '--..',
    '0': '-----', '1': '.----', '2': '..---', '3': '...--',
    '4': '....-', '5': '.....', '6': '-....', '7': '--...',
    '8': '---..', '9': '----.',
    '.': '.-.-.-', ',': '--..--', '?': '..--..', "'": '.----.',
    '!': '-.-.--', '/': '-..-.',  '(': '-.--.',  ')': '-.--.-',
    '&': '.-...',  ':': '---...', ';': '-.-.-.',
    '=': '-...-',  '+': '.-.-.',  '-': '-....-', '_': '..--.-',
    '"': '.-..-.', '$': '...-..-', '@': '.--.-.'
}

def convert_char_to_morse(char):
    """
    Convert a single character to Morse code.
    If the character has no Morse code representation,
    return the character itself.
    """
    # Convert letters to uppercase to match the keys in MORSE_CODE_DICT.
    if char.upper() in MORSE_CODE_DICT:
        return MORSE_CODE_DICT[char.upper()]
    else:
        return char

def convert_file(input_filename, output_filename):
    """
    Reads the content of input_filename, converts each character to its Morse code equivalent
    (if available), and writes the converted text to output_filename.
    """
    try:
        with open(input_filename, 'r', encoding='utf-8') as infile:
            text = infile.read()
    except FileNotFoundError:
        print(f"Error: {input_filename} not found.")
        return

    # Convert each character. For converted characters, add an extra space for readability.
    converted_text = ""
    for char in text:
        # Check if the character is convertible.
        converted = convert_char_to_morse(char)
        # If a conversion occurred (and the original char isn’t whitespace),
        # add a space after the Morse code for clarity.
        if char.upper() in MORSE_CODE_DICT:
            converted_text += converted + " "
        else:
            converted_text += converted

    with open(output_filename, 'w', encoding='utf-8') as outfile:
        outfile.write(converted_text)

    print(f"Conversion complete. Check the output in '{output_filename}'.")

if __name__ == "__main__":
    # Specify your input file containing the list of characters and the desired output file.
    input_file = "shakespeare_test.txt"
    output_file = "shakespeare_test_conv.txt"
    convert_file(input_file, output_file)