
import sys
from pathlib import Path

# ---------------------------------------------------------------------------
# 1.  Morse tables (identical to the encoder’s)
# ---------------------------------------------------------------------------
MORSE_CODE_DICT = {
    'A': '.-',    'B': '-...',  'C': '-.-.',  'D': '-..',
    'E': '.',     'F': '..-.',  'G': '--.',   'H': '....',
    'I': '..',    'J': '.---',  'K': '-.-',   'L': '.-..',
    'M': '--',    'N': '-.',    'O': '---',   'P': '.--.',
    'Q': '--.-',  'R': '.-.',   'S': '...',   'T': '-',
    'U': '..-',   'V': '...-',  'W': '.--',   'X': '-..-',
    'Y': '-.--',  'Z': '--..',
    '0': '-----', '1': '.----', '2': '..---', '3': '...--',
    '4': '....-', '5': '.....', '6': '-....', '7': '--...',
    '8': '---..', '9': '----.',
    '.': '.-.-.-', ',': '--..--', '?': '..--..', "'": '.----.',
    '!': '-.-.--', '/': '-..-.',  '(': '-.--.',  ')': '-.--.-',
    '&': '.-...',  ':': '---...', ';': '-.-.-.',
    '=': '-...-',  '+': '.-.-.',  '-': '-....-', '_': '..--.-',
    '"': '.-..-.', '$': '...-..-', '@': '.--.-.'
}

REVERSE_MORSE_CODE_DICT = {v: k for k, v in MORSE_CODE_DICT.items()}

# ---------------------------------------------------------------------------
# 2.  Core decoder
# ---------------------------------------------------------------------------
def convert_morse_to_text(morse_text: str) -> str:
    out = []
    i, n = 0, len(morse_text)

    while i < n:
        ch = morse_text[i]

        # Case 1: start of a Morse token
        if ch in ".-":
            j = i
            while j < n and morse_text[j] in ".-":
                j += 1
            token = morse_text[i:j]
            out.append(REVERSE_MORSE_CODE_DICT.get(token, token))
            i = j

            # Skip the *mandatory* separator space after the token
            if i < n and morse_text[i] == " ":
                i += 1
                # Any additional spaces are genuine user spaces
                while i < n and morse_text[i] == " ":
                    out.append(" ")
                    i += 1
            continue

        # Case 2: space that is not a separator → copy consecutive spaces
        if ch == " ":
            start = i
            while i < n and morse_text[i] == " ":
                i += 1
            out.append(morse_text[start:i])
            continue

        # Case 3: newline or any other char → copy verbatim
        out.append(ch)
        i += 1

    return "".join(out)

# ---------------------------------------------------------------------------
# 3.  File wrapper
# ---------------------------------------------------------------------------
def convert_file_back(input_filename: Path, output_filename: Path) -> None:
    try:
        morse_text = input_filename.read_text(encoding="utf-8")
    except FileNotFoundError:
        sys.exit(f"Error: '{input_filename}' not found.")

    decoded = convert_morse_to_text(morse_text)
    output_filename.write_text(decoded, encoding="utf-8")
    print(f"Re-conversion complete → {output_filename}")

# ---------------------------------------------------------------------------
# 4.  CLI entry-point
# ---------------------------------------------------------------------------
if __name__ == "__main__":
    in_path  = Path(sys.argv[1]) if len(sys.argv) >= 2 else Path("encoded.txt")
    out_path = Path(sys.argv[2]) if len(sys.argv) >= 3 else Path("decoded.txt")
    convert_file_back(in_path, out_path)