package org.vowpalwabbit.spark;

import common.Native;
import java.io.Closeable;
import java.util.Collection;
import java.util.Iterator;

/**
 * Main wrapper for VowpalWabbit native implementation.
 *
 * @author Markus Cozowicz
 */
public class VowpalWabbitNative implements Closeable {
    static {
        // load the native libraries
        Native.load();
    }

    /**
     * Initializes the native VW data structures.
     *
     * @param args VW command line arguments.
     * @return pointer to vw data structure defined in global_data.h.
     */
    private static native long initialize(String args);

    /**
     * Initializes the native VW data structures.
     *
     * <p>
     * Note: The {@code args} must be compatible with the command line arguments
     * stored in {@code model}.
     * </p>
     *
     * @param args  VW command line arguments.
     * @param model VW model to initialize this instance from.
     * @return pointer to vw data structure defined in global_data.h.
     */
    private static native long initializeFromModel(String args, byte[] model);

    /**
     * Invoke multi-line learning.
     *
     * @param examples the examples to learn from.
     * @return the one-step ahead prediction.
     */
    public native Object learn(VowpalWabbitExample[] examples);

    /**
     * Invoke multi-line learning. By default will interpret input as text format, pass `--dsjson` or `--json` to initialize to use DSJSON or JSON respectively.
     *
     * @param examplesString String representation of examples to learn from.
     * @return the one-step ahead prediction.
     */
    public native Object learnFromString(String examplesString);

    /**
     * Invoke multi-line prediction.
     *
     * @param examples the example to predict for.
     * @return the prediction.
     */
    public native Object predict(VowpalWabbitExample[] examples);

    /**
     * Invoke multi-line prediction. By default will interpret input as text format, pass `--dsjson` or `--json` to initialize to use DSJSON or JSON respectively.
     *
     * @param examplesString String representation of examples to learn from.
     * @return the prediction.
     */
    public native Object predictFromString(String examplesString);

    /**
     * Perform remaining passes.
     */
    public native void performRemainingPasses();

    /**
     * Returns a snapshot of the current model.
     *
     * @return serialized VW model.
     */
    public native byte[] getModel();

    /**
     * Returns a subset of the current arguments VW received (e.g. numbits)
     *
     * @return VW argument object.
     */
    public native VowpalWabbitArguments getArguments();

    public native String getOutputPredictionType();

    public native VowpalWabbitPerformanceStatistics getPerformanceStatistics();

    /**
     * Signals the end of the current pass over the data.
     */
    public native void endPass();

    /**
     * Free's the vw data structure.
     */
    private native void finish();

    /**
     * Invokes the native implementation of Murmur hash. Exposed through
     * VowpalWabbitMurmur.
     */
    static native int hash(byte[] data, int offset, int len, int seed);

    /**
     * Pointer to vw data structure defined in global_data.h
     */
    private long nativePointer;

    /**
     * Initializes the native VW data structures.
     *
     * @param args VW command line arguments.
     */
    public VowpalWabbitNative(String args) {
        this.nativePointer = initialize(args);
    }

    /**
     * Initializes the native VW data structures.
     *
     * <p>
     * Note: The {@code args} must be compatible with the command line arguments
     * stored in {@code model}.
     * </p>
     *
     * @param args  VW command line arguments.
     * @param model VW model to initialize this instance from.
     */
    public VowpalWabbitNative(String args, byte[] model) {
        this.nativePointer = initializeFromModel(args, model);
    }

    private VowpalWabbitNative(long existingWorkspace) {
        this.nativePointer = existingWorkspace;
    }

    /**
     * Creates a new VW example associated with this this instance.
     *
     * @return new {@code VowpalWabbitExample} object.
     */
    public VowpalWabbitExample createExample() {
        return new VowpalWabbitExample(this.nativePointer, false);
    }

    /**
     * Creates a new empty VW example associated with this this instance. This is
     * used to mark the end of a multiline example.
     *
     * @return new {@code VowpalWabbitExample} object.
     */
    public VowpalWabbitExample createEmptyExample() {
        return new VowpalWabbitExample(this.nativePointer, true);
    }

    /**
     * Merge several models together and return the result. Experimental API.
     * @param workspacePointers array of pointers to VW models.
     * @return merged VW model.
     */
    public static native VowpalWabbitNative mergeModels(VowpalWabbitNative baseWorkspace, VowpalWabbitNative[] workspacePointers);

    /**
     * Frees the native resources.
     */
    @Override
    final public void close() {
        if (this.nativePointer != 0) {
            finish();
            this.nativePointer = 0;
        }
    }
}