function xai_evaluate_cumulative_time(resultspaths)
% xai_evaluate_cumulative_time - visualizes the evaluation
%
% Syntax:
%    xai_evaluate_cumulative_time()
%
% Inputs:
%    resultspaths - cellarray of result paths
%
% Outputs:
%    -
%
% Other m-files required: none
% Subfunctions: none
% MAT-files required: none
%
% See also: -

% Authors:       Anonymous
% Written:       27-July-2024
% Last update:   ---
% Last revision: ---

% ------------------------------ BEGIN CODE -------------------------------

if nargin < 1
    resultspaths = {
        './cora_test/xai/results/mnist010sigmoid_fast',
        './cora_test/xai/results/cifar001_100',
        './cora_test/xai/results/gtsrb010fix_50'
    };
end
if ~iscell(resultspaths)
    resultspaths = {resultspaths};
end

f1 = figure; f2 = figure; f3 = figure; 
timeouts = [100,1000,10000];
ofwhichdataset = {'mnist','cifar','gtsrb'};

for j = 1:numel(resultspaths)
    dataset = resultspaths{j};
    timeout = timeouts(cellfun(@(which) contains(dataset, which), ofwhichdataset));
    fprintf('%s ----------------------\n',dataset)
    if isempty(timeout)
        % choose mnist timeout
        timeout = timeouts(1);
        disp('Choosing MNIST timeout..')
    end

    figure(f1)
    subplot(1,numel(resultspaths),j); hold on; box on
    for i=1:11
        try
            load(sprintf('%s/resRun_%i.mat',dataset,i),'resRun')
        catch ME
            disp(ME.message)
            continue
        end
    
        aux_showCumTime(resRun, i, true)
    end

    figure(f2)
    subplot(1,numel(resultspaths),j); hold on; box on
    for i=1:11
        try
            load(sprintf('%s/resRun_%i.mat',dataset,i),'resRun')
        catch ME
            disp(ME.message)
            continue
        end
    
        aux_showCumTime(resRun, i, false)
    end

    figure(f3)
    subplot(1,numel(resultspaths),j); hold on; box on
    for i=1:11
        try
            load(sprintf('%s/resRun_%i.mat',dataset,i),'resRun')
        catch ME
            disp(ME.message)
            continue
        end
        aux_showExplanationSizeVsTime(resRun, i, timeout)
    end

    % remove y-axis for later plots
    if j > 1
        ylabel('')
    end

    drawnow

end

% save figures
savefig(f1,sprintf('%s/showCumTimeIndividual.fig',dataset))
saveas(f1,sprintf('%s/showCumTimeIndividual.png',dataset))
savefig(f2,sprintf('%s/showCumTime.fig',dataset))
saveas(f2,sprintf('%s/showCumTime.png',dataset))
savefig(f3,sprintf('%s/showExplanationSizeVsTime.fig',dataset))
saveas(f3,sprintf('%s/showExplanationSizeVsTime.png',dataset))
close(f1); close(f2); close(f3);

end


% Auxiliary functions -----------------------------------------------------

function aux_showCumTime(resRun, i, showIndividually)
    % show cumulative time

    % read times
    timesPerFeat = resRun.timesPerFeat;

    % filter trivial examples
    timesPerFeat = timesPerFeat(resRun.time > 0);

    % convert to matrix
    timesPerFeat = cell2mat(timesPerFeat');

    % compute cumsum
    if showIndividually
        timesPerFeat_cum = timesPerFeat;
    else
        timesPerFeat_cum = cumsum(timesPerFeat,2);
    end

    % compute mean and std
    m = mean(timesPerFeat_cum,1);
    s = std(timesPerFeat_cum,1,1);

    % plot
    aux_doPlotting(1:numel(m),0,m,s,i,resRun,1);

    % legend
    legend('Location','northwest')
    xlabel('#Processed features')
    ylabel('Time per feature [s]')
    xlim([1,numel(m)])

end

function aux_showExplanationSizeVsTime(resRun, i, timeout)
    % show cumulative time

    % read values
    timesPerFeat = resRun.timesPerFeat;
    idxFreedFeats = resRun.idxFreedFeats;
    featOrders = resRun.featOrders;

    % filter trivial examples
    timesPerFeat = timesPerFeat(resRun.time > 0);
    idxFreedFeats = idxFreedFeats(resRun.time > 0);
    featOrders = featOrders(resRun.time > 0);
    nrFeats = numel(featOrders{1});

    % compute time ---

    % convert to matrix
    timesPerFeat = cell2mat(timesPerFeat');

    % compute cumsum
    timesPerFeat_cum = cumsum(timesPerFeat,2);
    % timesPerFeat_cum = timesPerFeat; % or show individually

    % compute mean and std
    times_m = mean(timesPerFeat_cum,1);
    times_s = std(timesPerFeat_cum,1,1);

    % compute explanation size ---

    size_all = zeros(numel(idxFreedFeats),numel(times_m));
    for j = 1:numel(idxFreedFeats)
        size_all(j,:) = cumsum(sum( ...
            idxFreedFeats{j}' == featOrders{j}' ...
        ));
    end

    % compute mean and std
    size_m = nrFeats-mean(size_all,1);
    size_s = std(size_all,1,1);

    % plot
    name = aux_doPlotting(times_m,times_s,size_m,size_s,i,resRun,-1);

    % legend
    legend('Location','northwest')
    ylabel('Explanation size')
    xlabel('Cumulative time [s]')
    ylim([1,numel(times_m)])



    % compute explanation size w.r.t. timeout ---

    size_all_timeout = zeros(numel(idxFreedFeats),numel(times_m));
    for j = 1:numel(idxFreedFeats)
        size_all_timeout(j,:) = cumsum(sum( ...
            idxFreedFeats{j}' == featOrders{j}' ...
            & timesPerFeat_cum(j,:) <= timeout    ...
        ));
    end

    % compute mean and std
    size_m_timeout = nrFeats-mean(size_all_timeout,1);
    size_s_timeout = std(size_all_timeout,1,1);

    fprintf('%-15s \t & $%.2f \\pm %6.2f$ \\\\\n', name, size_m_timeout(end), size_s_timeout(end))

end

function name = aux_doPlotting(x,x_s,y,y_s,i,resRun,inv)
    % plot
    updateColorIndex(i-1);
    color = CORAcolor('CORA:next');

    lineStyle = '-';
    alpha = 0.1;
    if i == 1
        name = 'abstract+refine';
    elseif i == 2
        name = 'standard';
    else
        name = ['abstract' num2str(resRun.config{7})];
        alpha = 0.05;
        lineStyle = '--';
    end

    warning off
    poly = polygon([x+x_s x(end)+x_s(end) x(end:-1:1)-x_s(end:-1:1)],[y-inv*y_s y(end)+inv*y_s(end) y(end:-1:1)+inv*y_s(end:-1:1)]);
    warning on

    V = poly.set.Vertices;
    fill(V(:,1),V(:,2),color,'FaceAlpha',alpha,'LineStyle','none','HandleVisibility','off')
    plot(x, y,lineStyle, 'Color',color,'DisplayName',name);

end

% ------------------------------ END OF CODE ------------------------------
