import xarray as xr
import glob
import numpy as np
import os
from paths_config import paths_config

def get_nc_filenames(dataset_folder, permutation=None):
    nc_filenames = sorted(glob.glob(dataset_folder + '/*.nc'))
    if len(nc_filenames) != 10_000:
        print(f"Can't find 10k nc files at {dataset_folder}")
        exit()
    for index, filename in enumerate(nc_filenames):
        ending = f"{index:06d}.nc"
        if not filename.endswith(ending):
            raise Exception(f"Images not consecutive, found {filename} while excepting it to end with {ending}")
    if permutation is not None:
        print("Permuting the file name list")
        permuted_nc_filenames = []
        for index in range(len(nc_filenames)):
            permuted_nc_filenames.append(nc_filenames[permutation[index]])
        return permuted_nc_filenames
    else:
        return nc_filenames

def save_npy_dataset(nc_filenames, start_index, end_index, npy_filename, permutation=None, spatial_shape=(512, 512)):
    print(f"Reading nc files {start_index:06d} to {(end_index - 1):06d}")
    np_dataset = np.zeros((end_index - start_index, 3, spatial_shape[0], spatial_shape[1]), dtype=np.float32)
    for index, filename in enumerate(nc_filenames[start_index:end_index]):
        print(f"\r{filename}", end="")
        nc_data = xr.open_dataset(filename)
        # Crop at the corner for compatibility with existing training runs. It would be better to center crop.
        r = nc_data['B04'][0].as_numpy()[:spatial_shape[0], :spatial_shape[1]]*2/10000 - 1
        g = nc_data['B03'][0].as_numpy()[:spatial_shape[0], :spatial_shape[1]]*2/10000 - 1
        b = nc_data['B02'][0].as_numpy()[:spatial_shape[0], :spatial_shape[1]]*2/10000 - 1
        np_dataset[index] = np.stack([r, g, b], axis=0)
    print()
    print(f"Writing npy file {npy_filename}")
    with open(npy_filename, 'wb') as f:
        np.save(f, np_dataset)

# We randomly shuffled the 10k dataset which is actually unnecessary but may need to be done to reproduce results of the paper. Sorry about that. Here is the permutation we used.
permutation = [3368, 5327, 9215, 3064, 5996, 7350, 326, 5977, 7337, 1613, 5306, 1880, 4443, 1961, 8144, 1007, 4823, 1341, 4157, 5922, 3846, 7561, 1158, 7716, 7418, 9269, 3212, 7942, 144, 4106, 6691, 9794, 2462, 668, 8173, 9342, 6811, 9015, 4522, 1094, 9467, 2310, 2387, 3377, 4226, 354, 492, 4347, 5710, 3857, 7208, 8659, 1675, 5553, 7511, 6189, 525, 8658, 3316, 4107, 6934, 747, 3079, 779, 184, 7821, 2788, 1733, 4626, 506, 940, 3674, 3044, 3681, 8943, 9240, 8634, 9303, 7303, 2398, 6199, 688, 9741, 3950, 5995, 8752, 5336, 221, 3298, 5810, 4889, 5626, 8204, 7126, 9425, 4707, 7909, 7122, 8230, 5720, 4323, 2953, 4994, 5634, 8281, 6772, 9461, 9141, 322, 8082, 32, 72, 6852, 8766, 2521, 4485, 7964, 3569, 109, 1004, 4409, 8392, 9505, 7323, 7371, 2623, 1426, 1994, 2877, 1040, 1754, 5180, 2884, 9145, 1718, 4483, 438, 7324, 4833, 5784, 8250, 4007, 7465, 7718, 4940, 4276, 6134, 2002, 685, 6445, 7011, 9299, 8544, 2441, 9489, 6174, 9783, 5639, 6801, 7328, 1611, 6166, 2279, 863, 4246, 4077, 9158, 7727, 2678, 7990, 150, 8491, 7534, 5757, 4280, 8969, 3356, 116, 7492, 5635, 2873, 9636, 3825, 3778, 8964, 9290, 557, 7519, 6703, 5407, 1549, 1815, 2261, 1141, 7630, 9304, 7741, 7268, 4844, 762, 3480, 6548, 550, 1013, 3805, 1145, 8941, 1177, 8550, 4608, 2156, 8957, 8567, 7702, 5191, 3748, 2931, 7695, 9282, 7250, 7347, 9068, 4207, 3515, 3576, 9775, 5107, 5895, 9306, 6754, 7777, 513, 4632, 2099, 2750, 5298, 6818, 555, 6073, 6494, 520, 6066, 924, 5600, 7355, 8516, 3390, 4037, 1940, 381, 7899, 3757, 2921, 2878, 6653, 9128, 639, 5852, 7621, 3427, 4703, 7675, 3620, 4919, 9163, 1046, 1921, 6243, 3476, 9651, 2209, 6416, 3525, 9704, 4350, 3623, 2133, 5013, 333, 1607, 1318, 7100, 428, 171, 9179, 2726, 4578, 1010, 4601, 7603, 4317, 562, 269, 7333, 1372, 6696, 9437, 6249, 7001, 987, 5926, 2610, 7481, 7595, 4627, 8260, 8638, 5886, 3457, 6117, 8791, 7706, 6082, 2598, 7006, 4327, 5416, 3512, 6928, 2238, 9168, 3795, 7880, 5244, 8050, 5938, 2714, 8463, 8718, 7070, 502, 6409, 6071, 7177, 1816, 79, 8623, 1564, 2893, 3341, 6035, 9736, 6057, 9561, 2203, 2211, 2692, 4180, 6509, 4570, 9388, 2582, 3125, 1986, 7171, 7138, 5588, 7589, 6878, 7747, 7142, 9751, 4124, 80, 7838, 9059, 8585, 9664, 819, 1314, 7591, 3075, 7545, 6528, 6958, 7786, 292, 9214, 2021, 85, 1517, 4000, 2538, 38, 771, 5704, 5924, 4817, 4966, 6794, 2139, 1525, 8363, 93, 989, 6209, 8471, 3001, 9873, 53, 4796, 8771, 7656, 8171, 4571, 2803, 9232, 579, 2437, 7327, 7052, 104, 43, 3545, 4030, 9572, 7442, 287, 2828, 4795, 9942, 10, 9954, 7237, 6838, 2122, 7373, 3922, 1973, 1181, 3106, 1239, 5186, 4308, 332, 9486, 5420, 8419, 3015, 82, 4686, 5112, 9587, 4166, 8869, 5915, 8611, 9515, 3528, 6946, 487, 8415, 8552, 5021, 8185, 3555, 2943, 5276, 1419, 4098, 7403, 526, 9792, 2179, 5558, 7345, 620, 973, 1692, 9717, 9973, 9219, 5963, 4025, 4554, 444, 1411, 3393, 3903, 1463, 4839, 2226, 489, 279, 5249, 3584, 9468, 1230, 1255, 5410, 3967, 467, 4691, 3217, 4153, 8093, 9160, 4496, 6990, 3012, 7218, 3527, 941, 1156, 8301, 2391, 8414, 793, 1992, 8767, 7930, 3890, 3124, 4122, 9978, 7598, 4747, 2357, 1568, 51, 4447, 3181, 177, 4420, 3007, 4936, 7840, 6617, 9966, 410, 8713, 6268, 2269, 8309, 5411, 5423, 2452, 453, 2753, 6112, 8006, 7760, 5931, 4197, 6006, 768, 2646, 2817, 3369, 4075, 8630, 5584, 3487, 3510, 7857, 9509, 2796, 6366, 9506, 2087, 6168, 1061, 7812, 8867, 4876, 9566, 5285, 7130, 1978, 7068, 3513, 7062, 298, 1782, 2701, 5782, 5325, 7185, 7508, 5330, 5422, 3160, 7787, 5591, 2115, 7894, 9858, 9454, 7758, 612, 5552, 7156, 6809, 2783, 5565, 2729, 4521, 1772, 2086, 7649, 4866, 6863, 9216, 8830, 6633, 8488, 5685, 4369, 9905, 1923, 4110, 3566, 6544, 7479, 210, 8747, 1839, 1508, 3999, 2913, 7753, 2142, 7563, 7583, 8514, 3834, 8460, 1685, 8262, 8504, 5043, 2378, 7445, 874, 5130, 1503, 8699, 5254, 1630, 8845, 4720, 2869, 6310, 1790, 7946, 4004, 2948, 3014, 5251, 7322, 1043, 4333, 2544, 5674, 1278, 1111, 1035, 6960, 7040, 4576, 5147, 7690, 6956, 7395, 4778, 9431, 5207, 380, 6222, 3506, 6585, 6712, 9357, 4813, 9972, 6449, 6622, 3279, 7032, 2849, 1752, 5898, 7767, 3861, 7422, 8909, 665, 8330, 559, 7257, 1833, 4055, 6410, 4933, 4849, 9089, 1444, 56, 7902, 404, 2740, 7393, 5019, 2601, 7662, 8283, 3040, 2613, 7979, 9528, 943, 8561, 2227, 3501, 6203, 1073, 8332, 2400, 2401, 6327, 4016, 6892, 5859, 6513, 6289, 7057, 7194, 4821, 1663, 6420, 4928, 864, 8502, 4212, 6025, 9974, 9201, 5894, 7064, 1528, 4269, 3696, 4552, 8210, 9764, 5092, 4450, 6417, 4335, 5630, 5381, 4836, 675, 4263, 4536, 937, 2367, 2988, 6170, 4229, 8979, 5513, 7700, 8123, 4825, 1384, 9708, 2361, 8998, 164, 8286, 1038, 4996, 8741, 8590, 3463, 5355, 503, 5619, 9863, 3346, 9369, 2996, 4457, 6615, 1100, 2221, 1539, 4978, 5409, 1086, 1578, 4755, 5967, 2525, 4497, 7699, 7165, 5322, 5023, 6013, 6616, 3052, 3190, 1922, 8467, 1345, 2200, 4888, 2711, 7217, 6670, 2954, 9570, 2937, 8578, 9638, 4251, 4258, 3636, 293, 8348, 1791, 4816, 2815, 8005, 7628, 7302, 1553, 3276, 5273, 4065, 3653, 9996, 6131, 2356, 8966, 2720, 3013, 6800, 8391, 2289, 7965, 5612, 708, 1502, 14, 4858, 2288, 2580, 3287, 7622, 2653, 3456, 3323, 4296, 5457, 7359, 2663, 2329, 2902, 9909, 4509, 2366, 8851, 9106, 7366, 3583, 5733, 8930, 3053, 4890, 7392, 4895, 7438, 3864, 9842, 7994, 6389, 3233, 2872, 9956, 4289, 9432, 8759, 9262, 8725, 8454, 1805, 6004, 6432, 5210, 3675, 8267, 8017, 9283, 1936, 609, 3666, 2829, 8508, 9628, 313, 2858, 2162, 7390, 4136, 4708, 2513, 7224, 7474, 1750, 1146, 633, 968, 1274, 3304, 29, 7643, 6568, 4224, 2464, 3221, 9615, 6537, 3990, 7932, 8946, 6797, 5845, 2672, 9733, 9521, 4583, 259, 4679, 2924, 5049, 6137, 1, 6635, 6351, 8152, 1458, 4010, 8148, 4696, 5548, 3866, 6916, 3268, 2043, 9965, 319, 5860, 5608, 9611, 2459, 8961, 482, 7262, 2453, 3462, 4804, 8688, 9740, 4771, 3436, 5960, 3640, 1915, 3739, 3925, 3691, 4368, 9027, 9917, 4492, 8456, 1388, 9903, 1999, 1210, 1063, 8885, 9527, 2867, 2144, 5171, 4167, 8239, 1213, 9821, 7652, 9359, 7669, 3282, 9967, 5018, 1427, 7383, 225, 8799, 6369, 8292, 1134, 7310, 6188, 4915, 9893, 300, 5497, 8435, 5799, 6375, 9415, 4721, 9238, 6153, 6079, 2945, 3294, 7901, 2093, 3228, 3969, 1799, 2504, 6344, 8220, 4882, 4311, 6572, 1788, 9560, 1538, 5326, 1407, 7671, 497, 6937, 6549, 6987, 4525, 1728, 5888, 8053, 7087, 1612, 203, 2309, 5765, 6775, 2318, 7139, 7235, 9961, 7588, 7506, 4084, 5816, 1695, 538, 6903, 4344, 6284, 8779, 7478, 8436, 8108, 4127, 7849, 8738, 3988, 4465, 1724, 4681, 6550, 1381, 5072, 3372, 7167, 6656, 8709, 7103, 4990, 8968, 826, 9894, 4050, 2411, 3785, 780, 2702, 7687, 2941, 4284, 6527, 1970, 2456, 4126, 6292, 997, 7026, 1580, 4279, 8055, 4894, 2864, 1872, 9280, 9810, 4631, 1206, 3761, 1439, 2915, 9324, 4256, 2146, 1912, 533, 2150, 969, 5739, 1257, 4772, 7058, 1165, 1344, 3740, 1895, 3872, 1506, 9963, 8768, 6569, 5240, 340, 2899, 8717, 9698, 7771, 9937, 7689, 226, 95, 7369, 3625, 5564, 1031, 9194, 8455, 3492, 8374, 9777, 1997, 5414, 6480, 4807, 3714, 1977, 4797, 4803, 8780, 4614, 6570, 3162, 3087, 9484, 8145, 851, 9655, 8387, 5033, 5681, 4438, 8732, 3889, 7444, 5512, 6879, 2568, 2036, 323, 1015, 8257, 543, 1712, 3312, 2712, 4406, 1554, 2819, 5063, 9953, 6173, 1233, 4239, 1745, 8842, 3716, 6700, 5319, 9494, 4419, 8962, 798, 4407, 556, 9460, 2451, 7396, 6285, 2231, 5547, 8835, 800, 3737, 9721, 5238, 7824, 9662, 8865, 2578, 4205, 2797, 6297, 4852, 288, 8051, 8840, 3742, 5842, 3347, 551, 6795, 7701, 9812, 1243, 1059, 7244, 7897, 1079, 9687, 6891, 9912, 3632, 9174, 8007, 8380, 4943, 7724, 8735, 1617, 278, 731, 3531, 5749, 8383, 7411, 4643, 9193, 5785, 4743, 4512, 9826, 4192, 6365, 6556, 2071, 6341, 141, 2265, 2109, 5864, 374, 9213, 9682, 2890, 813, 9183, 9267, 2688, 6046, 1359, 3392, 788, 2311, 1480, 3749, 1307, 2739, 3058, 537, 2442, 1385, 1106, 5455, 4981, 4083, 8606, 5606, 4386, 8540, 430, 5037, 1219, 3250, 433, 9420, 290, 3601, 8351, 6321, 6890, 1192, 8446, 3147, 4597, 6061, 4502, 3194, 1707, 4544, 3023, 7811, 6969, 7256, 1767, 8605, 459, 5485, 4640, 3061, 7175, 757, 6177, 6994, 222, 9404, 5774, 5218, 6651, 2665, 843, 3434, 1792, 4003, 6201, 9929, 8234, 2434, 9173, 6191, 3874, 8003, 2371, 571, 5743, 9241, 7858, 4811, 139, 7828, 2604, 9626, 9948, 215, 4072, 4575, 3499, 8226, 9381, 4581, 192, 1864, 4774, 9230, 6746, 8457, 5292, 1286, 9235, 4798, 7779, 2399, 7524, 8359, 6853, 5817, 3235, 925, 7249, 5773, 6116, 2808, 1514, 5598, 992, 5798, 1136, 2344, 850, 5398, 2406, 6278, 8922, 4615, 8320, 4261, 1786, 9339, 5936, 9603, 9959, 7601, 9176, 4294, 5313, 211, 2724, 8534, 5506, 4841, 8532, 640, 4562, 1766, 3826, 367, 9637, 2974, 5573, 5711, 7593, 8025, 4765, 3937, 2760, 3032, 9080, 6835, 2167, 6440, 547, 4392, 7647, 6055, 9197, 9325, 1023, 5034, 979, 7659, 9470, 6561, 9003, 5368, 8303, 8529, 3271, 6869, 611, 7769, 3660, 5098, 5814, 1373, 8645, 6016, 4634, 6694, 9228, 2424, 9298, 5863, 623, 1461, 490, 5542, 7562, 73, 7219, 7425, 3540, 2198, 3960, 6029, 7820, 9555, 6161, 2709, 1885, 6919, 4923, 5979, 8810, 4453, 9892, 8525, 9630, 5468, 1910, 9920, 5156, 3869, 7294, 9574, 6672, 857, 4893, 3240, 773, 4594, 7861, 1834, 2994, 6973, 7962, 1647, 1051, 9600, 1114, 6360, 6812, 9576, 1101, 7856, 8891, 3994, 1320, 512, 8711, 8188, 962, 4985, 6562, 2426, 3211, 4656, 9120, 7735, 5525, 1399, 5809, 1719, 7788, 8712, 8084, 7891, 6483, 3039, 8042, 6901, 8826, 812, 674, 6436, 2054, 390, 260, 9743, 1055, 4116, 2535, 876, 8494, 7367, 4412, 9205, 8619, 5515, 3905, 617, 6325, 1078, 8587, 6097, 7406, 1271, 9169, 8201, 2186, 1293, 2567, 5607, 1662, 7945, 8703, 3913, 552, 4071, 6917, 2440, 3923, 4956, 2290, 8468, 3428, 5601, 6108, 9439, 7615, 244, 560, 7538, 8307, 2065, 6098, 5916, 5518, 5166, 2029, 5837, 4244, 8620, 9508, 2041, 5421, 1637, 2845, 5812, 5628, 9469, 9318, 9579, 341, 7830, 8749, 8663, 4137, 1306, 804, 7042, 2674, 6477, 6710, 3481, 9727, 8940, 8079, 3403, 8608, 4099, 2244, 7807, 2624, 2471, 9633, 2077, 6953, 7420, 8279, 7287, 4745, 2703, 8973, 9429, 5955, 5090, 3331, 7996, 6722, 1415, 5504, 1352, 7756, 7907, 5402, 7967, 8213, 6425, 8660, 2560, 5998, 4702, 1859, 9590, 7449, 9166, 3167, 7818, 8439, 1490, 5646, 3183, 3485, 5142, 1914, 3887, 8110, 5204, 6342, 1819, 8325, 7118, 2981, 9476, 549, 3076, 2010, 9979, 7197, 6856, 1459, 6064, 5260, 213, 9416, 4574, 9310, 730, 4290, 1382, 1627, 3920, 4359, 897, 8342, 6984, 8035, 7957, 5078, 3026, 2547, 8297, 3116, 9705, 4162, 6725, 4507, 9715, 125, 7416, 8498, 7808, 2968, 4970, 5521, 9669, 870, 4395, 3689, 3126, 5197, 5024, 7128, 1893, 6335, 9700, 7131, 938, 6119, 8326, 6007, 7386, 2689, 5627, 5087, 5183, 4452, 671, 3850, 7059, 3936, 9231, 3702, 1290, 3710, 7833, 5786, 8205, 2201, 4089, 3690, 5620, 3596, 8319, 5126, 5367, 7184, 5593, 9004, 1905, 1743, 3043, 1298, 6138, 3602, 4320, 6012, 4134, 1548, 2484, 8806, 4588, 8240, 2175, 2527, 8918, 2887, 383, 1090, 1358, 2117, 8952, 6140, 5897, 9899, 19, 8503, 7626, 2503, 6367, 9523, 8547, 4555, 7147, 1292, 2206, 7693, 4953, 7974, 8111, 268, 4217, 872, 8785, 5463, 5696, 9125, 6573, 6068, 6519, 6099, 5870, 9885, 6286, 878, 8696, 8242, 2885, 1445, 2393, 1183, 4168, 4336, 3817, 4, 1854, 8948, 7304, 7434, 2669, 57, 7280, 7376, 4002, 9418, 9190, 2427, 6645, 7736, 4459, 31, 3539, 699, 9739, 6413, 8295, 4396, 7531, 5106, 8797, 597, 8255, 1571, 8344, 6028, 1147, 2684, 1234, 1565, 7336, 9171, 2984, 4859, 9189, 5948, 9152, 1690, 8199, 3796, 6457, 977, 7629, 6093, 3577, 6674, 6742, 1105, 286, 9537, 1806, 2912, 7596, 1308, 1208, 3781, 5687, 8604, 3011, 8399, 6329, 3265, 8854, 3222, 9042, 2268, 2585, 3894, 7203, 1689, 7233, 2306, 6643, 8170, 8078, 348, 8052, 1155, 3465, 8607, 5604, 3654, 8306, 1823, 4480, 2212, 5229, 7636, 4678, 6855, 1476, 4959, 2169, 8977, 3227, 122, 6598, 3550, 5910, 1356, 4423, 1489, 1534, 4729, 8305, 1485, 5170, 569, 9281, 256, 5684, 3197, 2717, 9246, 8345, 8632, 1402, 2460, 648, 2247, 7772, 8482, 8126, 9619, 9881, 4371, 4393, 7318, 435, 867, 6333, 8324, 9389, 4498, 7253, 2180, 8222, 3526, 275, 6824, 1261, 2621, 3637, 3034, 2852, 8939, 4805, 9419, 8296, 1285, 4995, 6976, 7918, 4949, 1962, 3647, 3891, 1669, 1204, 3460, 2818, 7553, 2364, 8106, 5340, 5042, 248, 7782, 2067, 7528, 9944, 463, 1333, 624, 3054, 1178, 2305, 3605, 3198, 1943, 1179, 7968, 3731, 6221, 7041, 7633, 4194, 6414, 8524, 8594, 5269, 2487, 4561, 1138, 2281, 6175, 5522, 1755, 3738, 2105, 7594, 4912, 1218, 2101, 3751, 5341, 3454, 7704, 828, 5811, 9524, 4682, 8014, 2694, 3759, 8886, 8453, 5781, 5190, 643, 2520, 11, 8554, 5128, 5077, 4854, 1316, 8331, 4009, 1391, 3447, 3886, 5447, 7417, 37, 115, 1383, 6128, 629, 7115, 9308, 4926, 2477, 3455, 9567, 4865, 2826, 7446, 7432, 5787, 230, 1505, 182, 2336, 9053, 4715, 146, 4590, 6263, 2283, 3884, 9964, 7765, 832, 9758, 4228, 8917, 6758, 8461, 4733, 4639, 2965, 6941, 3149, 8912, 8097, 2617, 7639, 5378, 9046, 2514, 5331, 818, 8038, 8649, 2333, 9706, 4547, 5056, 1033, 3939, 247, 6720, 3484, 9104, 8870, 9121, 2129, 902, 1299, 8827, 9902, 3943, 8579, 3443, 3970, 8960, 9716, 3961, 3673, 2998, 131, 8739, 4011, 6298, 1369, 2295, 9691, 4411, 4260, 2078, 2013, 3902, 8183, 7031, 6246, 5246, 6307, 9675, 885, 9679, 6282, 4227, 1858, 5161, 6078, 7154, 3818, 4249, 9605, 4855, 7035, 2952, 1731, 6580, 8919, 8764, 7572, 7676, 7491, 8098, 5274, 5150, 6358, 6912, 8180, 6401, 1798, 2191, 1189, 5384, 6143, 8402, 9380, 4775, 2919, 4989, 4557, 8396, 6759, 2422, 9838, 5868, 6306, 1295, 6192, 8911, 3790, 3420, 400, 4503, 8282, 7866, 9755, 8156, 264, 486, 4677, 8522, 5290, 9906, 4734, 7003, 8656, 8657, 7321, 2804, 1740, 2125, 982, 5516, 4761, 4809, 8624, 2420, 5950, 2353, 855, 1983, 7200, 5609, 5941, 2112, 9757, 568, 8621, 507, 7697, 3299, 258, 7634, 9922, 4200, 7472, 3342, 4388, 2735, 716, 5475, 2966, 6214, 766, 1590, 6125, 9063, 1678, 9666, 2944, 3489, 2522, 2478, 2989, 1124, 1626, 9320, 8714, 7293, 3300, 8071, 9998, 5100, 548, 8680, 9114, 7238, 39, 6804, 9102, 8586, 422, 8833, 7890, 3329, 9761, 9635, 1974, 9654, 5310, 5035, 1830, 4937, 1860, 7209, 2085, 5873, 1844, 6270, 3469, 8400, 6404, 5721, 8846, 2199, 2554, 9534, 2224, 5794, 509, 3782, 119, 2886, 6731, 4603, 9936, 74, 4837, 3435, 8549, 7082, 3987, 5261, 3328, 6305, 1501, 5081, 2235, 4287, 3098, 5616, 7298, 4591, 4193, 6070, 6274, 8027, 6899, 7803, 922, 2335, 4041, 6102, 2840, 9538, 8091, 9083, 5275, 9960, 2792, 7904, 9326, 5075, 682, 4899, 3371, 6893, 8135, 4916, 7953, 6790, 3552, 5824, 5758, 1336, 1679, 5267, 4950, 9614, 5707, 5095, 9338, 5277, 1452, 2928, 2631, 6845, 7844, 2210, 1614, 915, 4424, 5969, 6008, 194, 2291, 8893, 3621, 5974, 2556, 1784, 1519, 454, 2737, 6875, 2837, 5450, 5484, 4698, 8569, 6363, 6182, 9680, 8730, 5622, 2057, 2166, 9730, 6005, 5000, 5846, 3906, 3655, 2640, 5649, 2325, 2949, 5473, 4606, 3771, 1956, 7522, 1467, 5848, 6040, 6896, 8802, 2835, 8231, 4587, 9039, 5879, 4383, 2257, 3671, 3306, 3772, 7958, 7330, 5432, 8667, 4330, 7748, 5532, 5813, 1129, 4585, 4822, 6088, 6234, 3413, 4542, 976, 7370, 5499, 7928, 8914, 1324, 9293, 7648, 7429, 5256, 7794, 4909, 1594, 4907, 9500, 9132, 1442, 8310, 1455, 4416, 3200, 9622, 3992, 1151, 9631, 8166, 2749, 3226, 4468, 3708, 3775, 6050, 9167, 2352, 5446, 3682, 5406, 758, 1958, 6139, 4431, 5250, 3581, 6842, 3070, 3020, 6676, 6493, 6208, 1638, 6628, 4655, 4920, 81, 5372, 6791, 5048, 8706, 3340, 5791, 7991, 6787, 2800, 5328, 7832, 4305, 3357, 8473, 1747, 8323, 3409, 930, 6354, 266, 9640, 6431, 64, 1456, 8819, 2557, 2512, 1451, 5543, 8614, 2561, 4641, 1777, 1946, 60, 5291, 6200, 3568, 9752, 1668, 6806, 499, 5647, 8497, 8887, 1587, 1693, 702, 7275, 4651, 9613, 8413, 7517, 3823, 6359, 6743, 9837, 1778, 3089, 1918, 5053, 2012, 3317, 1045, 4066, 9098, 4980, 1894, 8695, 142, 1802, 2194, 3791, 5111, 6356, 7221, 6927, 4489, 9791, 2586, 41, 3594, 392, 5975, 4111, 2039, 170, 6424, 8736, 4753, 1284, 9841, 8892, 1748, 2733, 4695, 4358, 5544, 429, 6224, 4175, 5875, 6709, 1949, 5891, 7440, 356, 7809, 7665, 5253, 92, 1866, 6646, 981, 4417, 299, 7573, 3715, 8074, 8669, 1074, 8772, 5747, 2317, 4036, 4357, 2725, 3247, 7568, 3616, 6184, 9340, 7448, 616, 6512, 948, 4835, 9533, 4440, 2506, 398, 2092, 6827, 6338, 9787, 2563, 8507, 7792, 4529, 583, 2188, 6387, 6837, 7740, 4963, 8926, 4379, 2507, 7389, 8443, 9513, 8385, 1556, 6472, 5272, 8493, 6737, 1812, 1187, 3973, 9485, 8982, 522, 2934, 8143, 9559, 1843, 3483, 918, 1337, 1203, 4454, 6750, 3071, 3471, 5175, 811, 4719, 2159, 9277, 7920, 5014, 1447, 6638, 5469, 208, 3334, 9060, 2656, 9769, 4659, 3952, 2060, 5680, 4414, 3395, 2059, 7884, 626, 7415, 4794, 6447, 2082, 9433, 1343, 6567, 1683, 9800, 6423, 1122, 9131, 1417, 8334, 7853, 6383, 6630, 352, 2960, 3283, 4473, 7300, 2319, 309, 4282, 2098, 369, 3129, 8248, 9957, 3384, 1315, 9225, 3542, 1126, 3634, 5951, 9907, 112, 8000, 8744, 4231, 3619, 9504, 1311, 862, 8564, 5719, 5569, 5345, 2634, 1185, 2005, 2381, 8059, 3786, 837, 9123, 9713, 8584, 9807, 3672, 2360, 6685, 4068, 6603, 3878, 6111, 1066, 5728, 5227, 8631, 4845, 1069, 448, 1575, 9270, 1917, 9866, 6931, 3394, 1996, 7922, 5114, 7136, 89, 8381, 1773, 9623, 1935, 7579, 8521, 791, 2622, 7034, 7732, 9827, 5746, 756, 7785, 6895, 2992, 7916, 3383, 2373, 5120, 839, 9599, 4908, 2759, 6654, 3466, 7997, 1492, 8328, 5459, 7667, 2591, 5507, 54, 173, 5443, 4869, 6313, 3511, 2594, 2799, 4455, 5152, 7908, 6309, 8103, 8284, 3593, 4750, 483, 6798, 1605, 3196, 9256, 844, 1616, 803, 8513, 6096, 9130, 7091, 4434, 9309, 1976, 1184, 9649, 9393, 1070, 1919, 9918, 539, 5346, 5239, 6232, 5008, 4449, 2680, 9772, 3541, 3429, 3244, 7356, 8517, 4059, 2034, 4199, 25, 8934, 637, 6600, 3962, 6981, 9828, 5268, 3301, 8880, 8034, 6779, 1650, 7335, 6736, 8956, 3572, 971, 7013, 2925, 4504, 6379, 2414, 1729, 1939, 9818, 9157, 1354, 8480, 1558, 2628, 3991, 2871, 6564, 6805, 5567, 7214, 4372, 7799, 9458, 6319, 3363, 6900, 4169, 285, 6179, 7145, 6109, 1701, 7988, 2444, 3741, 7507, 8868, 4680, 6977, 8754, 4573, 778, 8451, 5713, 2388, 4806, 7755, 3670, 3419, 5590, 576, 2286, 1874, 8233, 4830, 7725, 6897, 3335, 6377, 9450, 3000, 9273, 9209, 3362, 9008, 3185, 5847, 8394, 5262, 9086, 6019, 5519, 9543, 9355, 7705, 6663, 5284, 8163, 9875, 4425, 3600, 1546, 4140, 8189, 311, 8825, 6596, 8447, 6971, 5984, 6880, 9718, 7299, 1282, 8844, 3706, 1304, 3234, 9742, 2898, 9394, 630, 1573, 9943, 3186, 2219, 4746, 554, 9865, 5561, 2492, 2368, 1414, 5010, 8431, 6520, 7992, 681, 1757, 5511, 9969, 1404, 6063, 8237, 6517, 7471, 5778, 3547, 5595, 234, 8445, 30, 1891, 4820, 4109, 932, 4979, 4151, 3152, 9971, 4633, 2485, 5145, 2197, 4914, 3156, 9239, 148, 5115, 4662, 9263, 3645, 8370, 7457, 9363, 2207, 7405, 7360, 6996, 3169, 1619, 1457, 5016, 8109, 8197, 5823, 4270, 907, 1113, 8280, 869, 4951, 621, 628, 4316, 6602, 9159, 6190, 3450, 1216, 9799, 2276, 4944, 9092, 9296, 3107, 7746, 228, 2297, 1117, 5028, 6481, 5101, 9673, 4791, 7911, 1759, 4165, 5533, 7784, 4559, 3679, 5965, 7980, 6444, 8011, 4793, 3965, 523, 6926, 6281, 6762, 2822, 4259, 2375, 6136, 899, 1164, 8580, 1572, 2066, 3575, 3591, 1225, 3077, 6460, 9477, 5117, 4233, 7635, 6817, 6965, 5849, 8881, 9481, 944, 8013, 7452, 6252, 1793, 3648, 9365, 9609, 6302, 2362, 233, 1869, 4665, 3614, 8705, 6870, 903, 3400, 9253, 4904, 2986, 3808, 3337, 3373, 9124, 5022, 8821, 8721, 9859, 7349, 7149, 3425, 8515, 711, 5026, 6080, 8440, 4303, 8045, 3856, 9540, 978, 207, 5492, 5669, 6560, 8140, 6538, 7462, 351, 7020, 2104, 439, 445, 3324, 9411, 9087, 9941, 4706, 4653, 1188, 8133, 4061, 2347, 3229, 8128, 2240, 9816, 3114, 1993, 9122, 8390, 5263, 8807, 5155, 1223, 8652, 3078, 8997, 3927, 6866, 592, 9999, 7614, 9855, 7090, 5437, 7775, 9199, 7272, 7810, 2682, 7729, 7566, 9354, 306, 5880, 7540, 23, 2028, 1363, 4462, 3641, 2855, 831, 2920, 1889, 3972, 9556, 8043, 9435, 3986, 253, 6002, 9172, 4701, 5538, 9997, 1711, 3251, 6105, 4705, 5348, 6235, 2395, 7770, 3863, 706, 6533, 9443, 317, 4735, 2576, 1810, 4986, 4964, 6154, 377, 2571, 7461, 8931, 1581, 9488, 3261, 305, 580, 1095, 7215, 183, 7398, 7867, 1482, 7162, 6371, 1110, 4638, 2252, 3242, 6349, 3104, 8016, 6318, 635, 6648, 3860, 6909, 767, 5305, 7080, 6948, 7829, 2120, 846, 6707, 2386, 6042, 1054, 9910, 2748, 7835, 451, 126, 4095, 9351, 6352, 5970, 967, 5902, 8191, 8146, 7325, 8602, 2195, 3223, 8809, 1148, 1769, 409, 5215, 2326, 4736, 1667, 4789, 2131, 5093, 3195, 2569, 8701, 4446, 9337, 1425, 3589, 8700, 2879, 4766, 3241, 7459, 9482, 9447, 3246, 8783, 8492, 825, 9684, 9279, 7354, 8506, 9358, 2983, 336, 9749, 4738, 528, 4757, 3008, 4535, 7029, 8972, 1717, 353, 9734, 283, 5020, 99, 2030, 9946, 5921, 6496, 7114, 7541, 1661, 995, 4237, 7428, 4176, 9483, 9426, 8258, 4965, 615, 1687, 1826, 8357, 3117, 1642, 1137, 8338, 5136, 2135, 8464, 440, 1026, 5760, 6557, 3134, 1881, 8563, 4171, 2110, 5877, 5918, 4749, 7860, 8317, 1174, 3338, 1947, 3475, 310, 7284, 4257, 1761, 4808, 4252, 3875, 8557, 3561, 5659, 2218, 1902, 4991, 4693, 1393, 573, 6288, 1730, 6245, 8950, 6582, 3047, 6850, 5036, 2746, 417, 701, 9707, 7085, 3474, 1990, 237, 3580, 3873, 9181, 4881, 4611, 2542, 1289, 7168, 1132, 2369, 947, 3137, 6995, 6241, 9886, 1406, 1524, 1477, 1585, 6269, 7646, 3880, 3724, 1401, 5727, 3353, 3563, 2606, 2346, 8877, 8086, 9517, 327, 8356, 1398, 1030, 7404, 8066, 9696, 473, 4642, 5714, 5029, 1071, 4306, 2324, 385, 796, 6009, 246, 4969, 4911, 5648, 2549, 2234, 5480, 69, 9593, 181, 3556, 3520, 8137, 2600, 8099, 8915, 329, 5932, 3929, 5946, 5572, 9814, 4674, 4020, 3458, 1379, 5116, 6181, 3189, 6395, 4331, 4042, 8591, 3, 8817, 5153, 1906, 9634, 4886, 1804, 6793, 6132, 4067, 4403, 7260, 3236, 7269, 9795, 5293, 1171, 1699, 9760, 5668, 515, 9850, 911, 6205, 4851, 9624, 1884, 2448, 5119, 7597, 5012, 1951, 3924, 163, 6340, 787, 1808, 1706, 2216, 6331, 7401, 6578, 1060, 6876, 3888, 6823, 1364, 6291, 1857, 7010, 7620, 4022, 7226, 3490, 1242, 8889, 4370, 1886, 8031, 6355, 4620, 9771, 3258, 3977, 4188, 3130, 8358, 5404, 1440, 4255, 2875, 324, 3158, 9138, 1555, 7107, 7178, 4245, 3942, 6415, 5435, 5004, 5296, 5912, 5613, 9210, 6453, 7672, 9001, 8132, 5347, 7574, 7915, 653, 5370, 5928, 9442, 9424, 1473, 2322, 124, 9075, 1182, 6713, 3590, 1631, 3381, 3558, 6017, 476, 7940, 8448, 8676, 3260, 6220, 6859, 9077, 7205, 3050, 4935, 7887, 484, 363, 4031, 9968, 6026, 2246, 6045, 9553, 5129, 9021, 1270, 5046, 8629, 355, 3700, 2519, 9463, 6761, 4832, 4265, 2985, 7585, 5997, 1448, 2528, 5770, 9139, 5426, 5818, 6851, 4600, 6529, 212, 4910, 7722, 3607, 7975, 6611, 4572, 5501, 157, 893, 4348, 6503, 9275, 5193, 2312, 9872, 3060, 3767, 5103, 7654, 9037, 427, 8828, 5460, 174, 2178, 2602, 8536, 558, 8642, 764, 8773, 672, 491, 4974, 5667, 3027, 7146, 986, 2670, 4667, 6887, 9115, 5164, 5211, 9233, 5393, 1048, 8987, 7061, 4690, 6148, 8543, 8459, 3003, 6914, 6780, 8434, 3432, 6699, 7150, 9093, 3121, 302, 8996, 5258, 5947, 8760, 7584, 7306, 138, 2532, 6476, 4609, 6932, 1019, 1028, 5867, 8142, 2897, 6129, 3968, 1928, 9989, 6963, 7152, 5259, 7290, 5825, 7971, 49, 6276, 4490, 2839, 5438, 923, 917, 4113, 458, 2220, 1520, 9744, 9864, 2756, 909, 2182, 619, 1313, 4784, 8832, 8386, 4402, 8489, 6692, 8527, 2854, 8665, 1331, 3045, 2304, 9994, 8726, 5140, 3989, 1162, 2270, 7637, 7155, 2545, 3410, 2145, 8958, 9302, 9212, 6796, 5199, 2263, 1913, 1702, 8646, 8523, 6502, 8264, 2629, 8293, 4905, 396, 9867, 4957, 9274, 4028, 4391, 8154, 4687, 7600, 4902, 4322, 842, 4312, 3603, 7937, 8677, 84, 5596, 1034, 6094, 9347, 5972, 8581, 44, 3885, 2910, 406, 6954, 9588, 5672, 1421, 2467, 6497, 8177, 9090, 7329, 963, 3611, 7170, 1228, 6376, 3461, 8526, 584, 415, 7018, 2836, 3421, 8541, 6054, 4523, 9975, 8633, 9887, 185, 7533, 8511, 4064, 8037, 6618, 4648, 2599, 1161, 1163, 419, 6076, 3286, 321, 7317, 6704, 3404, 4884, 2042, 883, 9, 4185, 8720, 8898, 2607, 6311, 7346, 3172, 3376, 9935, 3809, 3399, 1050, 7241, 5001, 2251, 4785, 2483, 6706, 5556, 9801, 3291, 1092, 6381, 8274, 3912, 6623, 8409, 8115, 9836, 3743, 753, 6756, 5030, 9329, 3713, 9412, 7983, 1433, 8568, 2806, 8404, 6251, 9356, 6539, 8294, 8172, 1768, 6058, 4900, 6100, 5523, 7805, 7959, 1648, 5956, 8648, 3858, 7078, 2233, 3684, 8182, 2625, 4283, 5396, 8023, 16, 342, 8798, 2446, 9182, 3709, 1186, 2130, 1510, 8407, 7086, 9750, 1279, 3174, 1530, 9502, 6400, 2699, 9735, 217, 6631, 4054, 7870, 9038, 5224, 3202, 5375, 1615, 4380, 46, 6986, 8246, 6546, 128, 2581, 149, 6959, 468, 4117, 6828, 242, 1851, 9900, 5444, 9958, 8368, 3123, 4628, 9793, 1559, 7106, 1340, 349, 599, 123, 5762, 2053, 1838, 2430, 2164, 4081, 6135, 2515, 5624, 1144, 3628, 9738, 9546, 5759, 7673, 4847, 1325, 7742, 6666, 2068, 8105, 3220, 9562, 6559, 808, 5165, 4987, 7956, 833, 5377, 3711, 9032, 627, 294, 8130, 408, 8196, 9627, 748, 5005, 5907, 2834, 7161, 1762, 8036, 160, 5631, 6829, 3646, 1431, 6361, 3343, 1822, 6947, 2857, 8122, 7497, 4906, 2647, 8096, 2436, 4236, 3387, 9671, 8077, 9878, 1531, 1207, 8321, 4556, 8272, 9353, 3214, 9052, 875, 8693, 9578, 3908, 3193, 3307, 2417, 4848, 9344, 1741, 5718, 2747, 4903, 4566, 7638, 9220, 4460, 1726, 8033, 4770, 7723, 6258, 2017, 659, 6575, 3270, 3812, 8908, 3717, 7016, 3639, 4971, 3829, 2529, 2321, 5807, 4389, 8679, 2652, 5143, 3388, 957, 4958, 1488, 2486, 366, 3534, 2089, 7307, 5666, 6255, 199, 1533, 9756, 2118, 5540, 2241, 9712, 8755, 5579, 5530, 1899, 2918, 8046, 7063, 7368, 4477, 8537, 2722, 5382, 3687, 6176, 8423, 9514, 5861, 2000, 257, 4017, 2425, 6113, 3587, 9266, 7198, 6299, 3842, 5865, 4499, 8104, 3964, 135, 3794, 7074, 8671, 2979, 4939, 6248, 9512, 6451, 4377, 9927, 2916, 4149, 4551, 3230, 9257, 7859, 9236, 5003, 2403, 331, 1998, 4546, 5157, 6940, 186, 2859, 3686, 8113, 450, 6551, 689, 7791, 2217, 7483, 9498, 6599, 9295, 9449, 7391, 5673, 8483, 2278, 1478, 7532, 7606, 6142, 4044, 7053, 544, 6075, 6979, 3935, 3688, 7912, 8994, 1813, 697, 9005, 9645, 5866, 6521, 678, 4458, 1957, 4530, 4273, 6558, 6152, 3859, 9805, 2734, 3816, 8329, 8009, 9659, 3610, 4092, 2798, 769, 4875, 6877, 6619, 642, 7421, 8588, 2069, 6147, 2906, 1371, 7277, 9586, 2541, 928, 179, 2176, 3732, 763, 9911, 7847, 9186, 7180, 7487, 3940, 5351, 3746, 5495, 7109, 6834, 6049, 1925, 3565, 2964, 472, 7966, 9988, 8788, 1098, 7774, 2942, 1987, 9100, 9375, 7104, 8673, 5872, 3649, 4810, 2555, 9352, 5025, 7924, 8379, 9811, 2052, 6437, 6159, 8224, 7549, 6473, 8465, 954, 2675, 176, 4032, 4713, 691, 2187, 1656, 8683, 8136, 1903, 4829, 2061, 1121, 8856, 7949, 193, 5731, 2080, 8910, 7542, 6865, 7326, 3705, 3843, 3892, 5017, 9058, 921, 1618, 4506, 7430, 4564, 4060, 7051, 9448, 113, 4114, 4184, 1649, 7096, 1841, 3411, 1597, 6204, 2114, 227, 2392, 5431, 1948, 1091, 2995, 2900, 6374, 2372, 5428, 2074, 1259, 9268, 5430, 2287, 1577, 9010, 9880, 9977, 5686, 7733, 6353, 2418, 2961, 4356, 8161, 8863, 8925, 108, 4139, 5571, 5602, 8641, 6216, 2063, 2236, 4221, 7113, 2232, 1604, 7520, 9496, 9620, 8088, 1009, 7976, 2649, 3378, 2552, 9901, 9074, 3838, 1929, 7486, 8278, 4988, 856, 785, 8441, 2405, 2795, 6317, 9148, 8598, 5828, 9931, 4711, 2429, 9594, 2328, 8697, 2889, 4024, 4482, 5651, 9839, 4278, 495, 9455, 3192, 8060, 2230, 7518, 4088, 854, 5148, 1653, 7410, 541, 6167, 2922, 3364, 3398, 4204, 1625, 2354, 2894, 7158, 1107, 2830, 6721, 2970, 3736, 446, 4012, 2605, 8644, 5761, 7065, 8664, 6933, 2024, 5151, 4754, 1450, 1232, 4586, 1169, 5803, 5738, 1809, 7683, 7259, 9531, 9774, 7186, 7790, 3136, 3726, 6267, 6764, 1867, 4777, 2978, 6763, 1449, 9497, 9784, 524, 3009, 6988, 8044, 3766, 8589, 6244, 9753, 6808, 6904, 132, 5308, 4840, 9548, 8200, 5208, 6898, 9536, 6295, 6492, 191, 3909, 5027, 3899, 3175, 6151, 5487, 3559, 7278, 8147, 6847, 5539, 6671, 2619, 1660, 9520, 2260, 9934, 1837, 1739, 1715, 6227, 5118, 7341, 2505, 6508, 2033, 4437, 4448, 9834, 8405, 3695, 4952, 4723, 2143, 8291, 4432, 465, 4744, 1443, 3397, 535, 2860, 188, 684, 4930, 2677, 5645, 3339, 7952, 7295, 1552, 3141, 1900, 657, 9224, 4469, 7179, 5528, 7348, 1128, 2415, 6621, 55, 357, 6820, 8049, 1283, 7488, 3720, 5474, 606, 8670, 6668, 5350, 5278, 8769, 9748, 1716, 4179, 8056, 7529, 5082, 5451, 9009, 2732, 3854, 9134, 8410, 334, 3416, 3633, 6060, 5047, 1116, 1522, 4097, 91, 4225, 7055, 308, 8938, 2443, 6781, 7694, 2779, 7801, 7364, 2583, 6164, 6051, 9808, 2891, 8808, 6924, 345, 7744, 1879, 6277, 9154, 5065, 1677, 8225, 3237, 5730, 6689, 134, 1280, 4658, 1496, 5805, 8678, 1464, 4201, 8178, 4975, 7060, 5039, 2654, 6660, 3284, 204, 3698, 2137, 1598, 3072, 3507, 5387, 488, 5058, 3437, 2495, 8951, 9492, 2926, 6260, 2917, 9480, 6604, 4464, 4510, 2472, 7514, 4103, 8661, 4102, 5820, 4998, 5061, 8022, 5869, 6032, 7512, 4517, 5862, 6726, 4461, 7251, 858, 2686, 7019, 65, 4675, 4781, 4216, 877, 3658, 3056, 9002, 7254, 7412, 4023, 9861, 6862, 2351, 5369, 9198, 4203, 2204, 243, 5930, 3703, 7435, 570, 9041, 914, 3173, 7989, 9830, 2526, 8433, 5716, 1542, 9525, 8075, 6003, 2273, 45, 7320, 3756, 3259, 3295, 829, 1109, 840, 5357, 8653, 4135, 8269, 5439, 9020, 9951, 2248, 5679, 2888, 6110, 1696, 7826, 6693, 8417, 7494, 8470, 416, 4080, 8512, 2165, 2358, 5906, 418, 6320, 7559, 1873, 4325, 3030, 5066, 1694, 6680, 7309, 8937, 5281, 3112, 3895, 3692, 3059, 6256, 6991, 129, 8287, 952, 1824, 9882, 6634, 5986, 1608, 3754, 8462, 2641, 5554, 8895, 4649, 3073, 2320, 2394, 1634, 913, 9361, 7798, 5041, 2316, 9947, 3209, 8159, 6998, 9252, 6240, 4376, 5314, 4654, 8499, 7454, 6370, 271, 4366, 196, 8010, 388, 1088, 1904, 5688, 3327, 4142, 9330, 6385, 3567, 5660, 3595, 6339, 9317, 67, 1353, 1624, 274, 2298, 8622, 8212, 5160, 5366, 9288, 6428, 6688, 6001, 3201, 8662, 4326, 7140, 7947, 4983, 2908, 5458, 9099, 3002, 6316, 4722, 5301, 3801, 5835, 8406, 7038, 9653, 6673, 5663, 3954, 1133, 2396, 3113, 6802, 4726, 8039, 9368, 4596, 1563, 498, 6825, 9062, 8179, 2015, 4513, 841, 5371, 8288, 3224, 9081, 6589, 5415, 1827, 634, 1862, 1305, 8814, 3548, 2239, 8486, 7557, 6678, 1620, 7515, 2282, 9473, 337, 9955, 565, 996, 6620, 3035, 1932, 6777, 9407, 3504, 2075, 3933, 9650, 4052, 602, 8519, 601, 5465, 5226, 8843, 365, 6328, 7780, 2342, 5176, 7160, 137, 1195, 5055, 1753, 3733, 8980, 6490, 3522, 7331, 2138, 3897, 4254, 7850, 4397, 6911, 9137, 4645, 3914, 9711, 2757, 2498, 1400, 4206, 5682, 5629, 9314, 614, 8285, 1681, 7981, 7875, 2592, 8528, 7893, 3945, 9047, 2463, 2648, 6872, 1468, 4872, 5335, 6590, 2280, 2975, 5483, 3955, 1509, 3468, 158, 87, 6408, 8899, 1197, 698, 4181, 7243, 935, 5510, 2433, 6574, 3083, 2384, 3446, 4948, 4877, 8263, 6458, 5471, 9056, 1032, 8476, 3787, 9896, 6399, 1984, 1321, 3765, 4143, 7605, 1260, 8876, 7941, 1633, 2956, 9362, 77, 799, 4913, 8784, 5412, 827, 4669, 4860, 8067, 5088, 9444, 8558, 6935, 6861, 9019, 4451, 2616, 1972, 873, 790, 8277, 6982, 152, 2237, 8612, 2695, 4006, 9710, 7516, 5343, 5958, 2091, 5819, 7424, 5702, 9692, 3412, 6482, 6275, 8704, 5882, 9161, 679, 5288, 1229, 7190, 7925, 4208, 4737, 27, 8795, 1570, 3956, 9930, 1014, 1909, 6684, 1140, 7929, 1831, 5182, 738, 8510, 1771, 1435, 8654, 8509, 6942, 4361, 6732, 2501, 1474, 197, 4408, 3135, 5801, 2781, 3163, 425, 7484, 1981, 3831, 4644, 5110, 4130, 3840, 836, 5769, 7778, 5202, 1258, 421, 1211, 2374, 8350, 8241, 4214, 66, 6418, 2820, 8873, 2940, 5675, 2250, 5397, 7728, 9897, 9218, 5198, 1481, 9335, 9648, 6586, 5122, 590, 2350, 7469, 1602, 3396, 8019, 2553, 2938, 3944, 3422, 625, 3629, 3128, 4621, 6226, 1493, 7726, 9806, 5829, 7033, 8002, 5981, 9251, 7220, 5983, 100, 5755, 8692, 7148, 9825, 6921, 3810, 561, 2185, 6858, 7986, 344, 7804, 1241, 5068, 6156, 534, 7710, 2264, 3216, 4582, 8270, 6860, 1700, 307, 5966, 4560, 6206, 6048, 5091, 3783, 5736, 673, 9441, 7546, 252, 1567, 1118, 9301, 1273, 1770, 3815, 9854, 17, 6683, 1190, 6753, 1966, 1931, 1543, 3049, 4115, 9857, 6717, 7169, 4415, 3543, 291, 9591, 2657, 9845, 3932, 7343, 7681, 8639, 3407, 8174, 8995, 4015, 4763, 8789, 6114, 1846, 2825, 6169, 2957, 7642, 4349, 7569, 5315, 8227, 4056, 2881, 6511, 4688, 4410, 1172, 7012, 3359, 376, 8904, 1075, 6279, 1245, 4867, 1606, 2615, 1099, 4456, 9259, 4186, 111, 882, 7164, 6158, 273, 6485, 347, 2767, 3659, 1721, 6701, 5453, 1828, 8209, 8479, 2389, 7587, 4896, 802, 6290, 9928, 6874, 5715, 8393, 5358, 5070, 5853, 2775, 1732, 6086, 1991, 929, 7451, 9050, 5383, 7308, 4520, 5618, 6127, 6555, 5935, 919, 1714, 5850, 751, 6980, 2668, 5878, 2627, 2870, 5496, 7141, 3821, 7135, 782, 8708, 1498, 4801, 9714, 8989, 5621, 960, 5307, 2412, 7467, 1845, 4741, 1832, 9564, 4515, 5159, 9397, 7493, 9685, 3958, 5795, 9759, 7423, 7193, 6499, 5491, 1835, 794, 3538, 6612, 4563, 3232, 2548, 6346, 1302, 295, 9853, 2148, 1551, 3478, 4079, 6650, 3095, 2323, 4799, 1544, 9665, 3979, 1484, 1930, 9323, 6294, 5311, 654, 3367, 8, 5889, 6514, 5031, 692, 5221, 4947, 727, 4094, 1583, 1472, 545, 6378, 5694, 4399, 7939, 5740, 4471, 2334, 1657, 7227, 1820, 6212, 2500, 8577, 6936, 666, 6463, 5917, 4663, 6023, 7764, 9237, 5051, 5124, 5576, 3729, 4274, 5741, 6210, 6641, 6254, 6427, 2903, 5131, 5154, 5751, 4863, 7427, 7846, 1319, 3573, 2044, 2951, 8570, 7231, 9456, 3918, 1797, 6229, 8988, 4549, 4309, 1375, 999, 6407, 8064, 695, 1744, 7653, 3231, 1471, 3644, 2664, 9307, 5775, 3033, 3255, 8160, 8535, 8836, 500, 9349, 6716, 5701, 8978, 1053, 5389, 4247, 2292, 7715, 3379, 9175, 1591, 2971, 2923, 9790, 884, 3350, 9883, 9453, 7677, 7535, 7271, 1821, 5185, 2620, 1339, 6507, 4992, 5821, 2999, 2313, 9034, 4085, 4925, 9384, 1119, 2502, 7181, 2574, 2455, 9096, 5903, 719, 3844, 7297, 1215, 792, 956, 927, 9350, 6705, 7024, 9119, 3110, 3132, 7610, 9877, 9618, 202, 7661, 2721, 2019, 2843, 5231, 7709, 2603, 7092, 5500, 359, 2550, 2258, 838, 5067, 2047, 105, 6637, 7495, 6488, 5104, 7151, 4787, 5876, 8551, 9583, 6171, 457, 200, 1920, 3941, 59, 8341, 8613, 2736, 5502, 3901, 1193, 9891, 8426, 6632, 6223, 9847, 9153, 7906, 7263, 3622, 8422, 7972, 1861, 9276, 8376, 5108, 7463, 8362, 9926, 6918, 1849, 3467, 6642, 9250, 7766, 8750, 7751, 9962, 1735, 663, 585, 7721, 8575, 3272, 9076, 7212, 3344, 145, 5754, 517, 2370, 5641, 2119, 7825, 3180, 2614, 3953, 7666, 4531, 5625, 7172, 7903, 229, 7436, 5771, 1085, 3764, 5943, 71, 6441, 5214, 7933, 5792, 1756, 413, 318, 7865, 1392, 3182, 1002, 8238, 7245, 9890, 8545, 5788, 6087, 3038, 4752, 9146, 1276, 2076, 4758, 1083, 5069, 9029, 8032, 1044, 6803, 7560, 4172, 2155, 2866, 8398, 644, 3326, 5508, 2208, 1377, 9297, 8762, 3164, 6015, 8686, 589, 1499, 8976, 6975, 743, 4756, 5242, 6296, 7993, 3305, 566, 9703, 5359, 4739, 4234, 1409, 6515, 7823, 8153, 5978, 7658, 7276, 2473, 4495, 4788, 2536, 1076, 8896, 1115, 3140, 2113, 9914, 4345, 3031, 5476, 3042, 896, 3417, 5137, 7490, 4299, 8318, 527, 9478, 9327, 9221, 9843, 2020, 3546, 3985, 3495, 9501, 4209, 7896, 7773, 9992, 4045, 8902, 723, 588, 1850, 7951, 1176, 3676, 823, 7028, 3041, 3630, 830, 114, 8999, 5665, 8597, 9921, 2407, 6718, 3105, 577, 2062, 4960, 3289, 985, 6419, 9803, 4210, 7686, 3176, 4553, 7375, 9763, 6162, 5266, 9607, 62, 1749, 3804, 3120, 5329, 5768, 7385, 8548, 9147, 7737, 7381, 3974, 2380, 1785, 8336, 4472, 9406, 8859, 7580, 3609, 4624, 9184, 3252, 3963, 5349, 7501, 2147, 1175, 4945, 9376, 5470, 1796, 6992, 5705, 1774, 4868, 4241, 3871, 8924, 734, 888, 8490, 5096, 5123, 9084, 8993, 1437, 7977, 5279, 382, 2189, 4394, 4387, 106, 7763, 8472, 9516, 5949, 7075, 8635, 2160, 3993, 4173, 9105, 6332, 6857, 9582, 6266, 3082, 1742, 2445, 1103, 594, 9641, 7564, 5044, 9399, 2778, 7504, 2780, 8384, 1705, 4762, 2704, 4857, 4589, 6985, 8236, 9689, 6089, 6041, 5993, 7258, 9133, 1236, 5692, 5203, 5392, 2421, 8861, 9383, 1512, 7201, 4697, 4954, 7305, 1202, 8346, 4660, 2762, 216, 6010, 6145, 1491, 7680, 9939, 2909, 3820, 9332, 8636, 8756, 1628, 8416, 5332, 6198, 2987, 9129, 2049, 1971, 9663, 5937, 6273, 8245, 2914, 2962, 8722, 1795, 1466, 1227, 9728, 4271, 5374, 7836, 4476, 6095, 4962, 4133, 8625, 8124, 3996, 2097, 7394, 7316, 6894, 6535, 3266, 8366, 40, 1942, 238, 8823, 466, 6218, 2785, 1540, 9291, 9782, 6069, 4087, 5216, 1330, 1536, 6518, 2072, 6435, 8150, 5323, 3753, 8070, 7334, 6783, 2577, 26, 6047, 6213, 736, 9373, 4748, 9203, 3065, 8965, 1470, 5514, 3037, 3415, 4972, 3024, 4864, 4132, 2153, 3245, 4501, 4898, 1817, 1221, 7550, 7555, 8866, 7679, 3204, 9573, 7567, 1671, 4838, 6430, 5610, 5289, 15, 866, 7301, 3355, 9642, 1309, 2933, 5486, 437, 214, 6239, 8427, 1097, 1412, 8975, 6925, 6814, 519, 9990, 1082, 4931, 8787, 3773, 7745, 2168, 6163, 2955, 6609, 4074, 7315, 1264, 3157, 8687, 4048, 6506, 172, 7094, 9110, 6039, 618, 8149, 5777, 4390, 9206, 6978, 364, 4219, 7112, 2193, 8229, 3099, 7927, 6771, 1877, 3612, 9915, 5002, 5520, 4328, 2824, 9438, 2004, 7216, 1985, 2976, 5141, 4478, 1112, 3665, 8063, 3975, 4078, 7056, 1871, 8674, 5797, 3153, 3159, 647, 3867, 7793, 5158, 1416, 1428, 7408, 4623, 1441, 1801, 3408, 4683, 2679, 1077, 3806, 7934, 6115, 7923, 3470, 8187, 912, 2683, 7754, 9164, 3551, 7489, 8761, 9409, 1288, 3366, 3281, 2611, 1329, 6752, 6915, 4195, 8748, 9940, 9391, 3423, 8595, 8571, 362, 6579, 9466, 2533, 1214, 7641, 4242, 4427, 2402, 593, 6146, 6695, 325, 6905, 8872, 7797, 3803, 9919, 1021, 1829, 7796, 2338, 7889, 7873, 8831, 8458, 1297, 5479, 8716, 8314, 2698, 8444, 9258, 3115, 3210, 5287, 3440, 2667, 5808, 389, 8119, 2731, 6261, 5181, 906, 2805, 8626, 7388, 2588, 926, 5243, 3883, 680, 581, 6543, 7362, 9025, 7864, 1238, 2744, 6949, 5425, 2807, 4353, 6944, 949, 8556, 4354, 7044, 4532, 8101, 1424, 1953, 3662, 1898, 1842, 563, 1446, 532, 9596, 4567, 7450, 4049, 651, 1394, 7088, 1787, 272, 8158, 88, 2963, 1157, 7313, 6233, 2841, 9670, 9140, 7914, 9552, 7387, 9247, 591, 2518, 5637, 4773, 3391, 6265, 3800, 721, 9405, 8092, 2202, 6888, 3433, 8933, 6659, 2579, 3442, 161, 1275, 7789, 6675, 2980, 2277, 2171, 2530, 6031, 1780, 8746, 2331, 1645, 8430, 316, 7577, 8466, 7505, 4725, 3835, 3916, 9382, 8411, 7913, 9747, 9135, 3051, 7407, 2128, 5235, 3314, 7285, 722, 3205, 3930, 5833, 662, 3274, 296, 2831, 8298, 9720, 931, 328, 96, 542, 5587, 7211, 1465, 9984, 3911, 3881, 6541, 5342, 1267, 9445, 4329, 6250, 5592, 8753, 9597, 8192, 3161, 5737, 6160, 807, 5597, 4292, 5563, 5196, 2482, 8408, 4714, 5220, 974, 7182, 6810, 1272, 2116, 6832, 4491, 1504, 7199, 5623, 3797, 7692, 493, 18, 2001, 8981, 9499, 9170, 991, 8076, 4211, 469, 6372, 9732, 8253, 4880, 8682, 5550, 4248, 7468, 2700, 7657, 9647, 1515, 5300, 3313, 7644, 8774, 9016, 664, 9542, 2404, 7025, 6178, 4834, 346, 4170, 7868, 3358, 2058, 5662, 9601, 4145, 6272, 9072, 1876, 4191, 1346, 3744, 7282, 8377, 735, 9459, 5677, 2480, 5337, 7021, 7099, 9726, 3518, 3063, 6830, 8068, 3966, 7523, 9196, 4035, 3498, 8928, 6957, 1370, 2409, 9831, 5764, 2355, 6247, 9272, 4618, 240, 1254, 5697, 2793, 9222, 1262, 8354, 6664, 6776, 2190, 3505, 3530, 9188, 3207, 1652, 605, 8816, 3088, 8850, 3900, 4338, 9464, 8923, 6343, 34, 1937, 5434, 4541, 7883, 5700, 3642, 8890, 2339, 1589, 2095, 7279, 1825, 1595, 9091, 6523, 6581, 1529, 5394, 5617, 2728, 7338, 6841, 7682, 4897, 3348, 750, 6681, 3776, 5456, 4360, 7047, 4367, 6439, 9945, 4351, 3453, 1266, 8853, 5570, 806, 4710, 2635, 3643, 8875, 9475, 2706, 4230, 5551, 6649, 3297, 1641, 1975, 1408, 4405, 3763, 891, 1235, 9333, 1547, 8751, 4144, 2789, 7581, 136, 1418, 7973, 3586, 2161, 86, 8944, 8373, 8261, 5189, 2812, 1950, 1166, 2245, 4617, 6734, 4742, 6999, 9392, 167, 9722, 8618, 4565, 8118, 8112, 3449, 5006, 1252, 9255, 2551, 3683, 7419, 3921, 970, 1226, 7558, 7048, 7247, 3870, 4929, 1057, 7027, 3414, 2524, 2151, 1072, 5527, 1863, 1326, 8777, 2676, 7586, 1198, 8478, 6565, 4548, 8420, 7252, 4526, 1355, 8062, 8102, 1387, 1722, 4942, 2546, 8028, 8073, 6768, 8302, 1173, 9286, 2154, 6531, 6337, 8822, 4622, 343, 6826, 4666, 5562, 2947, 4183, 3111, 1130, 4319, 3333, 572, 8501, 6974, 7173, 3802, 1342, 6868, 3360, 1968, 6849, 4917, 5282, 3057, 3213, 6020, 8546, 2184, 8100, 5927, 8089, 2842, 1380, 384, 1220, 5429, 4238, 7613, 6033, 632, 4824, 8452, 2376, 920, 9584, 7931, 4101, 4174, 7855, 5709, 567, 6038, 1982, 4593, 9136, 655, 5724, 8116, 9204, 8157, 3269, 5209, 2123, 2892, 3365, 5678, 5395, 6193, 5223, 6348, 9849, 6773, 1016, 470, 7072, 3915, 4418, 7017, 8304, 3354, 7000, 8252, 6475, 5264, 151, 3855, 4421, 6744, 2705, 8729, 1222, 5925, 2055, 1670, 7556, 641, 6303, 7431, 1405, 3755, 8888, 3336, 2172, 6326, 479, 1389, 8322, 426, 2390, 890, 7720, 4385, 6293, 5725, 7503, 7619, 3976, 4268, 2307, 2293, 4984, 8300, 2662, 7382, 2493, 1160, 6587, 5717, 7273, 236, 5418, 2765, 7526, 1738, 8401, 5577, 7917, 2499, 7841, 901, 6639, 9530, 7485, 5874, 8365, 9835, 431, 2874, 2457, 2079, 6412, 1776, 7357, 1152, 8117, 4800, 5318, 5929, 4150, 7714, 4543, 8450, 8175, 9023, 475, 312, 1020, 6368, 9142, 6902, 3191, 7183, 169, 4062, 403, 7043, 6484, 6930, 786, 6000, 8907, 9602, 604, 9773, 8094, 4518, 8691, 6211, 5854, 4190, 9551, 4625, 6429, 218, 4704, 1727, 1247, 8971, 7814, 7363, 4540, 820, 5976, 2332, 789, 94, 7023, 6225, 2102, 3509, 1246, 1251, 975, 2046, 3544, 7734, 1460, 7225, 2814, 6908, 4302, 7071, 2192, 5149, 1397, 3502, 8107, 8372, 3145, 1102, 2249, 6478, 4545, 4470, 2014, 9846, 4819, 8375, 9345, 9371, 9472, 3349, 6848, 5913, 2007, 4096, 6665, 7684, 8080, 8710, 3938, 5790, 9986, 4189, 20, 8596, 2302, 1058, 815, 5589, 2565, 2183, 127, 9428, 1545, 8218, 3494, 6964, 4105, 330, 5134, 4584, 3143, 3799, 7365, 3570, 2454, 8041, 4717, 1521, 3084, 3482, 3302, 7111, 4215, 6938, 8120, 9108, 2225, 3473, 9768, 5841, 4874, 1632, 2715, 5247, 1497, 3853, 5843, 9676, 8650, 1654, 546, 2895, 6729, 3025, 1963, 3273, 1037, 2782, 2660, 3493, 1794, 5708, 1006, 3315, 2626, 3017, 485, 5603, 5364, 3166, 3127, 48, 3532, 443, 1277, 8882, 707, 6084, 5815, 6661, 7604, 8805, 9598, 7623, 8675, 1269, 9746, 2946, 7571, 1011, 4941, 9336, 5083, 9606, 511, 4783, 6636, 9012, 2127, 3386, 9434, 5057, 3319, 9694, 8361, 2385, 6735, 5699, 7358, 3296, 7819, 5766, 1868, 6074, 4505, 4672, 5964, 8794, 3615, 2439, 6945, 5135, 6504, 1560, 8418, 3277, 7500, 9690, 7678, 9889, 5944, 5640, 6498, 9621, 5802, 7905, 4568, 5449, 9187, 8647, 9870, 7413, 8862, 5464, 28, 7014, 5855, 6103, 9321, 7066, 2045, 1736, 6576, 2876, 1934, 8469, 6133, 553, 8289, 4430, 5187, 9798, 1486, 2810, 3934, 5236, 8932, 1664, 2449, 5241, 2738, 6364, 4569, 737, 9754, 1518, 2213, 8081, 1888, 8796, 4008, 2727, 2784, 505, 4862, 3170, 5999, 8800, 75, 1454, 5184, 4871, 9379, 1429, 8824, 1887, 9522, 4579, 8162, 1988, 3578, 4646, 8628, 5419, 6172, 4955, 1194, 1969, 0, 4177, 9085, 2037, 3491, 6053, 7036, 90, 6967, 370, 6629, 6141, 1586, 1684, 3862, 5578, 2450, 5503, 9595, 3497, 3626, 6238, 3516, 6452, 5982, 6885, 955, 6769, 7663, 9465, 5179, 2559, 7344, 9563, 6614, 9674, 4891, 6345, 5783, 4831, 777, 6657, 4982, 9672, 3789, 3592, 3982, 3770, 7837, 5952, 3788, 656, 3928, 3488, 5642, 2716, 9322, 8021, 1927, 3150, 5836, 9242, 4533, 9823, 910, 4767, 6334, 5391, 6393, 276, 5653, 4815, 649, 2365, 3151, 1708, 4281, 9565, 2141, 6647, 397, 5138, 6748, 7204, 8668, 3094, 6446, 1775, 9109, 6461, 2713, 3946, 8256, 1938, 6118, 9625, 7340, 824, 1720, 8616, 8775, 2813, 9451, 5168, 386, 3280, 6792, 596, 9414, 3784, 784, 8232, 9993, 745, 7892, 4400, 7473, 7852, 153, 7696, 6456, 7768, 254, 6491, 3669, 8582, 6906, 5509, 3219, 7022, 4610, 4630, 7314, 7464, 2025, 7246, 6085, 6056, 1154, 6669, 5352, 2149, 5139, 4442, 1294, 358, 379, 7938, 189, 21, 6043, 1205, 8018, 676, 4769, 6536, 9557, 1892, 9745, 4218, 101, 959, 7590, 7223, 9547, 9580, 6180, 2181, 933, 7153, 5405, 9300, 5212, 1847, 5294, 3707, 7191, 6421, 5050, 8776, 575, 7460, 6394, 7004, 6667, 6740, 1967, 8715, 2470, 4694, 9217, 4466, 1882, 6386, 5890, 9367, 8247, 2612, 1680, 9819, 2474, 516, 7871, 5312, 361, 713, 6450, 8871, 6081, 4196, 1350, 8176, 159, 4508, 1603, 4266, 3249, 6443, 7691, 7752, 1883, 3288, 5583, 9778, 504, 2846, 7612, 3919, 6577, 8207, 5163, 2447, 9725, 5188, 7707, 2174, 6782, 7339, 2107, 117, 4973, 8637, 1348, 5957, 1765, 4759, 3685, 8114, 2991, 2023, 988, 2633, 2271, 8803, 5295, 2475, 8916, 1848, 6822, 2745, 8008, 8343, 3779, 2088, 8087, 3441, 7599, 6821, 6532, 8707, 1636, 1807, 78, 2152, 8848, 5440, 718, 7815, 6062, 9118, 9254, 5526, 6044, 7919, 3951, 3533, 1068, 4878, 2359, 6037, 9491, 6186, 314, 2124, 4182, 6090, 3256, 4038, 3868, 455, 3203, 1622, 6312, 8424, 8198, 6955, 961, 603, 9264, 8690, 2177, 8719, 3438, 6952, 7502, 7719, 8184, 1253, 5360, 4776, 3519, 5252, 6728, 198, 5831, 180, 6816, 1561, 9208, 1240, 8963, 3263, 5763, 50, 2645, 9681, 742, 5011, 1089, 1803, 33, 4856, 3571, 8733, 7202, 4018, 4141, 1582, 5881, 9024, 5255, 5445, 7843, 8884, 7527, 130, 3828, 3452, 1569, 9933, 8860, 4378, 7008, 1231, 424, 9558, 6682, 3262, 6525, 175, 1979, 9226, 6727, 3650, 1149, 8874, 5280, 6403, 3310, 1475, 2465, 6652, 3285, 9374, 2973, 2516, 3585, 7795, 2608, 4976, 8953, 2596, 6730, 2844, 7582, 4125, 5858, 9162, 6469, 9078, 3405, 2570, 3904, 9331, 5599, 3827, 9202, 335, 140, 8592, 7955, 8603, 4488, 3321, 8935, 5080, 5945, 5973, 9000, 9026, 7757, 6486, 5225, 7187, 8820, 7049, 4100, 8857, 7554, 6059, 9200, 3445, 6442, 6760, 9248, 7802, 4131, 2090, 97, 6195, 2294, 1087, 7776, 2377, 4148, 6789, 8335, 658, 1800, 6465, 7521, 5178, 8249, 4374, 9729, 9860, 2229, 1430, 1469, 6970, 5581, 4619, 2509, 8518, 9544, 1410, 3957, 595, 9452, 6150, 9796, 1125, 6350, 1964, 9639, 8812, 6336, 4718, 7409, 4384, 6813, 9390, 9287, 3311, 5971, 8312, 7270, 6571, 3406, 7380, 8879, 6091, 4364, 3303, 1557, 7831, 7050, 9856, 209, 4699, 8190, 2650, 2341, 3062, 822, 9833, 3971, 5549, 1296, 4235, 5032, 7163, 9822, 3562, 4768, 6380, 4021, 5962, 7077, 8432, 8970, 2880, 9045, 2136, 5524, 2930, 6065, 461, 8012, 4298, 2006, 1945, 2018, 4310, 9970, 4673, 5729, 8315, 7030, 2343, 7045, 7738, 1995, 478, 4993, 6751, 9377, 2868, 5580, 6677, 8369, 2811, 8203, 6922, 2630, 8801, 667, 700, 4138, 8061, 942, 3500, 5353, 8959, 2823, 2790, 5560, 7848, 5040, 845, 3029, 5988, 3845, 8129, 1042, 5987, 7863, 5585, 8793, 6470, 1212, 661, 650, 761, 4161, 2566, 4782, 2540, 5408, 4612, 6698, 3074, 4664, 7311, 9668, 3199, 9151, 9577, 660, 4187, 9526, 3618, 5923, 9103, 9629, 916, 4046, 9851, 5779, 9149, 3459, 6027, 724, 3006, 270, 5594, 8723, 8786, 5132, 3208, 3187, 219, 4339, 1852, 2081, 8562, 4731, 1709, 5344, 6405, 7456, 6257, 3948, 8553, 6397, 8221, 8299, 3239, 1579, 2982, 5201, 8259, 4577, 6253, 3841, 5942, 7009, 76, 9315, 2764, 4629, 5361, 8520, 7470, 9490, 9312, 165, 1479, 5417, 687, 5887, 4352, 5125, 6202, 7292, 5271, 8815, 958, 3839, 4286, 7105, 1347, 3847, 2770, 5991, 5270, 821, 5636, 936, 9632, 2929, 8839, 4592, 3774, 8601, 4160, 6723, 2777, 3837, 5237, 7834, 9925, 2435, 4792, 7342, 6939, 765, 5703, 4082, 6913, 8275, 578, 739, 6591, 1436, 6592, 6778, 9804, 5466, 8615, 8583, 6434, 5643, 3730, 2671, 4435, 6547, 5059, 4999, 9884, 2589, 1865, 2337, 8565, 2330, 6644, 5557, 4883, 8485, 2558, 7898, 2901, 249, 892, 2170, 2851, 2048, 1420, 6219, 2972, 9185, 7132, 9723, 990, 9294, 2173, 4439, 7361, 3444, 1084, 6583, 6943, 4494, 2584, 1139, 6881, 9402, 360, 1926, 9568, 7943, 6864, 7743, 5079, 3574, 9319, 9987, 8573, 220, 2673, 7885, 4422, 5498, 2537, 1933, 717, 8211, 9852, 9284, 4313, 7708, 3824, 5537, 8382, 6993, 7079, 3068, 2073, 6324, 3564, 6236, 6923, 7668, 3981, 4977, 2932, 8542, 1623, 3959, 3536, 7854, 3418, 2562, 4146, 5454, 1980, 2163, 2140, 2050, 2027, 8905, 8698, 4887, 3005, 5173, 2927, 8947, 2022, 8782, 8900, 5245, 4118, 8121, 3139, 6362, 5121, 2936, 7433, 8425, 245, 9982, 5169, 5989, 154, 1764, 729, 9932, 1870, 1248, 7125, 3155, 7895, 4730, 860, 1955, 9011, 781, 190, 881, 5582, 1697, 2666, 4253, 2564, 2408, 405, 8397, 6910, 3712, 894, 5309, 3879, 703, 3309, 9057, 3898, 8308, 3719, 9082, 7969, 4598, 951, 4104, 5683, 5007, 5303, 4381, 1703, 5744, 98, 6, 1167, 339, 8223, 2763, 5656, 8742, 9895, 5076, 8651, 9789, 9817, 7283, 70, 9223, 746, 2848, 7015, 1062, 9334, 1008, 8949, 7998, 9048, 726, 5234, 1584, 3448, 4222, 8763, 2259, 6388, 8778, 3048, 9876, 5324, 1200, 6357, 1781, 9832, 4516, 102, 6886, 3325, 514, 5695, 7288, 1897, 6287, 6391, 52, 9211, 5555, 646, 4337, 6196, 6627, 3917, 2003, 9408, 121, 7222, 187, 9840, 2134, 6495, 7886, 8131, 7098, 3631, 8337, 6280, 6719, 4700, 4293, 3661, 9686, 3508, 7120, 4213, 8942, 6034, 540, 303, 9095, 8804, 7281, 262, 5167, 1593, 441, 5385, 3664, 2479, 3777, 1300, 816, 9094, 9178, 5248, 7618, 3248, 4108, 24, 12, 9400, 4467, 2428, 8026, 6767, 8757, 6714, 5901, 2787, 1025, 7089, 2850, 4602, 8684, 5338, 4873, 8395, 7426, 8072, 1390, 2274, 9013, 1291, 6799, 9693, 1438, 1168, 8694, 1317, 9802, 7236, 7274, 6854, 4005, 6120, 4093, 6907, 5133, 755, 1783, 9770, 4307, 1651, 4635, 5172, 9767, 1596, 22, 8781, 3851, 4086, 9539, 9064, 7498, 35, 993, 3579, 2494, 9014, 4121, 5488, 8666, 1527, 8855, 7608, 6687, 2718, 2340, 1237, 7046, 8165, 9487, 6785, 8610, 1024, 1096, 2990, 4647, 5830, 6542, 7935, 7102, 7688, 2742, 3792, 156, 6122, 304, 3882, 712, 1907, 5857, 6101, 8990, 8758, 7685, 9923, 2593, 5961, 8770, 4599, 9244, 3253, 9724, 1434, 2697, 2242, 8834, 7076, 7499, 2056, 5840, 8920, 852, 7611, 8640, 1036, 9229, 7717, 263, 9289, 391, 6711, 7439, 8740, 9950, 3638, 2809, 775, 7627, 8168, 5980, 3332, 1682, 5102, 3663, 2993, 5283, 6715, 7970, 6774, 9386, 900, 9813, 4159, 5804, 7525, 5940, 9403, 5745, 9116, 6107, 1487, 1065, 608, 5615, 6873, 9785, 5094, 2223, 1135, 5748, 7570, 4220, 3036, 8186, 6524, 4924, 5796, 7944, 6411, 1012, 994, 7781, 4291, 368, 8974, 378, 1127, 2070, 8792, 2741, 5379, 8983, 477, 4373, 7332, 8024, 7097, 8685, 3496, 6594, 3983, 9364, 1142, 4607, 5472, 6474, 4481, 5009, 9646, 6516, 2121, 3080, 2215, 4760, 4158, 7713, 693, 5413, 715, 741, 6315, 1811, 749, 7108, 3725, 694, 8421, 4362, 6018, 7536, 8936, 8672, 9848, 1600, 3179, 9446, 3557, 7874, 7616, 7116, 289, 1959, 1413, 4156, 5045, 3133, 3514, 8214, 8054, 3606, 880, 5363, 6788, 4514, 6690, 1710, 4892, 9683, 797, 5605, 7530, 5732, 4728, 6077, 7447, 3865, 5772, 5400, 2419, 3735, 1328, 3833, 5493, 6467, 6962, 2534, 5655, 6839, 1673, 4297, 8992, 2431, 2752, 4616, 5654, 6840, 4152, 9462, 8442, 3822, 9535, 2761, 7664, 1495, 1287, 859, 4684, 5265, 4033, 8609, 9436, 7379, 4178, 3910, 8566, 4780, 4332, 7544, 7862, 5217, 2488, 8403, 8181, 4519, 2967, 1366, 8090, 5333, 2773, 3374, 8208, 3375, 6738, 9055, 1052, 8273, 2256, 5753, 6468, 8500, 2769, 4475, 4013, 2103, 7650, 9657, 2639, 201, 2094, 849, 4474, 946, 5933, 5302, 8737, 2597, 401, 6884, 2413, 8829, 5015, 5789, 5893, 2766, 2303, 235, 1500, 373, 5054, 2856, 8574, 162, 1599, 2299, 4732, 752, 8268, 9995, 1734, 5320, 7617, 3102, 6014, 3243, 9112, 8134, 4922, 2275, 5452, 7239, 480, 9111, 1916, 908, 7509, 530, 7948, 9427, 5844, 1018, 8047, 3188, 3055, 9028, 6217, 7670, 9658, 3656, 8811, 120, 7475, 7466, 4790, 4275, 6149, 239, 6584, 5671, 3554, 6770, 848, 3380, 6283, 8193, 6300, 4932, 8901, 9207, 5586, 5658, 5611, 301, 7121, 8095, 4300, 5, 1027, 2939, 9022, 9423, 3907, 6106, 1818, 3521, 6843, 1041, 1665, 9592, 9051, 9541, 2618, 8371, 7731, 598, 6624, 2363, 1621, 3118, 7842, 1386, 1357, 250, 2284, 7228, 83, 6765, 5752, 7817, 2802, 42, 8138, 7265, 9550, 6022, 8643, 7166, 8894, 3758, 6123, 8378, 7353, 7813, 241, 9107, 9088, 9417, 5546, 6402, 4429, 4232, 3723, 6836, 1643, 8313, 8475, 9510, 7291, 8576, 2438, 9983, 5722, 1911, 9868, 518, 4637, 2432, 2575, 1108, 6308, 1039, 6382, 3486, 9346, 669, 4341, 4968, 9519, 3745, 6262, 4605, 6323, 835, 7289, 3439, 178, 8195, 2637, 4685, 4404, 9313, 110, 871, 3168, 3479, 9766, 4486, 2266, 6230, 7, 8316, 6871, 795, 5883, 1281, 8327, 3819, 9440, 3177, 7995, 2847, 5321, 3184, 2348, 983, 4090, 3361, 8765, 7703, 3798, 4463, 4433, 6972, 3472, 5356, 8533, 3931, 3668, 5664, 8530, 5566, 4934, 3876, 6259, 9617, 5735, 6819, 4413, 9265, 6997, 3022, 9985, 905, 5477, 3896, 6500, 9070, 7882, 1056, 223, 4043, 6951, 1494, 8841, 6831, 4445, 8389, 7575, 9809, 2410, 8169, 2723, 6271, 6384, 2863, 2009, 2100, 4671, 6455, 4163, 2997, 7749, 740, 6708, 3752, 4668, 7879, 770, 9396, 5505, 7872, 4365, 5856, 8906, 8921, 1047, 7067, 107, 5644, 3021, 5195, 6522, 9195, 6330, 9898, 889, 155, 1323, 8029, 8083, 2776, 705, 9192, 4850, 8847, 998, 3582, 1646, 5574, 6187, 6724, 1303, 4901, 1199, 9073, 5985, 9786, 3018, 586, 9180, 9372, 6601, 9471, 5062, 7159, 3718, 2801, 9677, 6130, 61, 8340, 9177, 9554, 4250, 7759, 6501, 9410, 3322, 5723, 8991, 4363, 5871, 9660, 8790, 2685, 4264, 1610, 7312, 2011, 1217, 7264, 2786, 2267, 8617, 4119, 9395, 434, 9401, 4493, 7496, 6766, 412, 6745, 6920, 4073, 4689, 9422, 7137, 5517, 2794, 8838, 5614, 3780, 3402, 4918, 2382, 2, 8339, 1763, 9780, 5911, 7240, 5992, 1150, 3090, 4063, 2016, 801, 436, 9117, 3980, 9127, 2896, 4315, 3389, 2636, 6505, 5461, 7351, 744, 6036, 7414, 8085, 4853, 5105, 4636, 9702, 7987, 9652, 7189, 9604, 3651, 9156, 3146, 2958, 6194, 3998, 4014, 6030, 3293, 8216, 7144, 5959, 4652, 1924, 5892, 9610, 8243, 6144, 1104, 5146, 3877, 4285, 8125, 7377, 2423, 6264, 4334, 2254, 6104, 6406, 4039, 6655, 8412, 3019, 3727, 8360, 8813, 8057, 5299, 9126, 8040, 462, 4076, 5365, 1725, 636, 2638, 3103, 9069, 2038, 1676, 8347, 5632, 2651, 2345, 1526, 7384, 2084, 387, 9589, 8271, 9765, 939, 5038, 9474, 5661, 8727, 2774, 3351, 1120, 2040, 5968, 6697, 3893, 7229, 9097, 6950, 2950, 3588, 9101, 9370, 4223, 265, 6741, 6124, 4053, 9113, 7261, 1686, 4946, 9065, 6545, 8655, 4524, 7352, 7037, 2253, 8538, 2228, 9976, 6092, 3599, 7054, 7039, 2301, 4057, 972, 6702, 5071, 564, 9311, 2083, 3278, 2659, 9479, 5490, 1003, 720, 6755, 8681, 5192, 3382, 776, 772, 3926, 9731, 7881, 8015, 3254, 4870, 3451, 4123, 3553, 7005, 4321, 7878, 1403, 3949, 261, 1688, 432, 267, 4034, 677, 9776, 2771, 3604, 5073, 8531, 2816, 3426, 898, 3608, 1532, 7851, 4147, 4398, 7267, 1263, 4846, 447, 5884, 7876, 5334, 4342, 8600, 3330, 1093, 4538, 3215, 670, 9385, 1550, 1395, 3430, 7762, 5904, 1513, 6448, 2838, 4812, 2026, 1153, 3352, 2523, 4537, 6658, 1374, 7698, 4026, 9529, 395, 4441, 47, 4938, 9054, 4479, 9430, 4202, 1312, 4534, 6464, 5954, 4826, 68, 1170, 8244, 3138, 5109, 7400, 1655, 834, 8194, 5756, 8265, 1049, 7093, 8702, 7206, 1840, 6540, 1814, 1250, 3016, 2497, 728, 9874, 6610, 861, 3096, 7543, 1453, 423, 7374, 7480, 1000, 1944, 5442, 8481, 4676, 2031, 4650, 4967, 4128, 6489, 7242, 1751, 7234, 4272, 5433, 2315, 7230, 8496, 1908, 5174, 8560, 5827, 4436, 3238, 3257, 8955, 5478, 6595, 2661, 3108, 587, 7954, 5286, 9495, 7213, 5826, 1666, 6747, 6833, 277, 7110, 3401, 460, 4613, 8215, 456, 297, 3728, 7477, 3154, 2590, 375, 704, 2959, 7196, 8818, 195, 8164, 3318, 4786, 3524, 6347, 9150, 6398, 7625, 2754, 13, 8913, 5099, 4528, 3769, 143, 895, 3178, 3119, 103, 8945, 4426, 4112, 2296, 8878, 7255, 6083, 1159, 865, 6215, 3793, 2882, 7124, 1378, 4709, 1362, 3086, 4779, 7069, 6815, 3995, 5994, 1746, 4558, 774, 1396, 4827, 9737, 3852, 783, 2587, 1327, 7296, 2595, 6207, 4314, 2632, 2708, 8731, 7134, 9305, 809, 5194, 7877, 232, 2285, 6733, 7095, 2379, 9341, 964, 5362, 3067, 1698, 7123, 1064, 8858, 9079, 7443, 1180, 7578, 5990, 8139, 1001, 8069, 7624, 2573, 9191, 6390, 9387, 7192, 1989, 2214, 3218, 950, 147, 1737, 7565, 4324, 7632, 8438, 1960, 887, 1639, 3517, 5233, 2687, 9049, 9643, 945, 3431, 6749, 9719, 5900, 8429, 3028, 9503, 6605, 3464, 8030, 255, 1334, 6011, 6510, 536, 7437, 4879, 6459, 5482, 5638, 4727, 6301, 9044, 1541, 3308, 3046, 4375, 3947, 6121, 420, 5206, 1081, 5113, 3171, 3122, 4484, 9066, 3100, 3004, 8477, 4692, 9234, 2469, 1855, 1022, 2481, 868, 1005, 6597, 6966, 3424, 8437, 2768, 1131, 1574, 1360, 3148, 965, 8428, 610, 4198, 1349, 6433, 4240, 4091, 449, 9018, 5650, 1332, 6157, 8364, 8058, 9518, 9788, 4539, 886, 3370, 4716, 622, 7510, 3523, 5909, 9916, 2517, 3807, 231, 2907, 710, 7816, 1361, 1335, 9545, 574, 7651, 1853, 2821, 7513, 9249, 9366, 733, 6929, 5536, 4340, 2158, 7248, 5851, 6552, 9581, 3624, 6021, 9165, 7537, 2262, 4444, 205, 9245, 9888, 607, 9030, 9007, 8745, 3144, 8065, 5373, 7117, 1423, 474, 9457, 6185, 3747, 9398, 9585, 8217, 7739, 7188, 481, 1143, 36, 8001, 5822, 1249, 1080, 9699, 3010, 4129, 5908, 4511, 284, 9991, 6882, 6989, 9261, 3503, 6883, 5230, 529, 8864, 6640, 7176, 7609, 4277, 6613, 2222, 6786, 2609, 8599, 394, 4802, 6983, 9844, 6889, 4318, 2383, 8572, 521, 1954, 5462, 2690, 7157, 847, 4262, 9797, 6626, 6807, 4724, 7551, 3345, 7402, 6426, 9031, 3997, 496, 4843, 3085, 9904, 338, 984, 7982, 5535, 1713, 5213, 7921, 5914, 3290, 6466, 4301, 6566, 6237, 3264, 372, 6487, 4550, 4818, 1368, 805, 8206, 2853, 6396, 3535, 8495, 1432, 9667, 1265, 2691, 7631, 9035, 1507, 4355, 2108, 8167, 5467, 5806, 9879, 4346, 2707, 3206, 1196, 7441, 5060, 7319, 6165, 709, 9378, 9678, 5726, 1779, 1629, 5934, 5086, 7592, 5441, 980, 407, 8897, 9762, 5776, 953, 9709, 3598, 8728, 7266, 4343, 5354, 2644, 531, 2730, 4740, 5690, 8927, 1268, 1244, 879, 2862, 6593, 6968, 2883, 5742, 7985, 613, 3093, 1367, 7888, 280, 4921, 6844, 5698, 8954, 2008, 1322, 9869, 6373, 1224, 4243, 9278, 3225, 2543, 8276, 2255, 5304, 6242, 5481, 9260, 8474, 4040, 2911, 7711, 6686, 3721, 1209, 1310, 3292, 9612, 8048, 7372, 2106, 5832, 510, 9243, 9820, 7999, 8311, 2035, 3680, 281, 1191, 8505, 452, 5401, 5839, 5380, 9952, 714, 8352, 9348, 5127, 5222, 3734, 9701, 6553, 8020, 3091, 5427, 4428, 3693, 1878, 5317, 2681, 9824, 1535, 2064, 1760, 2832, 2196, 3760, 8967, 8484, 5691, 2655, 166, 3811, 7839, 1723, 2833, 5689, 3813, 9067, 1029, 904, 9071, 5074, 3320, 3699, 1376, 2905, 9493, 6784, 7960, 9575, 7127, 6471, 1965, 3849, 7910, 2157, 4997, 2468, 4861, 5200, 4069, 3066, 6392, 3165, 6322, 9343, 63, 7547, 9980, 4828, 3092, 6067, 754, 2490, 2719, 58, 7750, 760, 6197, 934, 2496, 5919, 6183, 696, 6228, 7207, 7660, 4047, 5767, 9061, 5652, 350, 8333, 3069, 8743, 690, 6961, 4842, 9413, 5559, 4288, 5064, 9549, 9328, 7645, 320, 1592, 4304, 5089, 5899, 9040, 9155, 5436, 5734, 9143, 1674, 1601, 8837, 5489, 1875, 402, 810, 9949, 7800, 9017, 7869, 1338, 1836, 508, 5386, 9285, 5297, 1896, 1422, 2861, 3677, 1351, 8151, 251, 7476, 2491, 3652, 9144, 1644, 168, 6422, 2751, 3081, 4657, 2743, 8903, 8202, 6534, 8849, 5534, 5448, 7963, 7133, 4580, 6625, 3613, 3762, 9571, 471, 5541, 5800, 3131, 5399, 5494, 2508, 3978, 7081, 7961, 3814, 5424, 7822, 3768, 8141, 4500, 5920, 7397, 5388, 1576, 1067, 4487, 4604, 1901, 5838, 5376, 7083, 3722, 3267, 7783, 1562, 1483, 4029, 814, 2539, 3142, 1658, 9006, 2466, 4527, 8627, 7827, 1301, 3109, 2643, 9292, 1691, 6155, 2308, 414, 5177, 5228, 9360, 5052, 7286, 2791, 2710, 6072, 7007, 3549, 6438, 2489, 4058, 5712, 315, 7640, 1566, 3750, 7674, 2977, 725, 9781, 6052, 582, 683, 6867, 3617, 4382, 2458, 5316, 8235, 732, 8155, 3627, 7655, 8367, 6462, 7845, 2510, 2572, 4070, 6563, 3275, 2243, 6304, 118, 2132, 3697, 8593, 1635, 3385, 5531, 7730, 3830, 5905, 4155, 7002, 1017, 5953, 5633, 5084, 1201, 5568, 7101, 2696, 1537, 4120, 2476, 6231, 1511, 4751, 2397, 4164, 5097, 3701, 8449, 7073, 2904, 9688, 6454, 8559, 631, 1659, 5575, 8251, 4764, 3657, 2827, 645, 6024, 638, 4267, 7539, 759, 2126, 8388, 6526, 6739, 371, 7548, 7210, 9656, 7399, 7129, 494, 5676, 8539, 9507, 4027, 9511, 8004, 5219, 600, 7978, 5939, 6662, 6588, 5793, 4401, 8127, 442, 3832, 4019, 8487, 5085, 3667, 2642, 8353, 9569, 9829, 3836, 7378, 2096, 6314, 2693, 2327, 7602, 9227, 7143, 7950, 133, 3537, 7900, 9661, 8734, 853, 7926, 7453, 464, 2755, 5693, 8929, 9981, 966, 3560, 1462, 817, 4295, 9271, 2111, 1256, 1758, 9532, 2032, 1523, 7232, 9908, 7936, 6479, 7761, 411, 8290, 1588, 1704, 5545, 6846, 5257, 686, 206, 4670, 9043, 1856, 1672, 8355, 3597, 9924, 7455, 7482, 5162, 2461, 9421, 7984, 8555, 652, 8228, 6679, 4154, 4661, 8985, 7806, 4885, 393, 2531, 9608, 224, 1952, 501, 2416, 6608, 5529, 3984, 282, 8986, 7607, 2772, 2758, 1365, 4001, 2865, 9695, 8724, 5670, 7119, 4927, 2314, 9033, 5885, 3848, 9862, 8852, 1609, 9913, 1516, 5144, 9697, 7712, 4595, 6606, 7174, 1890, 4814, 7552, 3529, 3101, 2935, 9871, 9938, 7084, 2051, 6530, 3678, 8883, 1789, 9644, 5339, 8984, 3097, 4712, 8254, 6126, 2511, 2272, 5896, 5205, 1941, 6607, 9316, 8219, 7458, 5403, 3477, 5750, 2349, 9616, 2300, 2969, 4051, 9036, 5706, 5834, 7576, 6757, 5657, 3635, 8266, 1640, 8349, 399, 7195, 9779, 5390, 2205, 3694, 5232, 9815, 5780, 3704, 8689, 1123, 2658, 6554, 4961]

if __name__ == "__main__":
    nc_filenames = get_nc_filenames(paths_config["dataset_folder"], permutation)
    save_npy_dataset(nc_filenames, 0, 9_000, os.path.join(paths_config["dataset_folder"], paths_config["train_data_filename"]))
    save_npy_dataset(nc_filenames, 9_000, 10_000, os.path.join(paths_config["dataset_folder"], paths_config["test_data_filename"]))
    print("Deleting nc files")
    for filename in nc_filenames:
        os.remove(filename)
