#!/usr/bin/env python3
"""
Script to convert PhysioNet 2012 data to Raindrop format and run MTGNN baseline
"""

import os
import sys
import subprocess
from raindrop_data_converter import convert_physionet_to_raindrop_format

def main():
    """Convert data and run MTGNN baseline."""
    
    print("="*60)
    print("CONVERTING DATA AND RUNNING MTGNN BASELINE")
    print("="*60)
    
    # Data paths
    data_dir = "/home/dcm.aau.dk/km20bf/Biomarker_FeatureGroup_GNAN/tmp"
    output_dir = "Raindrop/P12data"
    
    # Check if data directory exists
    if not os.path.exists(data_dir):
        print(f"❌ Error: Data directory '{data_dir}' not found!")
        return False
    
    # Check for PSV files (pipe-separated values)
    psv_files = []
    for root, dirs, files in os.walk(data_dir):
        for file in files:
            if file.endswith('.psv'):
                psv_files.append(os.path.join(root, file))
    
    if not psv_files:
        print(f"❌ Error: No PSV files found in '{data_dir}'!")
        return False
    
    print(f"✅ Found {len(psv_files)} PSV files")
    
    # Convert data to Raindrop format
    print(f"\n🔄 Converting data to Raindrop format...")
    try:
        convert_physionet_to_raindrop_format(data_dir, output_dir)
        print(f"✅ Data conversion completed!")
    except Exception as e:
        print(f"❌ Error during data conversion: {e}")
        return False
    
    # Check if conversion was successful
    required_files = [
        os.path.join(output_dir, 'processed_data', 'PTdict_list.npy'),
        os.path.join(output_dir, 'processed_data', 'arr_outcomes.npy'),
        os.path.join(output_dir, 'splits', 'phy12_split1.npy')
    ]
    
    for file_path in required_files:
        if not os.path.exists(file_path):
            print(f"❌ Error: Required file '{file_path}' not found after conversion!")
            return False
    
    print(f"✅ All required files created successfully!")
    
    # Run MTGNN baseline
    print(f"\n🚀 Running MTGNN baseline...")
    
    # Create the command to run MTGNN
    cmd = [
        "python", "-m", "baselines.Raindrop.code.baselines.MTGNN_baseline",
        "--dataset", "P12",
        "--epochs", "20",
        "--batch_size", "32",
        "--splittype", "random",
        "--feature_removal_level", "no_removal",
        "--seed", "42"
    ]
    
    print(f"Command: {' '.join(cmd)}")
    
    try:
        # Run the MTGNN baseline
        result = subprocess.run(cmd, capture_output=True, text=True, cwd=os.getcwd())
        
        print(f"\n📋 MTGNN Output:")
        print(result.stdout)
        
        if result.stderr:
            print(f"\n⚠️  MTGNN Errors:")
            print(result.stderr)
        
        if result.returncode == 0:
            print(f"\n✅ MTGNN baseline completed successfully!")
        else:
            print(f"\n❌ MTGNN baseline failed with return code {result.returncode}")
            
    except Exception as e:
        print(f"❌ Error running MTGNN baseline: {e}")
        return False
    
    return True

if __name__ == "__main__":
    success = main()
    if success:
        print(f"\n🎉 All tasks completed successfully!")
    else:
        print(f"\n❌ Some tasks failed. Please check the errors above.") 