"""Setup script for Raindrop experiments on PhysioNet 2012 dataset"""

import os
import sys
from raindrop_data_converter import convert_physionet_to_raindrop_format

def setup_raindrop_experiment():
    """
    Setup script to prepare Raindrop experiments.
    This will:
    1. Convert your PhysioNet 2012 data to Raindrop format
    2. Create the necessary directory structure
    3. Provide instructions for running experiments
    """
    
    print("="*60)
    print("RAINDROP EXPERIMENT SETUP")
    print("="*60)
    
    # Check if Raindrop code exists
    if not os.path.exists("baselines/Raindrop/code"):
        print("❌ Error: Raindrop code directory not found!")
        print("Please make sure the Raindrop folder is in the current directory.")
        return False
    
    # Check if your data directory exists
    data_dir = "/home/dcm.aau.dk/km20bf/Biomarker_FeatureGroup_GNAN/tmp"
    
    if not os.path.exists(data_dir):
        print(f"❌ Error: Data directory '{data_dir}' not found!")
        return False
    
    # Check for PSV files (pipe-separated values)
    psv_files = []
    for root, dirs, files in os.walk(data_dir):
        for file in files:
            if file.endswith('.psv'):
                psv_files.append(os.path.join(root, file))
    
    if not psv_files:
        print(f"❌ Error: No PSV files found in '{data_dir}'!")
        return False
    
    print(f"✅ Found {len(psv_files)} PSV files")
    
    # Ask for output directory
    output_dir = input("Enter output directory for converted data (default: Raindrop/P12data): ").strip()
    if not output_dir:
        output_dir = "Raindrop/P12data"
    
    # Convert data
    print(f"\n🔄 Converting data to Raindrop format...")
    try:
        convert_physionet_to_raindrop_format(data_dir, output_dir)
        print(f"✅ Data conversion completed!")
    except Exception as e:
        print(f"❌ Error during data conversion: {e}")
        return False
    
    # Create experiment configuration
    config = {
        "data_path": output_dir,
        "n_splits": 5,
        "num_epochs": 20,
        "learning_rate": 0.0001,
        "batch_size": 128,
        "n_runs": 1
    }
    
    # Save configuration
    import json
    config_path = "raindrop_experiment_config.json"
    with open(config_path, 'w') as f:
        json.dump(config, f, indent=2)
    
    print(f"\n✅ Configuration saved to {config_path}")
    
    # Print instructions
    print(f"\n📋 NEXT STEPS:")
    print(f"1. Review the converted data in: {output_dir}")
    print(f"2. Run a single split experiment:")
    print(f"   python run_raindrop_p12.py")
    print(f"3. Run all splits experiment:")
    print(f"   python run_raindrop_all_splits.py")
    print(f"4. Check results in: raindrop_results/")
    
    print(f"\n🔧 CONFIGURATION:")
    print(f"   Data path: {output_dir}")
    print(f"   Number of splits: {config['n_splits']}")
    print(f"   Epochs per split: {config['num_epochs']}")
    print(f"   Learning rate: {config['learning_rate']}")
    print(f"   Batch size: {config['batch_size']}")
    
    return True

def check_dependencies():
    """Check if required dependencies are available."""
    
    print("🔍 Checking dependencies...")
    
    required_packages = [
        'torch', 'numpy', 'pandas', 'sklearn', 'tqdm'
    ]
    
    missing_packages = []
    
    for package in required_packages:
        try:
            __import__(package)
            print(f"✅ {package}")
        except ImportError:
            print(f"❌ {package} - MISSING")
            missing_packages.append(package)
    
    if missing_packages:
        print(f"\n⚠️  Missing packages: {', '.join(missing_packages)}")
        print("Please install them using:")
        print(f"pip install {' '.join(missing_packages)}")
        return False
    
    print("✅ All dependencies available!")
    return True

def main():
    """Main setup function."""
    
    print("Welcome to Raindrop Experiment Setup!")
    print("This will help you convert your PhysioNet 2012 data to Raindrop format.")
    
    # Check dependencies
    if not check_dependencies():
        return
    
    # Setup experiment
    if setup_raindrop_experiment():
        print(f"\n🎉 Setup completed successfully!")
        print(f"You can now run the Raindrop experiments.")
    else:
        print(f"\n❌ Setup failed. Please check the errors above.")

if __name__ == "__main__":
    main() 