import os

def generate_run_scripts():
    """
    Generates 15 Python scripts to run 30 hyperparameter trials
    distributed across two machines and their respective GPUs.
    """
    # --- Configuration ---

    # 1. Define the 5 config paths
    config_paths = [f'P12_configs/run{i}.yaml' for i in range(11, 16)]

    # 2. Define the 6 core hyperparameter sets
    hp_sets = [
        {'lr': 1e-3, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        {'lr': 5e-4, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        {'lr': 1e-4, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        {'lr': 5e-4, 'hidden_channels': 128, 'n_layers': 3, 'dropout': 0.3},
        {'lr': 1e-3, 'hidden_channels': 64, 'n_layers': 2, 'dropout': 0.1},
        {'lr': 5e-4, 'hidden_channels': 64, 'n_layers': 3, 'dropout': 0.2},
    ]

    hp_sets = [
        # {'lr': 1e-3, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        # {'lr': 5e-4, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        # {'lr': 1e-4, 'hidden_channels': 128, 'n_layers': 2, 'dropout': 0.2},
        # {'lr': 5e-4, 'hidden_channels': 128, 'n_layers': 3, 'dropout': 0.3},
        {'lr': 1e-3, 'hidden_channels': 32, 'n_layers': 4, 'dropout': 0.1},
        {'lr': 5e-4, 'hidden_channels': 32, 'n_layers': 4, 'dropout': 0.2},
    ]
    
    batch_size = 16
    executable_script = "run_physionet_wo_ddp_new.py"

    # --- Script Generation Logic ---

    # Create a flat list of all 30 trials.
    # The loop order is changed to ensure config_paths are distributed first.
    all_trials = []
    for hp_set in hp_sets:
        for config in config_paths:
            trial = hp_set.copy()
            trial['config_path'] = config
            all_trials.append(trial)

    # Define the machine and GPU assignments
    # 8 GPUs for machine 11, 7 for machine 12
    gpu_assignments = []
    # for gpu_id in range(8):
    #     gpu_assignments.append({'machine': 11, 'gpu': gpu_id})
    # for gpu_id in range(7):
    #     gpu_assignments.append({'machine': 12, 'gpu': gpu_id})
    for gpu_id in range(5):
        gpu_assignments.append({'machine': 100, 'gpu': gpu_id})

    # Create the output directory if it doesn't exist
    output_dir = "run_scripts"
    os.makedirs(output_dir, exist_ok=True)

    # Iterate through the 15 GPUs and assign 2 trials to each
    for i, assignment in enumerate(gpu_assignments):
        machine_id = assignment['machine']
        gpu_id = assignment['gpu']

        # Get the two trials for this specific GPU
        trial1 = all_trials[i * 2]
        trial2 = all_trials[i * 2 + 1]

        # Define the content of the script for this GPU
        script_content = f"""
import os

# This script was auto-generated to run two trials on Machine {machine_id}, GPU {gpu_id}.

print(f"--- Starting trials on Machine {machine_id}, GPU {gpu_id} ---")

# --- Trial 1 ---
print("\\nRunning Trial 1...")
config_path_1 = '{trial1['config_path']}'
n_layers_1 = {trial1['n_layers']}
hidden_channels_1 = {trial1['hidden_channels']}
lr_1 = {trial1['lr']}
dropout_1 = {trial1['dropout']}
batch_size_1 = {batch_size}

command1 = (
    f"CUDA_VISIBLE_DEVICES={gpu_id} python {executable_script} "
    f"--config_path {{config_path_1}} --n_layers {{n_layers_1}} "
    f"--hidden_channels {{hidden_channels_1}} --lr {{lr_1}} "
    f"--batch_size {{batch_size_1}} --dropout {{dropout_1}}"
)
print(f"Executing: {{command1}}\\n")
os.system(command1)
print("\\nTrial 1 finished.")

print("\\n" + "="*60 + "\\n")

# --- Trial 2 ---
print("Running Trial 2...")
config_path_2 = '{trial2['config_path']}'
n_layers_2 = {trial2['n_layers']}
hidden_channels_2 = {trial2['hidden_channels']}
lr_2 = {trial2['lr']}
dropout_2 = {trial2['dropout']}
batch_size_2 = {batch_size}

command2 = (
    f"CUDA_VISIBLE_DEVICES={gpu_id} python {executable_script} "
    f"--config_path {{config_path_2}} --n_layers {{n_layers_2}} "
    f"--hidden_channels {{hidden_channels_2}} --lr {{lr_2}} "
    f"--batch_size {{batch_size_2}} --dropout {{dropout_2}}"
)
print(f"Executing: {{command2}}\\n")
os.system(command2)
print("\\nTrial 2 finished.")

print(f"\\n--- All trials on Machine {machine_id}, GPU {gpu_id} are complete. ---")
"""

        # Write the content to a new Python file
        file_name = f"0708_run_machine{machine_id}_gpu{gpu_id}.py"
        file_path = os.path.join(output_dir, file_name)

        with open(file_path, "w") as f:
            f.write(script_content.strip())

    print(f"Success! Generated {len(gpu_assignments)} run scripts in the '{output_dir}' directory.")
    print("You can now copy the relevant scripts to each machine and execute them.")

if __name__ == '__main__':
    generate_run_scripts()