#!/bin/bash
# 简化版配准脚本 - 基于固定模式匹配mask文件

# --- 配置 FSL 环境 ---
export FSLDIR=/root/fsl  # 修改为您的实际 FSL 安装路径
source ${FSLDIR}/etc/fslconf/fsl.sh
export PATH=${FSLDIR}/bin:$PATH

# --- 配置参数 ---
MNI_TEMPLATE="${FSLDIR}/data/standard/MNI152_T1_1mm_brain.nii.gz"
INPUT_ROOT="/data/qiuhui/data/adni/adni_with_seg"
OUTPUT_ROOT="/data/qiuhui/data/adni/adni_registered"

# --- 创建输出目录结构 ---
mkdir -p "$OUTPUT_ROOT"/{brain_extracted,registered_images,registered_labels}

# --- 设置并行度 ---
PARALLEL_JOBS=$(nproc)
echo "将使用 $PARALLEL_JOBS 个并行任务"

# --- 创建临时 FSL 环境包装器 ---
create_fsl_wrapper() {
    local wrapper_path="$1"
    cat > "$wrapper_path" << 'EOF'
#!/bin/bash
# FSL 环境包装器
export FSLDIR=/root/fsl
source ${FSLDIR}/etc/fslconf/fsl.sh
export PATH=${FSLDIR}/bin:$PATH
exec "$@"
EOF
    chmod +x "$wrapper_path"
}

# 创建包装器脚本
WRAPPER_SCRIPT="/tmp/fsl_wrapper_$$.sh"
create_fsl_wrapper "$WRAPPER_SCRIPT"
echo "已创建 FSL 环境包装器: $WRAPPER_SCRIPT"

# --- 使用 find+xargs 进行高效并行处理 ---
echo "开始搜索和处理图像文件..."

find "$INPUT_ROOT" -type f -name "*.nii" ! -name "MALPEM-*" -print0 | \
xargs -0 -P $PARALLEL_JOBS -I {} bash -c '
    # 内联处理脚本
    img_path="$1"
    wrapper="$2"
    mni_template="$3"
    output_root="$4"
    
    # 从路径中提取信息
    subject=$(dirname "$(dirname "$img_path")" | xargs basename)
    month=$(dirname "$img_path" | xargs basename)
    base_name=$(basename "$img_path" .nii)
    
    # 获取图像目录
    img_dir=$(dirname "$img_path")
    
    # 构建 mask 路径 - 简化的匹配逻辑
    # 1. 提取序列号部分 (如 S77189_I165289)
    series_id=$(echo "$base_name" | grep -oE "S[0-9]+_I[0-9]+")
    
    # 2. 搜索匹配的 mask 文件
    mask_path=$(find "$img_dir" -maxdepth 1 -name "MALPEM-ADNI*${series_id}*_6targets.nii.gz" -print -quit)
    
    # 验证 mask 是否存在
    if [ -z "$mask_path" ] || [ ! -f "$mask_path" ]; then
        echo "警告：找不到对应的 mask 文件，跳过处理 $img_path"
        echo "搜索模式: MALPEM-ADNI*${series_id}*_6targets.nii.gz"
        exit 1
    fi
    
    # 创建输出子目录
    out_subdir="$output_root/brain_extracted/$subject/$month"
    mkdir -p "$out_subdir"
    out_img_dir="$output_root/registered_images/$subject/$month"
    mkdir -p "$out_img_dir"
    out_label_dir="$output_root/registered_labels/$subject/$month"
    mkdir -p "$out_label_dir"
    
    # 定义输出文件路径
    brain_img="$out_subdir/${base_name}_brain.nii.gz"
    reg_img="$out_img_dir/${base_name}_MNI.nii.gz"
    reg_mask="$out_label_dir/${base_name}_MNI_mask.nii.gz"
    affine="$out_img_dir/${base_name}_affine.mat"
    
    # Step 1: 使用 BET 去颅骨
    $wrapper bet "$img_path" "$brain_img" -f 0.5 -g 0
    
    # 检查 BET 是否成功
    if [ ! -f "$brain_img" ]; then
        echo "错误：大脑提取失败 - $img_path"
        exit 1
    fi
    
    # Step 2: 配准到 MNI 空间
    $wrapper flirt -in "$brain_img" \
          -ref "$mni_template" \
          -out "$reg_img" \
          -omat "$affine" \
          -dof 12 \
          -interp spline
    
    # Step 3: 应用变换到 mask
    $wrapper flirt -in "$mask_path" \
          -ref "$mni_template" \
          -out "$reg_mask" \
          -init "$affine" \
          -applyxfm \
          -interp nearestneighbour
    
    echo "成功: $subject/$month/$base_name → mask: $(basename "$mask_path")"
' _ "{}" "$WRAPPER_SCRIPT" "$MNI_TEMPLATE" "$OUTPUT_ROOT"

# 清理包装器脚本
rm -f "$WRAPPER_SCRIPT"
echo "已删除临时包装器脚本"

echo "全部任务完成！结果保存在: $OUTPUT_ROOT"