import torch.nn.functional as F
import torch.optim as optim
from k_level_policy_gradients.src.algorithms.actor_critic.ddpg import DDPG
from k_level_policy_gradients.src.policy.noise_policy import GaussianPolicy
from k_level_policy_gradients.src.networks.ac_networks import (
    ContinuousActorNetwork,
    ContinuousCriticNetwork,
)
from k_level_policy_gradients.src.utils.replay_memory import ReplayMemoryObs


def setup_ddpg_agent(mdp_info, idx_agent, agent_params, **kwargs):
    """
    Instantiates a multiplayer DDPG Discrete agent.
    """
    pi_sigma = agent_params["pi_sigma"]
    batch_size = agent_params["batch_size"]
    target_update_frequency = agent_params["target_update_frequency"]
    tau = agent_params["tau"]
    warmup_replay_size = agent_params["warmup_replay_size"]
    max_replay_size = agent_params["max_replay_size"]
    n_features_actor = agent_params["n_features_actor"]
    if type(n_features_actor) is str:
        n_features_actor = eval(n_features_actor)
    lr_actor = float(agent_params["lr_actor"])
    n_features_critic = agent_params["n_features_critic"]
    if type(n_features_critic) is str:
        n_features_critic = eval(n_features_critic)
    lr_critic = float(agent_params["lr_critic"])
    centralized_critic = agent_params["centralized_critic"]
    grad_norm_clip = agent_params["grad_norm_clip"]
    share_agent_params = agent_params["share_agent_params"]
    use_mixer = agent_params.get("use_mixer", False)

    pi = GaussianPolicy(sigma=pi_sigma, action_space=mdp_info.action_space[idx_agent])

    actor_input_dim = mdp_info.observation_space[idx_agent].shape[0]
    critic_input_dim = mdp_info.observation_space[idx_agent].shape[0]
    if share_agent_params:
        actor_input_dim += mdp_info.n_agents
        critic_input_dim += mdp_info.n_agents
    if centralized_critic:
        for i in range(mdp_info.n_agents):
            critic_input_dim += mdp_info.action_space[i].shape[0]
    else:
        critic_input_dim += mdp_info.action_space[idx_agent].shape[0]
    actor_input_shape = (actor_input_dim,)
    critic_input_shape = (critic_input_dim,)

    actor_params = dict(
        input_shape=actor_input_shape,
        output_shape=mdp_info.action_space[idx_agent].shape,
        network=ContinuousActorNetwork,
        optimizer={
            "class": optim.Adam,
            "params": {"lr": lr_actor},
        },
        n_features=n_features_actor,
        use_cuda=agent_params["use_cuda"],
    )

    critic_params = dict(
        input_shape=critic_input_shape,
        output_shape=(1,),
        network=ContinuousCriticNetwork,
        optimizer={"class": optim.Adam, "params": {"lr": lr_critic}},
        loss=F.mse_loss,
        n_features=n_features_critic,
        use_cuda=agent_params["use_cuda"],
    )

    replay_memory = ReplayMemoryObs(
        max_replay_size,
        mdp_info.state_space.shape[0],
        actor_input_shape[0],
        mdp_info.action_space[idx_agent].shape[0],
        discrete_actions=False,
    )

    if share_agent_params and idx_agent != 0:
        primary_agent = kwargs["primary_agent"]
    else:
        primary_agent = None

    agent = DDPG(
        mdp_info=mdp_info,
        idx_agent=idx_agent,
        policy=pi,
        actor_params=actor_params,
        critic_params=critic_params,
        centralized_critic=centralized_critic,
        batch_size=batch_size,
        target_update_frequency=target_update_frequency,
        tau=tau,
        warmup_replay_size=warmup_replay_size,
        replay_memory=replay_memory,
        grad_norm_clip=grad_norm_clip,
        use_cuda=agent_params["use_cuda"],
        primary_agent=primary_agent,
        use_mixer=use_mixer,
    )

    return agent
