import numpy as np
import cv2
import matplotlib.pyplot as plt 
from skimage.metrics import structural_similarity as ssim
import glob
from utils import compute_iou, compute_mse, compute_ssim, compute_feature_match_score


if __name__ == "__main__":
    print("Loading outputs from NumPy arrays and ground truth from image...")

    # Load NumPy outputs
    output_files = sorted(glob.glob("output_*.npy"))
    outputs = [np.load(file) for file in output_files]

    # Load ground truth image (grayscale)
    groundtruth = cv2.imread("groundtruth.png", cv2.IMREAD_GRAYSCALE)

    # Show all output arrays
    for i, output in enumerate(outputs):
        plt.imshow(output, cmap='gray', vmin=0, vmax=255)
        plt.title(f"Output {i}")
        plt.axis("off")
        plt.show()

    # Define metrics
    metrics = {
        "IoU": compute_iou,
        "Mean Squared Error": compute_mse,
        "Structural Similarity Index": compute_ssim,
        "Feature Match Score": compute_feature_match_score,
    }

    # Compare each output to the ground truth
    for i, output in enumerate(outputs):
        print(f"\nMetrics for output_{i} vs groundtruth:")
        for metric_name, metric_func in metrics.items():
            print(f"{metric_name}: {metric_func(output, groundtruth)}")

    # Optional: Ground truth vs itself (sanity check)
    print(f"\nMetrics for groundtruth vs groundtruth:")
    for metric_name, metric_func in metrics.items():
        print(f"{metric_name}: {metric_func(groundtruth, groundtruth)}")