#!/usr/bin/env python3
import os, glob, time
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.data import DataLoader
import matplotlib.pyplot as plt
from pathlib import Path
from PIL import Image

# Install this first: pip install pytorch-msssim
from pytorch_msssim import ssim, ms_ssim

# Updated orthonormal matrix generation for 16x16 patch processing (no subsampling)
def make_gaussian_random_orthonormal_rows(h=256, w=256, seed=42):
    """
    Generate a matrix A of size [h, w] where rows are orthonormal.
    Note: Requires h <= w for orthonormal rows to exist.
    For 16x16 patches, we use 256x256 matrix (same as first code).
    """
    if seed is not None:
        torch.manual_seed(seed)
    # Step 1: Random Gaussian matrix
    A = torch.randn(h, w)
    # Step 2: QR decomposition on transpose to orthonormalize rows
    # A^T = Q*R → A = R^T * Q^T
    # We want orthonormal rows, so we work with A^T first
    Q, R = torch.linalg.qr(A.T)  # A.T is [w, h]
    # Q is [w, h] with orthonormal columns
    # Q.T is [h, w] with orthonormal rows
    return Q.T  # [h, w] with orthonormal rows

class PatchwiseOrthonormalDataset:
    """
    Dataset that applies patch-wise orthonormal transformation to images.
    Each 16x16 patch gets transformed through a 256x256 orthonormal matrix.
    NO SUBSAMPLING - keeps all 256 dimensions after transformation.
    Output: 224x224 masked input → 224x224 original target (same resolution inverse problem)
    """
    def __init__(self, data_dir, seed=42, verbose=False):
        self.data_dir = data_dir
        
        # Generate fixed orthonormal matrix A for 16x16 patch-wise transformation
        self.A = make_gaussian_random_orthonormal_rows(h=256, w=256, seed=seed)
        
        # Get all image files
        self.data_path = Path(data_dir)
        if not self.data_path.exists():
            raise FileNotFoundError(f"Data directory not found: {data_dir}")
        
        image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.tiff', '.JPEG', '.JPG'}
        self.image_files = [f for f in self.data_path.iterdir() 
                           if f.is_file() and f.suffix in image_extensions]
        
        if len(self.image_files) == 0:
            raise ValueError(f"No images found in {data_dir}")
        
        if verbose:
            print(f"Loaded {len(self.image_files)} images from {data_dir}")
            print(f"Using 16x16 patch-wise orthonormal transformation with matrix shape: {self.A.shape}")
            print(f"Output: 224x224 masked → 224x224 original (inverse problem, NO subsampling)")

    def __len__(self):
        return len(self.image_files)

    def resize_min_side(self, img, min_side=224):
        w, h = img.size
        s = min_side / min(w, h)
        return img.resize((int(round(w*s)), int(round(h*s))), Image.Resampling.LANCZOS)

    def center_crop(self, img, size=224):
        w, h = img.size
        left = (w - size) // 2
        top = (h - size) // 2
        return img.crop((left, top, left + size, top + size))

    def preprocess_image(self, img):
        img = img.convert("RGB")
        img_resized = self.resize_min_side(img, 224)
        img_crop = self.center_crop(img_resized, 224)
        x = np.array(img_crop).astype(np.float32) / 255.0
        return x

    def process_image_with_orthonormal_masks(self, np_img, mask_matrix):
        """
        Apply orthonormal transformation to 16x16 patches of a 224x224 image.
        
        Args:
            np_img: numpy array of shape [224, 224, 3]
            mask_matrix: torch tensor of shape [256, 256] with orthonormal rows
        
        Returns:
            transformed_patches: torch tensor of shape [14, 14, 256] (NO subsampling)
        """
        # Convert to torch and extract patches
        img_tensor = torch.from_numpy(np_img).float()
        
        # Convert to grayscale for 256 = 16*16
        if img_tensor.shape[2] == 3:
            img_gray = img_tensor.mean(dim=2)  # Convert to grayscale
        else:
            img_gray = img_tensor
        
        # Extract 16x16 patches from 224x224 image (14x14 patches total)
        patches = img_gray.unfold(0, 16, 16).unfold(1, 16, 16)  # [14, 14, 16, 16]
        
        # Flatten each patch and apply transformation
        transformed_patches = torch.zeros(14, 14, 256)  # Keep all 256 dimensions
        
        for i in range(14):
            for j in range(14):
                # Flatten 16x16 patch to 256x1
                patch_flat = patches[i, j].flatten()  # [256]
                
                # Apply orthonormal transformation: [256, 256] @ [256] → [256]
                transformed = mask_matrix @ patch_flat
                
                # NO SUBSAMPLING - keep all 256 dimensions
                transformed_patches[i, j] = transformed
        
        return transformed_patches

    def reconstruct_masked_image(self, transformed_patches):
        """
        Reconstruct 224x224 image from 14x14x256 transformed patches.
        
        Args:
            transformed_patches: torch tensor of shape [14, 14, 256]
        
        Returns:
            masked_image: torch tensor of shape [224, 224]
        """
        masked_image = torch.zeros(224, 224)
        
        for i in range(14):
            for j in range(14):
                # Get the 256-dimensional transformed patch
                transformed_patch = transformed_patches[i, j]  # [256]
                
                # Reshape back to 16x16
                patch_16x16 = transformed_patch.reshape(16, 16)
                
                # Place in the correct position in the 224x224 image
                start_h = i * 16
                end_h = start_h + 16
                start_w = j * 16
                end_w = start_w + 16
                
                masked_image[start_h:end_h, start_w:end_w] = patch_16x16
        
        return masked_image

    def apply_patchwise_orthonormal_transform(self, x):
        """
        Apply patch-wise orthonormal transformation to RGB image.
        
        Args:
            x: numpy array of shape [224, 224, 3] (original image)
        
        Returns:
            y: numpy array of shape [224, 224, 3] (transformed image - same resolution)
        """
        y_channels = []
        
        # Apply transformation to each channel separately
        for c in range(3):
            # Create single-channel image for processing
            single_channel = x[..., c]
            
            # Apply patch-wise orthonormal transformation
            transformed_patches = self.process_image_with_orthonormal_masks(
                np.expand_dims(single_channel, axis=2), self.A
            )
            
            # Reconstruct to 224x224
            masked_channel = self.reconstruct_masked_image(transformed_patches)
            
            # Store result
            y_channels.append(masked_channel.numpy())
        
        # Stack channels to create [224, 224, 3]
        y = np.stack(y_channels, axis=2)
        
        # Normalize to [0, 1] range
        y_min = y.min()
        y_max = y.max()
        y_norm = (y - y_min) / (y_max - y_min + 1e-8)
        
        return y_norm

    def __getitem__(self, idx):
        img_path = self.image_files[idx]
        
        try:
            img = Image.open(img_path)
        except Exception as e:
            print(f"Warning: Could not load image {img_path}: {e}")
            img = Image.new('RGB', (224, 224), color=(0, 0, 0))
        
        # Preprocess to get x (target - original 224x224 image)
        x = self.preprocess_image(img)  # (224, 224, 3)
        
        # Apply patch-wise orthonormal transform to get y (input - 224x224 masked image)
        y = self.apply_patchwise_orthonormal_transform(x)  # (224, 224, 3)
        
        # Convert to torch tensors and change to CHW format
        x_tensor = torch.from_numpy(x).permute(2, 0, 1)  # (3, 224, 224) - target
        y_tensor = torch.from_numpy(y).permute(2, 0, 1)  # (3, 224, 224) - input
        
        return y_tensor, x_tensor  # (input 224x224, target 224x224)

# Traditional U-Net Components
class DoubleConv(nn.Module):
    """Double convolution block used in U-Net"""
    def __init__(self, in_channels, out_channels, mid_channels=None):
        super().__init__()
        if not mid_channels:
            mid_channels = out_channels
        self.double_conv = nn.Sequential(
            nn.Conv2d(in_channels, mid_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(mid_channels),
            nn.ReLU(inplace=True),
            nn.Conv2d(mid_channels, out_channels, kernel_size=3, padding=1, bias=False),
            nn.BatchNorm2d(out_channels),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        return self.double_conv(x)

class Down(nn.Module):
    """Downscaling with maxpool then double conv"""
    def __init__(self, in_channels, out_channels):
        super().__init__()
        self.maxpool_conv = nn.Sequential(
            nn.MaxPool2d(2),
            DoubleConv(in_channels, out_channels)
        )

    def forward(self, x):
        return self.maxpool_conv(x)

class Up(nn.Module):
    """Upscaling then double conv"""
    def __init__(self, in_channels, out_channels, bilinear=True):
        super().__init__()

        # if bilinear, use the normal convolutions to reduce the number of channels
        if bilinear:
            self.up = nn.Upsample(scale_factor=2, mode='bilinear', align_corners=True)
            self.conv = DoubleConv(in_channels, out_channels, in_channels // 2)
        else:
            self.up = nn.ConvTranspose2d(in_channels, in_channels // 2, kernel_size=2, stride=2)
            self.conv = DoubleConv(in_channels, out_channels)

    def forward(self, x1, x2):
        x1 = self.up(x1)
        # input is CHW
        diffY = x2.size()[2] - x1.size()[2]
        diffX = x2.size()[3] - x1.size()[3]

        x1 = F.pad(x1, [diffX // 2, diffX - diffX // 2,
                       diffY // 2, diffY - diffY // 2])
        
        x = torch.cat([x2, x1], dim=1)
        return self.conv(x)

class OutConv(nn.Module):
    """Final output convolution"""
    def __init__(self, in_channels, out_channels):
        super(OutConv, self).__init__()
        self.conv = nn.Conv2d(in_channels, out_channels, kernel_size=1)

    def forward(self, x):
        return self.conv(x)

# Traditional U-Net Model for Inverse Problem (No Subsampling Version)
class UNetForInverseProblem(nn.Module):
    """
    Traditional U-Net architecture for orthonormal inverse problem.
    NO subsampling version - same as first code task formulation
    
    Input: 224x224x3 (masked/transformed image)  
    Output: 224x224x3 (reconstructed original image)
    """
    def __init__(self, n_channels=3, n_classes=3, bilinear=False):
        super(UNetForInverseProblem, self).__init__()
        self.n_channels = n_channels
        self.n_classes = n_classes
        self.bilinear = bilinear

        # U-Net encoder path (no input upsampling needed since input is 224x224)
        self.inc = DoubleConv(n_channels, 64)
        self.down1 = Down(64, 128)
        self.down2 = Down(128, 256)
        self.down3 = Down(256, 512)
        factor = 2 if bilinear else 1
        self.down4 = Down(512, 1024 // factor)
        
        # U-Net decoder path
        self.up1 = Up(1024, 512 // factor, bilinear)
        self.up2 = Up(512, 256 // factor, bilinear)
        self.up3 = Up(256, 128 // factor, bilinear)
        self.up4 = Up(128, 64, bilinear)
        
        # Output layer
        self.outc = OutConv(64, n_classes)
        
        # Final activation
        self.final_activation = nn.Sigmoid()

        # Skip connection from input (for residual learning)
        self.skip_connection = nn.Sequential(
            nn.Conv2d(n_channels, 32, kernel_size=3, padding=1),
            nn.BatchNorm2d(32),
            nn.ReLU(inplace=True),
            nn.Conv2d(32, n_classes, kernel_size=3, padding=1),
            nn.Tanh()  # Allow both positive and negative contributions
        )

        print("Traditional U-Net Architecture (No Subsampling):")
        print(f"   Input: {n_channels} channels (224x224)")
        print(f"   Output: {n_classes} channels (224x224)")
        print(f"   Bilinear upsampling: {bilinear}")
        print(f"   Task: Same resolution inverse problem (224x224 → 224x224)")
        print(f"   Skip connection: Input → Output")

    def forward(self, x):
        # x is [B, 3, 224, 224] - already correct size
        original_input = x
        
        # Create skip connection from original input
        skip_features = self.skip_connection(x)  # [B, 3, 224, 224]
        
        # U-Net encoder
        x1 = self.inc(x)      # [B, 64, 224, 224]
        x2 = self.down1(x1)   # [B, 128, 112, 112]  
        x3 = self.down2(x2)   # [B, 256, 56, 56]
        x4 = self.down3(x3)   # [B, 512, 28, 28]
        x5 = self.down4(x4)   # [B, 1024, 14, 14] or [B, 512, 14, 14] if bilinear
        
        # U-Net decoder with skip connections
        x = self.up1(x5, x4)  # [B, 512, 28, 28]
        x = self.up2(x, x3)   # [B, 256, 56, 56] 
        x = self.up3(x, x2)   # [B, 128, 112, 112]
        x = self.up4(x, x1)   # [B, 64, 224, 224]
        
        # Output layer
        logits = self.outc(x)  # [B, 3, 224, 224]
        
        # Combine with skip connection
        output = logits + skip_features  # [B, 3, 224, 224]
        
        # Final activation
        output = self.final_activation(output)  # [B, 3, 224, 224] in [0, 1]
        
        return output

# Combined Loss Function (same as before)
class CombinedLoss(nn.Module):
    """
    Combined loss function with MSE, L1, and SSIM terms.
    """
    def __init__(self, mse_weight=1.0, l1_weight=1.0, ssim_weight=1.0, use_ms_ssim=False):
        super().__init__()
        self.mse_weight = mse_weight
        self.l1_weight = l1_weight
        self.ssim_weight = ssim_weight
        self.use_ms_ssim = use_ms_ssim
        
        # MSE and L1 losses
        self.mse_loss = nn.MSELoss()
        self.l1_loss = nn.L1Loss()
        
        print(f"Combined Loss - MSE: {mse_weight}, L1: {l1_weight}, SSIM: {ssim_weight}")
        if use_ms_ssim:
            print("Using Multi-Scale SSIM")
        else:
            print("Using Standard SSIM")
    
    def forward(self, pred, target):
        # Ensure inputs are in [0, 1] range for SSIM
        pred_clamped = torch.clamp(pred, 0, 1)
        target_clamped = torch.clamp(target, 0, 1)
        
        # Calculate individual losses
        mse = self.mse_loss(pred, target)
        l1 = self.l1_loss(pred, target)
        
        # Calculate SSIM loss (1 - SSIM since we want to minimize)
        if self.use_ms_ssim:
            ssim_val = ms_ssim(pred_clamped, target_clamped, data_range=1.0, size_average=True)
        else:
            ssim_val = ssim(pred_clamped, target_clamped, data_range=1.0, size_average=True)
        
        ssim_loss = 1 - ssim_val
        
        # Combined loss
        total_loss = (self.mse_weight * mse + 
                     self.l1_weight * l1 + 
                     self.ssim_weight * ssim_loss)
        
        return {
            'total_loss': total_loss,
            'mse_loss': mse,
            'l1_loss': l1,
            'ssim_loss': ssim_loss,
            'ssim_value': ssim_val
        }

# Perceptual Loss (Optional)
class PerceptualLoss(nn.Module):
    """
    Optional: Perceptual loss using VGG features for better reconstruction quality.
    """
    def __init__(self, feature_layers=[3, 8, 15, 22]):
        super().__init__()
        # Load pre-trained VGG16
        vgg = torch.hub.load('pytorch/vision:v0.10.0', 'vgg16', pretrained=True)
        self.features = vgg.features
        self.feature_layers = feature_layers
        
        # Freeze VGG parameters
        for param in self.features.parameters():
            param.requires_grad = False
            
        # Normalization for ImageNet pre-trained model
        self.register_buffer('mean', torch.tensor([0.485, 0.456, 0.406]).view(1, 3, 1, 1))
        self.register_buffer('std', torch.tensor([0.229, 0.224, 0.225]).view(1, 3, 1, 1))
    
    def forward(self, pred, target):
        # Normalize inputs
        pred_norm = (pred - self.mean) / self.std
        target_norm = (target - self.mean) / self.std
        
        # Extract features
        pred_features = self.extract_features(pred_norm)
        target_features = self.extract_features(target_norm)
        
        # Calculate perceptual loss
        loss = 0
        for pred_feat, target_feat in zip(pred_features, target_features):
            loss += F.mse_loss(pred_feat, target_feat)
        
        return loss / len(pred_features)
    
    def extract_features(self, x):
        features = []
        for i, layer in enumerate(self.features):
            x = layer(x)
            if i in self.feature_layers:
                features.append(x)
        return features

# Advanced Combined Loss with optional perceptual term
class AdvancedCombinedLoss(nn.Module):
    """
    Advanced loss combining MSE, L1, SSIM, and optional Perceptual loss.
    """
    def __init__(self, mse_weight=1.0, l1_weight=1.0, ssim_weight=1.0, 
                 perceptual_weight=0.0, use_ms_ssim=False):
        super().__init__()
        self.mse_weight = mse_weight
        self.l1_weight = l1_weight
        self.ssim_weight = ssim_weight
        self.perceptual_weight = perceptual_weight
        self.use_ms_ssim = use_ms_ssim
        
        # Basic losses
        self.mse_loss = nn.MSELoss()
        self.l1_loss = nn.L1Loss()
        
        # Perceptual loss (optional)
        if perceptual_weight > 0:
            self.perceptual_loss = PerceptualLoss()
            print(f"Including Perceptual Loss with weight: {perceptual_weight}")
        
        print(f"Advanced Combined Loss - MSE: {mse_weight}, L1: {l1_weight}, SSIM: {ssim_weight}")
    
    def forward(self, pred, target):
        # Ensure inputs are in [0, 1] range for SSIM
        pred_clamped = torch.clamp(pred, 0, 1)
        target_clamped = torch.clamp(target, 0, 1)
        
        # Calculate individual losses
        mse = self.mse_loss(pred, target)
        l1 = self.l1_loss(pred, target)
        
        # SSIM loss
        if self.use_ms_ssim:
            ssim_val = ms_ssim(pred_clamped, target_clamped, data_range=1.0, size_average=True)
        else:
            ssim_val = ssim(pred_clamped, target_clamped, data_range=1.0, size_average=True)
        
        ssim_loss = 1 - ssim_val
        
        # Perceptual loss (if enabled)
        perceptual = torch.tensor(0.0, device=pred.device)
        if self.perceptual_weight > 0:
            perceptual = self.perceptual_loss(pred_clamped, target_clamped)
        
        # Combined loss
        total_loss = (self.mse_weight * mse + 
                     self.l1_weight * l1 + 
                     self.ssim_weight * ssim_loss +
                     self.perceptual_weight * perceptual)
        
        return {
            'total_loss': total_loss,
            'mse_loss': mse,
            'l1_loss': l1,
            'ssim_loss': ssim_loss,
            'ssim_value': ssim_val,
            'perceptual_loss': perceptual
        }

# Loss Configuration Presets
def get_loss_configs():
    """
    Different loss configurations for different training strategies.
    """
    configs = {
        # Basic combination - good starting point
        'basic': {
            'mse_weight': 1.0,
            'l1_weight': 1.0, 
            'ssim_weight': 1.0,
            'use_ms_ssim': False
        },
        
        # SSIM-focused - better perceptual quality
        'ssim_focused': {
            'mse_weight': 0.5,
            'l1_weight': 1.0,
            'ssim_weight': 2.0,
            'use_ms_ssim': True
        },
        
        # L1-focused - sharper edges
        'sharp_edges': {
            'mse_weight': 0.5,
            'l1_weight': 2.0,
            'ssim_weight': 1.0,
            'use_ms_ssim': False
        },
        
        # Balanced with perceptual
        'perceptual': {
            'mse_weight': 1.0,
            'l1_weight': 1.0,
            'ssim_weight': 1.0,
            'perceptual_weight': 0.1,
            'use_ms_ssim': True
        }
    }
    return configs

# Training function
def run_epoch(loader, model, optim, device, criterion, train=True, visualize_every=0, vis_dir=None):
    if train:
        model.train()
    else:
        model.eval()

    # Track individual loss components
    total_loss = 0.0
    total_mse = 0.0
    total_l1 = 0.0
    total_ssim_loss = 0.0
    total_ssim_value = 0.0
    total_perceptual = 0.0
    num_batches = 0

    with torch.set_grad_enabled(train):
        for i, (y_batch, x_batch) in enumerate(loader):
            # y_batch: transformed images (input), x_batch: original images (target)
            y_batch = y_batch.to(device, non_blocking=True)
            x_batch = x_batch.to(device, non_blocking=True)

            # Forward pass: predict x from y
            pred_x = model(y_batch)
        
            # Ensure shapes match
            if pred_x.shape != x_batch.shape:
                pred_x = F.interpolate(pred_x, size=x_batch.shape[-2:], mode='bilinear', align_corners=True)

            # Compute combined loss
            loss_dict = criterion(pred_x, x_batch)
            loss = loss_dict['total_loss']
        
            if train:
                optim.zero_grad(set_to_none=True)
                loss.backward()
                optim.step()

            # Accumulate losses for monitoring
            total_loss += loss.item()
            total_mse += loss_dict['mse_loss'].item()
            total_l1 += loss_dict['l1_loss'].item()
            total_ssim_loss += loss_dict['ssim_loss'].item()
            total_ssim_value += loss_dict['ssim_value'].item()
            if 'perceptual_loss' in loss_dict:
                total_perceptual += loss_dict['perceptual_loss'].item()
            num_batches += 1

            # Visualization during validation
            if (not train) and visualize_every and (i % visualize_every == 0) and vis_dir:
                os.makedirs(vis_dir, exist_ok=True)
            
                # Convert to numpy for visualization
                y_vis = y_batch[0].detach().cpu().numpy().transpose(1, 2, 0)  # CHW -> HWC
                pred_vis = pred_x[0].detach().cpu().numpy().transpose(1, 2, 0)
                x_vis = x_batch[0].detach().cpu().numpy().transpose(1, 2, 0)
            
                fig, ax = plt.subplots(1, 3, figsize=(15, 5))
                ax[0].imshow(np.clip(y_vis, 0, 1))
                ax[0].set_title("Input (224x224 Masked)")
                ax[0].axis('off')
            
                ax[1].imshow(np.clip(pred_vis, 0, 1))
                ax[1].set_title(f"Predicted (224x224)\nSSIM: {loss_dict['ssim_value'].item():.4f}")
                ax[1].axis('off')
            
                ax[2].imshow(np.clip(x_vis, 0, 1))
                ax[2].set_title("Target (224x224 Original)")
                ax[2].axis('off')
            
                plt.tight_layout()
                plt.savefig(os.path.join(vis_dir, f"val_{i:05d}.png"), dpi=120, bbox_inches='tight')
                plt.close()

    # Return averaged losses
    avg_losses = {
        'total_loss': total_loss / max(1, num_batches),
        'mse_loss': total_mse / max(1, num_batches),
        'l1_loss': total_l1 / max(1, num_batches),
        'ssim_loss': total_ssim_loss / max(1, num_batches),
        'ssim_value': total_ssim_value / max(1, num_batches),  # Higher is better
        'perceptual_loss': total_perceptual / max(1, num_batches)
    }
    return avg_losses

def main():
    args = {
        # Data paths (update these to your paths)
        'train_dir': r"F:\imgnet\data\train",
        'val_dir': r"F:\imgnet\data\val",
        'save_dir': "./unet_no_subsample_checkpoints",
        'load_path': "",  # Path to checkpoint to resume from (optional)
    
        # Model settings
        'target_size': (224, 224),  # Output image size
        'seed': 42,  # Fixed seed for orthonormal matrix A
        'bilinear': True,  # Use bilinear upsampling in U-Net (faster, less memory)
    
        # Training settings
        'batch_size': 32,  # Can be higher for U-Net compared to ViT models
        'lr': 1e-4,
        'epochs': 100,
        'save_every': 10,
        'viz_every': 50,  # Visualize every N validation batches
        
        # Loss settings - choose one of the configurations
        'loss_config': 'basic',  # 'basic', 'ssim_focused', 'sharp_edges', 'perceptual'
    }

    # Set device
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    print(f"Using device: {device}")

    # Create save directory
    os.makedirs(args['save_dir'], exist_ok=True)

    # Save args for reference
    import json
    with open(os.path.join(args['save_dir'], 'training_args.json'), 'w') as f:
        json.dump(args, f, indent=2)

    # Create datasets
    train_ds = PatchwiseOrthonormalDataset(
        data_dir=args['train_dir'],
        seed=args['seed'],
        verbose=True
    )

    val_ds = PatchwiseOrthonormalDataset(
        data_dir=args['val_dir'],
        seed=args['seed'],  # Same seed for consistent A matrix
        verbose=True
    )

    # Create dataloaders
    train_loader = DataLoader(
        train_ds, batch_size=args['batch_size'], shuffle=True,
        num_workers=4, pin_memory=True, drop_last=True
    )
    val_loader = DataLoader(
        val_ds, batch_size=args['batch_size'], shuffle=False,
        num_workers=4, pin_memory=True, drop_last=False
    )

    # Create model - Traditional U-Net
    model = UNetForInverseProblem(n_channels=3, n_classes=3, bilinear=args['bilinear']).to(device)

    # Print model info
    total_params = sum(p.numel() for p in model.parameters())
    trainable_params = sum(p.numel() for p in model.parameters() if p.requires_grad)
    print(f"Model Parameters: {total_params:,} total, {trainable_params:,} trainable")

    # Load checkpoint if provided
    if args['load_path'] and os.path.isfile(args['load_path']):
        ckpt = torch.load(args['load_path'], map_location=device)
        state_dict = ckpt['model_state_dict'] if 'model_state_dict' in ckpt else ckpt
        model.load_state_dict(state_dict, strict=False)
        print(f"Loaded checkpoint: {args['load_path']}")

    # Create combined loss function
    loss_configs = get_loss_configs()
    config = loss_configs[args['loss_config']]
    
    # Choose between basic combined loss or advanced with perceptual
    if 'perceptual_weight' in config:
        criterion = AdvancedCombinedLoss(**config)
    else:
        criterion = CombinedLoss(**config)

    # Optimizer and scheduler
    optimizer = torch.optim.AdamW(model.parameters(), lr=args['lr'], weight_decay=1e-4)
    scheduler = torch.optim.lr_scheduler.ReduceLROnPlateau(
        optimizer, mode='min', factor=0.5, patience=5, verbose=True
    )

    print(f"Starting Traditional U-Net training for orthonormal inverse problem (No Subsampling)")
    print(f"Train: {len(train_ds)} images, Val: {len(val_ds)} images")
    print(f"Batch size: {args['batch_size']}")
    print(f"Task: 224x224 masked input → 224x224 original target (same resolution)")
    print(f"Loss configuration: {args['loss_config']}")

    best_val_loss = float('inf')

    for epoch in range(args['epochs']):
        t0 = time.time()
    
        # Training
        train_losses = run_epoch(train_loader, model, optimizer, device, criterion, train=True)
    
        # Validation
        val_losses = run_epoch(
            val_loader, model, optimizer, device, criterion, train=False,
            visualize_every=args['viz_every'],
            vis_dir=os.path.join(args['save_dir'], "val_vis")
        )
    
        scheduler.step(val_losses['total_loss'])

        elapsed = time.time() - t0
        lr = optimizer.param_groups[0]['lr']
        
        # Enhanced logging with all loss components
        print(f"Epoch {epoch+1:03d}/{args['epochs']:03d} | Time: {elapsed:.1f}s | LR: {lr:.2e}")
        print(f"  Train - Total: {train_losses['total_loss']:.6f} | "
              f"MSE: {train_losses['mse_loss']:.6f} | "
              f"L1: {train_losses['l1_loss']:.6f} | "
              f"SSIM: {train_losses['ssim_value']:.4f}")
        print(f"  Val   - Total: {val_losses['total_loss']:.6f} | "
              f"MSE: {val_losses['mse_loss']:.6f} | "
              f"L1: {val_losses['l1_loss']:.6f} | "
              f"SSIM: {val_losses['ssim_value']:.4f}")
        
        if val_losses['perceptual_loss'] > 0:
            print(f"  Perceptual - Train: {train_losses['perceptual_loss']:.6f} | "
                  f"Val: {val_losses['perceptual_loss']:.6f}")
        
        # Save checkpoints
        os.makedirs(args['save_dir'], exist_ok=True)
        
        # Save best model
        if val_losses['total_loss'] < best_val_loss:
            best_val_loss = val_losses['total_loss']
            torch.save({
                'epoch': epoch + 1,
                'model_state_dict': model.state_dict(),
                'optimizer_state_dict': optimizer.state_dict(),
                'val_losses': val_losses,
                'train_losses': train_losses,
                'args': args
            }, os.path.join(args['save_dir'], "best_model.pth"))
        
        # Save periodic checkpoints
        if (epoch + 1) % args['save_every'] == 0:
            torch.save({
                'epoch': epoch + 1,
                'model_state_dict': model.state_dict(),
                'optimizer_state_dict': optimizer.state_dict(),
                'val_losses': val_losses,
                'train_losses': train_losses,
                'args': args
            }, os.path.join(args['save_dir'], f"epoch_{epoch+1}.pth"))

    print("U-Net training completed!")

if __name__ == "__main__":
    # First install pytorch-msssim: pip install pytorch-msssim
    torch.cuda.empty_cache()
    main()