#!/usr/bin/env python3
import os, glob, time
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
from torch.utils.data import DataLoader
import matplotlib.pyplot as plt
from transformers import ViTModel, ViTConfig
from pathlib import Path
from PIL import Image

# Updated orthonormal matrix generation for patch-wise processing
def make_gaussian_random_orthonormal_rows(h=256, w=256, seed=42):
    """
    Generate a matrix A of size [h, w] where rows are orthonormal.
    Note: Requires h <= w for orthonormal rows to exist.
    """
    if seed is not None:
        torch.manual_seed(seed)
    # Step 1: Random Gaussian matrix
    A = torch.randn(h, w)
    # Step 2: QR decomposition on transpose to orthonormalize rows
    # A^T = Q*R → A = R^T * Q^T
    # We want orthonormal rows, so we work with A^T first
    Q, R = torch.linalg.qr(A.T)  # A.T is [w, h]
    # Q is [w, h] with orthonormal columns
    # Q.T is [h, w] with orthonormal rows
    return Q.T  # [h, w] with orthonormal rows

class PatchwiseOrthonormalDataset:
    """
    Dataset that applies patch-wise orthonormal transformation to images.
    Each 16x16 patch gets transformed through a 256x256 orthonormal matrix.
    """
    def __init__(self, data_dir, seed=42, verbose=False):
        self.data_dir = data_dir
        
        # Generate fixed orthonormal matrix A for patch-wise transformation
        self.A = make_gaussian_random_orthonormal_rows(h=256, w=256, seed=seed)
        
        # Get all image files
        self.data_path = Path(data_dir)
        if not self.data_path.exists():
            raise FileNotFoundError(f"Data directory not found: {data_dir}")
        
        image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.tiff', '.JPEG', '.JPG'}
        self.image_files = [f for f in self.data_path.iterdir() 
                           if f.is_file() and f.suffix in image_extensions]
        
        if len(self.image_files) == 0:
            raise ValueError(f"No images found in {data_dir}")
        
        if verbose:
            print(f"Loaded {len(self.image_files)} images from {data_dir}")
            print(f"Using patch-wise orthonormal transformation with matrix shape: {self.A.shape}")

    def __len__(self):
        return len(self.image_files)

    def resize_min_side(self, img, min_side=224):
        w, h = img.size
        s = min_side / min(w, h)
        return img.resize((int(round(w*s)), int(round(h*s))), Image.Resampling.LANCZOS)

    def center_crop(self, img, size=224):
        w, h = img.size
        left = (w - size) // 2
        top = (h - size) // 2
        return img.crop((left, top, left + size, top + size))

    def preprocess_image(self, img):
        img = img.convert("RGB")
        img_resized = self.resize_min_side(img, 224)
        img_crop = self.center_crop(img_resized, 224)
        x = np.array(img_crop).astype(np.float32) / 255.0
        return x

    def process_image_with_orthonormal_masks(self, np_img, mask_matrix):
        """
        Apply orthonormal transformation to 16x16 patches of a 224x224 image.
        
        Args:
            np_img: numpy array of shape [224, 224, 3]
            mask_matrix: torch tensor of shape [256, 256] with orthonormal rows
        
        Returns:
            transformed_patches: torch tensor of shape [14, 14, 256] 
        """
        # Convert to torch and extract patches
        img_tensor = torch.from_numpy(np_img).float()
        
        # Convert to grayscale for 256 = 16*16
        if img_tensor.shape[2] == 3:
            img_gray = img_tensor.mean(dim=2)  # Convert to grayscale
        else:
            img_gray = img_tensor
        
        # Extract 16x16 patches from 224x224 image
        patches = img_gray.unfold(0, 16, 16).unfold(1, 16, 16)  # [14, 14, 16, 16]
        
        # Flatten each patch and apply transformation
        transformed_patches = torch.zeros(14, 14, 256)
        
        for i in range(14):
            for j in range(14):
                # Flatten 16x16 patch to 256x1
                patch_flat = patches[i, j].flatten()  # [256]
                
                # Apply orthonormal transformation: [256, 256] @ [256] → [256]
                transformed = mask_matrix @ patch_flat
                
                transformed_patches[i, j] = transformed
        
        return transformed_patches

    def reconstruct_masked_image(self, transformed_patches):
        """
        Reconstruct 224x224 image from 14x14x256 transformed patches.
        
        Args:
            transformed_patches: torch tensor of shape [14, 14, 256]
        
        Returns:
            masked_image: torch tensor of shape [224, 224]
        """
        masked_image = torch.zeros(224, 224)
        
        for i in range(14):
            for j in range(14):
                # Get the 256-dimensional transformed patch
                transformed_patch = transformed_patches[i, j]  # [256]
                
                # Reshape back to 16x16
                patch_16x16 = transformed_patch.reshape(16, 16)
                
                # Place in the correct position in the 224x224 image
                start_h = i * 16
                end_h = start_h + 16
                start_w = j * 16
                end_w = start_w + 16
                
                masked_image[start_h:end_h, start_w:end_w] = patch_16x16
        
        return masked_image

    def apply_patchwise_orthonormal_transform(self, x):
        """
        Apply patch-wise orthonormal transformation to RGB image.
        
        Args:
            x: numpy array of shape [224, 224, 3] (original image)
        
        Returns:
            y: numpy array of shape [224, 224, 3] (transformed image)
        """
        y = np.zeros_like(x, dtype=np.float32)
        
        # Apply transformation to each channel separately
        for c in range(3):
            # Create single-channel image for processing
            single_channel = x[..., c]
            
            # Apply patch-wise orthonormal transformation
            transformed_patches = self.process_image_with_orthonormal_masks(
                np.expand_dims(single_channel, axis=2), self.A
            )
            
            # Reconstruct to 224x224
            masked_channel = self.reconstruct_masked_image(transformed_patches)
            
            # Store result
            y[..., c] = masked_channel.numpy()
        
        # Normalize to [0, 1] range
        y_min = y.min()
        y_max = y.max()
        y_norm = (y - y_min) / (y_max - y_min + 1e-8)
        
        return y_norm

    def __getitem__(self, idx):
        img_path = self.image_files[idx]
        
        try:
            img = Image.open(img_path)
        except Exception as e:
            print(f"Warning: Could not load image {img_path}: {e}")
            img = Image.new('RGB', (224, 224), color=(0, 0, 0))
        
        # Preprocess to get x (target - original image)
        x = self.preprocess_image(img)  # (224, 224, 3)
        
        # Apply patch-wise orthonormal transform to get y (input - transformed image)
        y = self.apply_patchwise_orthonormal_transform(x)  # (224, 224, 3)
        
        # Convert to torch tensors and change to CHW format
        x_tensor = torch.from_numpy(x).permute(2, 0, 1)  # (3, 224, 224)
        y_tensor = torch.from_numpy(y).permute(2, 0, 1)  # (3, 224, 224)
        
        return y_tensor, x_tensor  # (input, target)

# -----------------------------
# ViT-UNet for RGB inverse problem
# -----------------------------
class ViTUNetForInverseProblem(nn.Module):
    def __init__(self, pretrained_model_name="google/vit-base-patch16-224", output_size=(224, 224)):
        super().__init__()
    
        # Load config and explicitly disable pooling layer
        cfg = ViTConfig.from_pretrained(pretrained_model_name)
        cfg.add_pooling_layer = False  # Explicitly disable pooler
    
        # Load ViT without pooler to avoid unused parameters
        self.vit = ViTModel.from_pretrained(pretrained_model_name, config=cfg, ignore_mismatched_sizes=True)
    
        self.output_size = output_size
        self.hidden_dim = 768

        # Feature pools - smaller since we're working with 224x224
        self.adaptive_pool1 = nn.AdaptiveAvgPool2d((28, 28))   # ~1/8 of 224
        self.adaptive_pool2 = nn.AdaptiveAvgPool2d((14, 14))   # ~1/16 of 224  
        self.adaptive_pool3 = nn.AdaptiveAvgPool2d((7, 7))     # ~1/32 of 224
        self.adaptive_pool_final = nn.AdaptiveAvgPool2d((7, 7))

        self.skip_conn1 = nn.Conv2d(self.hidden_dim, 128, kernel_size=1)
        self.skip_conn2 = nn.Conv2d(self.hidden_dim, 256, kernel_size=1)
        self.skip_conn3 = nn.Conv2d(self.hidden_dim, 512, kernel_size=1)

        # Decoder path
        self.up1 = nn.Sequential(
            nn.Upsample(size=(14, 14), mode='bilinear', align_corners=True),
            nn.Conv2d(self.hidden_dim, 512, 3, padding=1), 
            nn.BatchNorm2d(512), nn.ReLU(True)
        )
        self.up2 = nn.Sequential(
            nn.Upsample(size=(28, 28), mode='bilinear', align_corners=True),
            nn.Conv2d(512, 256, 3, padding=1), 
            nn.BatchNorm2d(256), nn.ReLU(True)
        )
        self.up3 = nn.Sequential(
            nn.Upsample(size=(56, 56), mode='bilinear', align_corners=True),
            nn.Conv2d(256, 128, 3, padding=1), 
            nn.BatchNorm2d(128), nn.ReLU(True)
        )
        self.up4 = nn.Sequential(
            nn.Upsample(size=(112, 112), mode='bilinear', align_corners=True),
            nn.Conv2d(128, 64, 3, padding=1), 
            nn.BatchNorm2d(64), nn.ReLU(True)
        )
    
        # Final output layer - RGB output
        self.final = nn.Sequential(
            nn.Upsample(size=(224, 224), mode='bilinear', align_corners=True),
            nn.Conv2d(64, 32, 3, padding=1),
            nn.BatchNorm2d(32), nn.ReLU(True),
            nn.Conv2d(32, 3, 3, padding=1),  # 3 channels for RGB
            nn.Sigmoid()  # Output in [0, 1] range
        )

    def _extract(self, x3):
        out = self.vit(x3, output_hidden_states=True)
        early = out.hidden_states[3]   # Early features
        mid   = out.hidden_states[6]   # Mid features  
        late  = out.hidden_states[9]   # Late features
        last  = out.last_hidden_state  # Final features
        return early, mid, late, last

    def _to_spatial(self, tokens):
        B, N, C = tokens.shape
        HW = int((N - 1) ** 0.5)  # Drop CLS token
        t = tokens[:, 1:, :].permute(0, 2, 1)
        return t.reshape(B, C, HW, HW)

    def forward(self, x):
        # x is already [B, 3, 224, 224] RGB input
        if x.shape[-2:] != (224, 224):
            x = F.interpolate(x, size=(224, 224), mode='bilinear', align_corners=True)

        # Extract multi-scale features from ViT
        e, m, l, f = self._extract(x)
        e, m, l, f = map(self._to_spatial, (e, m, l, f))

        # Create skip connections
        skip1 = self.adaptive_pool1(e)   # 28x28
        skip2 = self.adaptive_pool2(m)   # 14x14
        skip3 = self.adaptive_pool3(l)   # 7x7
        x = self.adaptive_pool_final(f)  # 7x7

        # Decoder with skip connections
        x = self.up1(x)  # 7x7 -> 14x14
        x = x + F.interpolate(self.skip_conn3(skip3), size=(14, 14), mode='bilinear', align_corners=True)
    
        x = self.up2(x)  # 14x14 -> 28x28
        x = x + F.interpolate(self.skip_conn2(skip2), size=(28, 28), mode='bilinear', align_corners=True)
    
        x = self.up3(x)  # 28x28 -> 56x56
        x = x + F.interpolate(self.skip_conn1(skip1), size=(56, 56), mode='bilinear', align_corners=True)
    
        x = self.up4(x)  # 56x56 -> 112x112
    
        # Final output
        out = self.final(x)  # 112x112 -> 224x224, 3 channels
    
        # Resize to target if needed
        if self.output_size != (224, 224):
            out = F.interpolate(out, size=self.output_size, mode='bilinear', align_corners=True)
        
        return out

# -----------------------------
# Simple MSE Loss
# -----------------------------
class MSELoss(nn.Module):
    def forward(self, pred, target):
        return F.mse_loss(pred, target)

# -----------------------------
# Train / Val loop
# -----------------------------
def run_epoch(loader, model, optim, device, criterion, train=True, visualize_every=0, vis_dir=None):
    if train:
        model.train()
    else:
        model.eval()

    total_loss = 0.0
    num_batches = 0

    with torch.set_grad_enabled(train):
        for i, (y_batch, x_batch) in enumerate(loader):
            # y_batch: transformed images (input), x_batch: original images (target)
            y_batch = y_batch.to(device, non_blocking=True)
            x_batch = x_batch.to(device, non_blocking=True)

            # Forward pass: predict x from y
            pred_x = model(y_batch)
        
            # Ensure shapes match
            if pred_x.shape != x_batch.shape:
                pred_x = F.interpolate(pred_x, size=x_batch.shape[-2:], mode='bilinear', align_corners=True)

            # Compute loss
            loss = criterion(pred_x, x_batch)
        
            if train:
                optim.zero_grad(set_to_none=True)
                loss.backward()
                optim.step()

            total_loss += loss.item()
            num_batches += 1

            # Visualization during validation
            if (not train) and visualize_every and (i % visualize_every == 0) and vis_dir:
                os.makedirs(vis_dir, exist_ok=True)
            
                # Convert to numpy for visualization
                y_vis = y_batch[0].detach().cpu().numpy().transpose(1, 2, 0)  # CHW -> HWC
                pred_vis = pred_x[0].detach().cpu().numpy().transpose(1, 2, 0)
                x_vis = x_batch[0].detach().cpu().numpy().transpose(1, 2, 0)
            
                fig, ax = plt.subplots(1, 3, figsize=(15, 5))
                ax[0].imshow(np.clip(y_vis, 0, 1))
                ax[0].set_title("Input (Patch-wise Masked)")
                ax[0].axis('off')
            
                ax[1].imshow(np.clip(pred_vis, 0, 1))
                ax[1].set_title("Predicted (x̂)")
                ax[1].axis('off')
            
                ax[2].imshow(np.clip(x_vis, 0, 1))
                ax[2].set_title("Target (Original)")
                ax[2].axis('off')
            
                plt.tight_layout()
                plt.savefig(os.path.join(vis_dir, f"val_{i:05d}.png"), dpi=120, bbox_inches='tight')
                plt.close()

    return total_loss / max(1, num_batches)

def train_single_gpu(args):
    # Setup device
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    print(f"Using device: {device}")

    # Create datasets with new patch-wise orthonormal approach
    train_ds = PatchwiseOrthonormalDataset(
        data_dir=args['train_dir'],
        seed=args['seed'],
        verbose=True
    )

    val_ds = PatchwiseOrthonormalDataset(
        data_dir=args['val_dir'],
        seed=args['seed'],  # Same seed for consistent A matrix
        verbose=True
    )

    # Create dataloaders
    train_loader = DataLoader(
        train_ds, batch_size=args['batch_size'], shuffle=True,
        num_workers=4, pin_memory=True, drop_last=True
    )
    val_loader = DataLoader(
        val_ds, batch_size=args['batch_size'], shuffle=False,
        num_workers=4, pin_memory=True, drop_last=False
    )

    # Create model
    model = ViTUNetForInverseProblem(output_size=args['target_size']).to(device)

    # Load checkpoint if provided
    if args['load_path'] and os.path.isfile(args['load_path']):
        ckpt = torch.load(args['load_path'], map_location=device)
        state_dict = ckpt['model_state_dict'] if 'model_state_dict' in ckpt else ckpt
        model.load_state_dict(state_dict, strict=False)
        print(f"Loaded checkpoint: {args['load_path']}")

    # Optimizer and scheduler
    optimizer = torch.optim.AdamW(model.parameters(), lr=args['lr'], weight_decay=1e-4)
    scheduler = torch.optim.lr_scheduler.ReduceLROnPlateau(
        optimizer, mode='min', factor=0.5, patience=5, verbose=True
    )
    criterion = MSELoss()

    print(f"Starting patch-wise orthonormal training on single GPU")
    print(f"Train: {len(train_ds)} images, Val: {len(val_ds)} images")
    print(f"Batch size: {args['batch_size']}")

    best_val_loss = float('inf')

    for epoch in range(args['epochs']):
        t0 = time.time()
    
        # Training
        train_loss = run_epoch(train_loader, model, optimizer, device, criterion, train=True)
    
        # Validation
        val_loss = run_epoch(
            val_loader, model, optimizer, device, criterion, train=False,
            visualize_every=args['viz_every'],
            vis_dir=os.path.join(args['save_dir'], "val_vis")
        )

        scheduler.step(val_loss)

        elapsed = time.time() - t0
        lr = optimizer.param_groups[0]['lr']
        print(f"Epoch {epoch+1:03d}/{args['epochs']:03d} | "
              f"Train: {train_loss:.6f} | Val: {val_loss:.6f} | "
              f"Time: {elapsed:.1f}s | LR: {lr:.2e}")
        
        # Save checkpoints
        os.makedirs(args['save_dir'], exist_ok=True)
        
        # Save best model
        if val_loss < best_val_loss:
            best_val_loss = val_loss
            torch.save({
                'epoch': epoch + 1,
                'model_state_dict': model.state_dict(),
                'optimizer_state_dict': optimizer.state_dict(),
                'val_loss': val_loss,
                'args': args
            }, os.path.join(args['save_dir'], "best_model.pth"))
            print(f"New best model saved with val_loss: {val_loss:.6f}")
        
        # Save periodic checkpoints
        if (epoch + 1) % args['save_every'] == 0:
            torch.save({
                'epoch': epoch + 1,
                'model_state_dict': model.state_dict(),
                'optimizer_state_dict': optimizer.state_dict(),
                'val_loss': val_loss,
                'args': args
            }, os.path.join(args['save_dir'], f"epoch_{epoch+1}.pth"))

# -----------------------------
# Main function
# -----------------------------
def main():
    args = {
        # Data paths (update these to your paths)
        'train_dir': r"F:\imgnet\data\train",
        'val_dir': r"F:\imgnet\data\val",
        'save_dir': "./trust_16x16_checkpoints",
        'load_path': "",  # Path to checkpoint to resume from (optional)
    
        # Model settings
        'target_size': (224, 224),  # Output image size
        'seed': 42,  # Fixed seed for orthonormal matrix A
    
        # Training settings
        'batch_size': 16,  # Reduced for single GPU
        'lr': 1e-4,
        'epochs': 100,
        'save_every': 10,
        'viz_every': 50,  # Visualize every N validation batches
    }

    # Create save directory
    os.makedirs(args['save_dir'], exist_ok=True)

    # Save args for reference
    import json
    with open(os.path.join(args['save_dir'], 'training_args.json'), 'w') as f:
        json.dump(args, f, indent=2)

    print(f"Detected {torch.cuda.device_count()} GPU(s)")
    print("Starting single-GPU training")
    
    train_single_gpu(args)

if __name__ == "__main__":
    torch.cuda.empty_cache()
    main()