import os
from enum import Enum
from typing import Optional

from dotenv import load_dotenv

NO_KEY_NEEDED = "NO_KEY_NEEDED"

class Provider(Enum):
    OPENAI = "openai"
    GROQ = "groq"
    DEEPSEEK = "deepseek"
    VLLM = "vllm"
    OPENROUTER = "openrouter"
    MOONSHOT = "moonshot"
    SGLANG = "sglang"
    @property
    def models(self):
        return PROVIDER_MODELS[self]
    
    @property
    def api_base_url(self):
        return PROVIDER_API_BASE_URL[self]
    
    @property
    def supports_multiple_completions(self):
        return self != Provider.DEEPSEEK

    def get_api_key(self, env_file) -> Optional[str]:
        load_dotenv(dotenv_path=env_file)
        env_key = PROVIDER_API_ENV_KEYS[self]
        if env_key == NO_KEY_NEEDED:
            return NO_KEY_NEEDED
        return os.getenv(env_key)

class ModelEnum(Enum):
    @property
    def requires_max_completion_tokens(self):
        return self in REQUIRES_MAX_COMPLETION_TOKENS
    
    @property
    def unsupported_kwargs(self):
        return UNSUPPORTED_KWARGS.get(self, [])

class OpenAIModels(ModelEnum):
    GPT_4 = "gpt-4"
    GPT_4_TURBO = "gpt-4-turbo"
    GPT_4O = "gpt-4o"
    GPT_35_TURBO = "gpt-3.5-turbo"
    GPT_4O_MINI = "gpt-4o-mini"

    O1 = "o1-2024-12-17"
    O1_PREVIEW = "o1-preview-2024-09-12"
    O1_MINI = "o1-mini-2024-09-12"
    O3_MINI = "o3-mini-2025-01-31"
    GPT_4_1_MINI = "gpt-4.1-mini"
    GPT_4_1 = "gpt-4.1"


class GroqModels(ModelEnum):
    LLAMA3_70B_8192 = "llama3-70b-8192"
    MIXTRAL_8X7B_32768 = "mixtral-8x7b-32768"
    LLAMA3_70B_128K = "llama-3.3-70b-versatile"


class DeepSeekModels(ModelEnum):
    DEEPSEEKV3 = "deepseek-chat"
    DEEPSEEKR1 = "deepseek-reasoner"


class MoonshotModels(ModelEnum):
    KIMI_K1_8K = "moonshot-v1-8k"
    KIMI_K1_32K = "moonshot-v1-32k"


class VLLMModels(ModelEnum):
    LLAMA3_70B = "meta-llama/Meta-Llama-3-70B-Instruct"
    QWQ_32B = "Qwen/QwQ-32B"
    QWEN2_5_VL_7B = "Qwen/Qwen2.5-VL-7B-Instruct"
    QWEN2_5_VL_72B = "Qwen/Qwen2.5-VL-72B-Instruct"
    QWEN2_5_72B = "Qwen/Qwen2.5-72B-Instruct"
    QWEN2_5_32B = "Qwen/Qwen2.5-32B-Instruct"
    QWEN2_5_14B = "Qwen/Qwen2.5-14B-Instruct"
    QWEN2_5_7B = "Qwen/Qwen2.5-7B-Instruct"
    INTERNVL3_38B = "OpenGVLab/InternVL3-38B"


class SGLangModels(ModelEnum):
    DEEPSEEK_R1_DISTILL_QWEN_7B = "deepseek-ai/DeepSeek-R1-Distill-Qwen-7B"
    DEEPSEEK_R1_DISTILL_QWEN_32B = "deepseek-ai/DeepSeek-R1-Distill-Qwen-32B"
    DEEPSEEK_R1_DISTILL_LLAMA_70B = "deepseek-ai/DeepSeek-R1-Distill-Llama-70B"
    LLAMA3_70B = "meta-llama/Llama-3.3-70B-Instruct"
    QWQ_32B = "Qwen/QwQ-32B"
    QWEN2_5_72B = "Qwen/Qwen2.5-72B-Instruct"
    QWEN2_5_32B = "Qwen/Qwen2.5-32B-Instruct"
    QWEN2_5_14B = "Qwen/Qwen2.5-14B-Instruct"
    QWEN2_5_7B = "Qwen/Qwen2.5-7B-Instruct"

class OpenRouterModels(ModelEnum):
    SONNET35 = "anthropic/claude-3.5-sonnet:beta"
    LLAMA3_70B = "meta-llama/llama-3-70b-instruct"
    DEEPSEEKCODER = "deepseek/deepseek-coder"


PROVIDER_MODELS = {
    Provider.OPENAI: OpenAIModels,
    Provider.GROQ: GroqModels,
    Provider.DEEPSEEK: DeepSeekModels,
    Provider.VLLM: VLLMModels,
    Provider.OPENROUTER: OpenRouterModels,
    Provider.SGLANG: SGLangModels,
    Provider.MOONSHOT: MoonshotModels,
}

PROVIDER_API_ENV_KEYS = {
    Provider.OPENAI: "OPENAI_API_KEY",
    Provider.GROQ: "GROQ_API_KEY",
    Provider.DEEPSEEK: "DEEPSEEK_API_KEY",
    Provider.MOONSHOT: "MOONSHOT_API_KEY",
    Provider.OPENROUTER: "OPENROUTER_API_KEY",
    Provider.VLLM: NO_KEY_NEEDED,
    Provider.SGLANG: NO_KEY_NEEDED,
}

PROVIDER_API_BASE_URL = {
    # Provider.OPENAI: "https://api.openai.com/v1",
    Provider.OPENAI: "https://api.vveai.com/v1",
    Provider.GROQ: "https://api.groq.com/openai/v1",
    Provider.DEEPSEEK: "https://api.deepseek.com/v1",
    Provider.VLLM: "http://localhost:8000/v1",
    Provider.OPENROUTER: "https://openrouter.ai/api/v1",
    Provider.MOONSHOT: "https://api.moonshot.cn/v1",
    Provider.SGLANG: "http://127.0.0.1:30000/v1"
}

REQUIRES_MAX_COMPLETION_TOKENS = {
    OpenAIModels.O3_MINI,
}

UNSUPPORTED_KWARGS = {
    OpenAIModels.O3_MINI: ["temperature", "top_p"]
}
