# Create the profiling directory if it doesn't exist
PROFILING_DIR="./profiling"
mkdir -p "$PROFILING_DIR"

# Get the current date and time
CURRENT_DATE=$(date +"%Y%m%d_%H%M%S")

# Clear previous benchmark executables and profiling files
echo "Clearing previous benchmark executables and profiling files..."
rm -f target/release/deps/benchmarks-*
rm -f perf.data perf.data.old cachegrind.out.*

# Generate flamegraph using cargo flamegraph
echo "Generating flamegraph..."
cargo flamegraph --bench benchmarks

# Move and rename flamegraph
FLAMEGRAPH_OUTPUT="flamegraph_$CURRENT_DATE.svg"
mv flamegraph.svg "$PROFILING_DIR/$FLAMEGRAPH_OUTPUT"
echo "Flamegraph generated and saved as $PROFILING_DIR/$FLAMEGRAPH_OUTPUT"

# Find the benchmark executable after running cargo flamegraph
BENCHMARK_EXEC=$(find target/release/deps -maxdepth 1 -type f -executable -name 'benchmarks-*' | head -n 1)

# Check if benchmark executable is found
if [[ -z "$BENCHMARK_EXEC" ]]; then
  echo "No benchmark executable found."
  exit 1
fi

echo "Benchmark executable found: $BENCHMARK_EXEC"

# Run cachegrind
echo "Running cachegrind..."
valgrind --tool=cachegrind --branch-sim=yes "$BENCHMARK_EXEC"

# Find the cachegrind output
CACHEGRIND_OUT=$(ls -t cachegrind.out.* | head -n 1)

# Check if cachegrind output is found
if [[ -z "$CACHEGRIND_OUT" ]]; then
  echo "No cachegrind output found."
  exit 1
fi

# Rename and move cachegrind output
CACHEGRIND_OUTPUT="cachegrind_$CURRENT_DATE.out"
mv "$CACHEGRIND_OUT" "$PROFILING_DIR/$CACHEGRIND_OUTPUT"

# Analyze cachegrind output
CACHEGRIND_REPORT="cachegrind_report_$CURRENT_DATE.txt"
cg_annotate "$PROFILING_DIR/$CACHEGRIND_OUTPUT" > "$PROFILING_DIR/$CACHEGRIND_REPORT"
echo "Cachegrind analysis saved as $PROFILING_DIR/$CACHEGRIND_REPORT"

# Open cachegrind GUI
echo "Opening cachegrind GUI..."
kcachegrind "$PROFILING_DIR/$CACHEGRIND_OUTPUT"

# Move perf.data files to profiling directory
mv perf.data "$PROFILING_DIR/perf_$CURRENT_DATE.data"
mv perf.data.old "$PROFILING_DIR/perf_$CURRENT_DATE.data.old"