use std::ops::{Add, AddAssign};


// use rustc_hash::FxHasher;
// use std::hash::BuildHasherDefault;
// use std::collections::HashSet;

pub type Float = f32;

// pub type FxHashSet<T> = HashSet<T, BuildHasherDefault<FxHasher>>;


// MARK: -Error struct

#[allow(dead_code)]
#[derive(Debug)]
pub enum Error{
    NodeNotFound(Index),
    NodeInBothSubtrees(Index),
    NodeAlreadyInserted(Index),
    EmptyTree,
}

// MARK: -Newtypes

#[derive(Eq, PartialEq, Hash, Copy, Clone, Debug)]
pub struct Index(pub usize);

impl From<usize> for Index{
    fn from(index: usize) -> Self{
        Index(index)
    }
}

#[derive(Copy, Clone, Debug)]
pub struct Weight(pub Float);

impl From<Float> for Weight{
    fn from(weight: Float) -> Self{
        Weight(weight)
    }
}
impl Add for Weight{
    type Output = Weight;
    fn add(self, other: Weight) -> Weight{
        Weight(self.0 + other.0)
    }
}

impl AddAssign for Weight{
    fn add_assign(&mut self, other: Weight){
        self.0 += other.0;
    }
}


#[derive(Copy, Clone, Debug, PartialEq, PartialOrd)]
pub struct CoresetCrossTerm(pub Float);
impl From<Float> for CoresetCrossTerm{
    fn from(coreset_cross_term: Float) -> Self{
        CoresetCrossTerm(coreset_cross_term)
    }
}


#[derive(Copy, Clone, Debug)]
pub struct SelfAffinity(pub Float);

impl From<Float> for SelfAffinity{
    fn from(self_affinity: Float) -> Self{
        SelfAffinity(self_affinity)
    }
}

// MARK: Datapoint struct

#[derive(Debug)]
pub struct Datapoint{
    pub weight: Weight,
    pub self_affinity: SelfAffinity,
}

impl Datapoint{
    #[allow(dead_code)]
    pub fn contribution(&self, smallest_coreset_self_affinity: Float) -> Float{
        self.weight.0*(self.self_affinity.0 + smallest_coreset_self_affinity)
    }

    #[allow(dead_code)]
    pub fn smoothed_contribution(&self,smallest_coreset_self_affinity: Float, cost: Float, coreset_star_weight: Weight) -> Float{
        self.contribution(smallest_coreset_self_affinity)/cost + self.weight.0/coreset_star_weight.0
    }
}


#[derive(Debug, Copy, Clone)]
pub struct DatapointWithCoresetCrossTerm{
    pub weight: Weight,
    pub self_affinity: SelfAffinity,
    pub coreset_cross_term: CoresetCrossTerm
}

impl DatapointWithCoresetCrossTerm{
    #[allow(dead_code)]
    pub fn contribution(&self) -> Float{
        self.weight.0*(self.self_affinity.0 + self.coreset_cross_term.0)
    }

    #[allow(dead_code)]
    pub fn smoothed_contribution(&self, cost: Float, coreset_start_weight: Weight) -> Float{
        self.contribution()/cost + self.weight.0/coreset_start_weight.0
    }
}

// tests:
#[cfg(test)]
mod tests{
    pub use super::*;

    #[test]
    fn test_datapoint_contribution(){
        let datapoint = Datapoint{
            weight: Weight(1.0),
            self_affinity: SelfAffinity(2.0),
        };
        let smallest_coreset_self_affinity = 0.5;
        let contribution = datapoint.contribution(smallest_coreset_self_affinity);
        assert_eq!(contribution, 2.5);
    }

    #[test]
    fn test_datapoint_smoothed_contribution(){
        let datapoint = DatapointWithCoresetCrossTerm{
            weight: Weight(1.0),
            self_affinity: SelfAffinity(2.0),
            coreset_cross_term: CoresetCrossTerm(0.5),
        };
        let cost = 5.0;
        let coreset_start_weight = Weight(10.0);
        let smoothed_contribution = datapoint.smoothed_contribution(cost, coreset_start_weight);
        assert_eq!(smoothed_contribution, 0.6);
    }
}