use std::fmt::Formatter;
use std::fmt::Debug;
use std::ptr::NonNull;
use ordered_float::OrderedFloat;
use rand::{rngs::ThreadRng, Rng};
use bumpalo::Bump;

use crate::improved::common::{Index, Weight, SelfAffinity, CoresetCrossTerm, Float, Error, DatapointWithCoresetCrossTerm};




// MARK: -Node structs

#[derive(Debug, Clone)]
#[allow(dead_code)]
pub struct IncidentLeafNode{
    pub index: Index,
    pub weight: Weight,
    pub self_affinity: SelfAffinity,
    pub coreset_cross_term: CoresetCrossTerm,
    pub parent: Option<NonNull<TreeNode>>
}

impl IncidentLeafNode{
    pub fn contribution(&self) -> Float{
        self.weight.0*(self.self_affinity.0 + self.coreset_cross_term.0)
    }
    pub fn smoothed_contribution(&self, cost: Float, coreset_star_weight: Float) -> Float{
        self.contribution()/cost + self.weight.0/coreset_star_weight
    }
}



pub struct InternalNode{
    left_child: Option<(NonNull<TreeNode>, usize)>,
    right_child: Option<(NonNull<TreeNode>, usize)>,
    contribution: Float,
    weight: Weight,
    parent: Option<NonNull<TreeNode>>
}

impl InternalNode{
    #[allow(dead_code)]
    fn smoothed_contribution(&self, cost: Float, coreset_star_weight: Float) -> Float{
        self.contribution/cost + self.weight.0/coreset_star_weight
    }

    #[allow(dead_code)]
    fn contribution(&self) -> Float{
        self.contribution
    }
}

// Don't need to implement Drop because we are using a bump allocator.

// impl Drop for InternalNode{
//     fn drop(&mut self) {
//         if let Some((left_child_pointer, _)) = &self.left_child{
//             drop(unsafe{Box::from_raw(*left_child_pointer)});
//         }
//         if let Some((right_child_pointer, _)) = &self.right_child{
//             drop(unsafe{Box::from_raw(*right_child_pointer)});
//         }
//     }
// }

impl Debug for InternalNode{
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {

        let left_child_string = match &self.left_child{
            None => None,
            Some((left_child_pointer, size)) => {
                let left_child = unsafe{left_child_pointer.as_ptr().as_ref().unwrap()};
                Some((size, left_child))
            }
        };

        let right_child_string = match &self.right_child{
            None => None,
            Some((right_child_pointer, size)) => {
                let right_child = unsafe{right_child_pointer.as_ptr().as_ref().unwrap()};
                Some((size, right_child))
            }
        };

        f.debug_struct("InternalNode")
            .field("weight", &self.weight)
            .field("contribution", &self.contribution())
            .field("left_child", &left_child_string)
            .field("right_child", &right_child_string)
            .field("parent", &self.parent)
            .finish()
    }

}


// MARK: -Node enum
#[derive(Debug)]
#[allow(dead_code)]
pub enum TreeNode{
    Leaf(IncidentLeafNode),
    Internal(InternalNode)
}



impl TreeNode{
    fn contribution(&self) -> Float{
        match self{
            TreeNode::Leaf(leaf) => leaf.contribution(),
            TreeNode::Internal(internal) => internal.contribution()
        }
    }
    fn smoothed_contribution(&self, cost: Float, coreset_star_weight: Float) -> Float{
        match self{
            TreeNode::Leaf(leaf) => leaf.smoothed_contribution(cost, coreset_star_weight),
            TreeNode::Internal(internal) => internal.smoothed_contribution(cost, coreset_star_weight)
        }
    }

    fn parent(&self) -> Option<NonNull<TreeNode>>{
        match self{
            TreeNode::Leaf(leaf) => leaf.parent,
            TreeNode::Internal(internal) => internal.parent
        }
    }

    pub fn index(&self) -> Index{
        match self{
            TreeNode::Leaf(leaf) => leaf.index,
            TreeNode::Internal(_) => unreachable!("Internal nodes don't have indices")
        }
    }

    pub fn weight(&self) -> Weight{
        match self{
            TreeNode::Leaf(leaf) => leaf.weight,
            TreeNode::Internal(internal) => internal.weight
        }
    }

}


pub struct IncidentTree{
    root: Option<(*mut TreeNode, usize)>,
    bump_allocator: Bump,
}


impl Debug for IncidentTree{
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        match self.root{
            None =>{
                f.debug_struct("IncidentTree")
                    .field("root", &Option::<()>::None)
                    .finish()
            },
            Some((root_pointer, size)) =>{
                let root = unsafe{root_pointer.as_ref().unwrap()};
                f.debug_struct("IncidentTree")
                    .field("size", &size)
                    .field("root", root)
                    .finish()
            }
        }
    }
}

#[allow(dead_code)]
impl IncidentTree{

    pub fn contribution(&self) -> Float{
        self.root.as_ref().map_or(0.0, |(root_pointer, _)| unsafe{&**root_pointer}.contribution())
    }
    pub fn smoothed_contribution(&self, cost: Float, coreset_star_weight: Float) -> Float{
        self.root.as_ref().map_or(0.0, |(root_pointer,_)| unsafe{&**root_pointer}.smoothed_contribution(cost, coreset_star_weight))
    }

    pub fn new() -> Self{
        Self{
            root: None,
            bump_allocator: Bump::new()
        }
    }

    #[cfg(test)]
    fn _compute_sampling_probability(&self, node_pointer: *mut TreeNode, cost: Float, coreset_star_weight: Float, smoothed:bool) -> Float{
        let node_to_sample = unsafe{& *node_pointer};
        let mut child_contribution = match smoothed{
            true => node_to_sample.smoothed_contribution(cost, coreset_star_weight),
            false => node_to_sample.contribution()
        };
        let mut maybe_parent_pointer = node_to_sample.parent();
        let mut prob = 1.0;
        while let Some(parent_pointer) = maybe_parent_pointer{
            if let TreeNode::Internal(ref internal_parent) = unsafe{&*parent_pointer.as_ptr()}{
                let node_contribution = match smoothed{
                    true => internal_parent.smoothed_contribution(cost, coreset_star_weight),
                    false => internal_parent.contribution()
                };
                prob *= child_contribution/node_contribution;
                child_contribution = node_contribution;
                maybe_parent_pointer = unsafe{& *parent_pointer.as_ptr()}.parent();
            }else{
                unreachable!("We should never reach here since leaf's don't have children");
            }
        }
        prob
    }

    #[cfg(test)]
    pub fn compute_sampling_probability(&self, node_pointer: *mut TreeNode) -> Float{
        self._compute_sampling_probability(node_pointer, 0.0, 0.0, false)
    }
    #[cfg(test)]
    pub fn compute_smoothed_sampling_probability(&self, node_pointer: *mut TreeNode, cost: Float, coreset_star_weight: Float) -> Float{
        self._compute_sampling_probability(node_pointer, cost, coreset_star_weight, true)
    }

    pub fn _sample_node(&self, cost: Float, coreset_star_weight: Float, rng: &mut ThreadRng, smoothed:bool)-> Result<(&TreeNode, Float), Error>{
        match self.root{
            None => Err(Error::EmptyTree),
            Some((root_pointer, _)) =>{
                let mut maybe_node_pointer = Some(root_pointer);
                let mut prob = 1.0;
                while let Some(node_pointer) = maybe_node_pointer{
                    let node = unsafe{&*node_pointer};
                    match node{
                        TreeNode::Internal(internal_node)  =>{
                            let (lhs_contribution,rhs_contribution) =  match smoothed{
                                true =>{
                                    (internal_node.left_child.as_ref()
                                    .map_or(0.0, |(pointer, _)| {
                                        unsafe{
                                            (& *pointer.as_ptr())
                                            .smoothed_contribution(cost, coreset_star_weight)
                                    }}),
                                    internal_node.right_child.as_ref()
                                    .map_or(0.0, |(pointer, _)| {
                                        unsafe{
                                            (& *pointer.as_ptr())
                                            .smoothed_contribution(cost, coreset_star_weight)}}))
                                },
                                false =>{
                                    (internal_node.left_child.as_ref()
                                    .map_or(0.0, |(pointer, _)| {
                                        unsafe{
                                            (& *pointer.as_ptr())
                                            .contribution()
                                    }}),
                                    internal_node.right_child.as_ref()
                                    .map_or(0.0, |(pointer, _)| {
                                        unsafe{
                                            (& *pointer.as_ptr())
                                            .contribution()}}))
                                }
                            };
                            let total_contribution = lhs_contribution + rhs_contribution;
                            let sample = rng.gen_range(0.0..total_contribution);
                            if sample < lhs_contribution{
                                maybe_node_pointer = internal_node.left_child.as_ref().map(|(pointer, _)| pointer.as_ptr());
                                prob *= lhs_contribution/total_contribution;
                            }else{
                                maybe_node_pointer = internal_node.right_child.as_ref().map(|(pointer, _)| pointer.as_ptr());
                                prob *= rhs_contribution/total_contribution;
                            }
                        },
                        TreeNode::Leaf(_) => return Ok((node,prob))
                    }
                }
                unreachable!("We should never reach here");
            }
        }
    }

    pub fn sample_node_smoothed(&self, cost: Float, coreset_star_weight: Float, rng: &mut ThreadRng) -> Result<(&TreeNode, Float), Error>{
        self._sample_node(cost, coreset_star_weight, rng, true)
    }
    pub fn sample_node(&self, rng: &mut ThreadRng) -> Result<(&TreeNode, Float), Error>{
        self._sample_node(0.0, 0.0, rng, false)
    }


    pub fn update_node_coreset_cross_term(&mut self, node_pointer: *mut TreeNode, coreset_cross_term: CoresetCrossTerm){
        // update the coreset crossterm of the node pointed to. Bubble up the change to the root.
        let node_to_update = unsafe{&mut *node_pointer};

        if let TreeNode::Leaf(leaf) = node_to_update{
            let new_cross_term: CoresetCrossTerm = OrderedFloat::min(leaf.coreset_cross_term.0.into(), coreset_cross_term.0.into()).0.into();
            let contribution_diff = (new_cross_term.0 - leaf.coreset_cross_term.0)*leaf.weight.0;
            leaf.coreset_cross_term = new_cross_term;
            let mut maybe_parent_pointer = leaf.parent;
            while let Some(parent_pointer) = maybe_parent_pointer{
                if let TreeNode::Internal(ref mut parent_internal_node) = unsafe{&mut *parent_pointer.as_ptr()}{
                    parent_internal_node.contribution += contribution_diff;
                    maybe_parent_pointer = parent_internal_node.parent;
                } else{
                    unreachable!("We should only be updating internal nodes");
                }
            }
        } else{
            unreachable!("We should only be updating leaf nodes");
        }
    }

    #[allow(unused_assignments)]
    pub fn delete_node(&mut self, node_pointer: *mut TreeNode) -> (Weight,SelfAffinity){
        // Delete node by updating it's anscestors all the way to the root. 
        // Since we use a bump allocator, we don't need to worry about freeing memory.
        // We just need to update the parent pointers of the node's ancestors.
        let node_to_delete = unsafe{& *node_pointer};
        let mut contribution = 0.0;
        let mut weight = 0.0;
        let mut self_affinity = 0.0;
        if let TreeNode::Leaf(leaf) = node_to_delete{
            contribution = leaf.contribution();
            weight = leaf.weight.0;
            self_affinity = leaf.self_affinity.0;
        }else{
            unreachable!("We should only be deleting leaf nodes");
        }
        let mut child_pointer = node_pointer;
        let mut maybe_parent_pointer = unsafe{(*node_pointer).parent()};
        let mut delete_child = true;
        while let Some(parent_pointer) = maybe_parent_pointer{
            if let TreeNode::Internal(ref mut parent_internal_node) = unsafe{&mut *parent_pointer.as_ptr()}{
                // update contribution and weight
                parent_internal_node.contribution -= contribution;
                parent_internal_node.weight.0 -= weight;
                // find the child pointer to update
                let target_child_pointer = match &mut parent_internal_node.left_child{
                    None => &mut parent_internal_node.right_child,
                    Some((left_child_pointer,_)) =>{
                        if left_child_pointer.as_ptr() == child_pointer{
                            &mut parent_internal_node.left_child
                        }else{
                            &mut parent_internal_node.right_child                        
                        }
                    }
                };
                match delete_child{
                    true => *target_child_pointer = None,
                    false =>{
                        *target_child_pointer = target_child_pointer.map(|(pointer,size)| (pointer, size-1));
                    }
                }
                // Now check if both children are None. If so mark this node for deletion.
                if parent_internal_node.left_child.is_none() && parent_internal_node.right_child.is_none(){
                    delete_child = true;
                } else{
                    delete_child = false;
                }
                maybe_parent_pointer = parent_internal_node.parent;
                child_pointer = parent_pointer.as_ptr();
            }else{
                unreachable!("We should only be updating internal nodes");
            }
        }
        return (Weight(weight), SelfAffinity(self_affinity));
    }
    pub fn insert_node(&mut self, datapoint: DatapointWithCoresetCrossTerm, index: Index) -> *mut TreeNode{
        if let None = self.root{
            let new_leaf_pointer = self.bump_allocator.alloc(
                TreeNode::Leaf(
                    IncidentLeafNode{
                        index,
                        weight: datapoint.weight,
                        self_affinity: datapoint.self_affinity,
                        coreset_cross_term: datapoint.coreset_cross_term,
                        parent: None
                    }
                )
            );
            self.root = Some((new_leaf_pointer, 1));
            return new_leaf_pointer;
        } else{
            self.root = self.root.map(|(root_pointer, size)| (root_pointer, size+1));
            return self._insert(index,datapoint);
        }
    }
    fn _insert(&mut self,index: Index, datapoint: DatapointWithCoresetCrossTerm) -> *mut TreeNode{

        let root_tree_node_pointer = self.root.unwrap().0;
        let mut tree_node_pointer = root_tree_node_pointer;

        while let TreeNode::Internal(internal_node) = unsafe{&mut *tree_node_pointer}{
            // first update the internal node's contribution and weight
            internal_node.contribution += datapoint.contribution();
            internal_node.weight += datapoint.weight;

            // Now decide which subtree to go down
            let target_subtree = match (internal_node.left_child,internal_node.right_child){
                (None,None) => unreachable!(),
                (Some(_), None) => {
                    &mut internal_node.right_child
                },
                (None, Some(_)) => {
                    unreachable!("We should always have a left child if we have a right child");
                },
                (Some((_,left_size)),Some((_,right_size))) =>{
                    match left_size <= right_size{
                        true => &mut internal_node.left_child,
                        false => &mut internal_node.right_child
                    }
                }
            };

            match target_subtree.is_none(){
                true =>{
                    // Need to insert the new node here. We use the bump allocator to allocate the new node.
                    let new_leaf_mut_ref = self.bump_allocator.alloc(
                        TreeNode::Leaf(IncidentLeafNode{
                            index,
                            weight: datapoint.weight,
                            self_affinity: datapoint.self_affinity,
                            coreset_cross_term: datapoint.coreset_cross_term,
                            parent: Some(unsafe{NonNull::new_unchecked(tree_node_pointer)})
                        })
                    ) as &mut TreeNode;
                    // update the internal node to point to the new leaf.
                    let new_leaf_pointer = new_leaf_mut_ref as *mut TreeNode;
                    *target_subtree = target_subtree.map(|(_, size)| (unsafe{NonNull::new_unchecked(new_leaf_pointer)}, size+1));
                    return new_leaf_pointer;
                },
                false =>{
                    // Need to increment the size of the target subtree then traverse into it.
                    *target_subtree = target_subtree.map(|(pointer, size)| (pointer, size+1));
                    tree_node_pointer = target_subtree.unwrap().0.as_ptr();
                }
            }
        }

        // Now we are at a leaf node. We need to replace the leaf node with an internal node which has the leaf node and the new node as children.
        // Taking care to only allocate the new node. Otherwise we are just performing pointer surgery and updating nodes.
        match unsafe{&mut *tree_node_pointer}{
            TreeNode::Leaf(leaf) =>{
                // We need to replace the leaf node with an internal node which has the leaf node and the new node as children.
                // First get the old parent pointer
                let old_parent_pointer = leaf.parent;
                let new_internal_node_pointer = self.bump_allocator.alloc(TreeNode::Internal(InternalNode{
                    left_child: Some((unsafe{NonNull::new_unchecked(tree_node_pointer.clone())}, 1)),
                    right_child: None,
                    contribution: leaf.contribution() + datapoint.contribution(),
                    weight: leaf.weight + datapoint.weight,
                    parent: leaf.parent
                })) as *mut TreeNode;
                // update the current leaf node to point to the new internal node.
                leaf.parent = Some(unsafe{NonNull::new_unchecked(new_internal_node_pointer)});
                // Create the new leaf node
                let new_leaf_pointer = self.bump_allocator.alloc(TreeNode::Leaf(IncidentLeafNode{
                    index: index,
                    weight: datapoint.weight,
                    self_affinity: datapoint.self_affinity,
                    coreset_cross_term: datapoint.coreset_cross_term,
                    parent: Some(unsafe{NonNull::new_unchecked(new_internal_node_pointer)})
                })) as *mut TreeNode;
                // Update the new internal node to point to the new leaf node.
                if let TreeNode::Internal(internal_node) = unsafe{&mut *new_internal_node_pointer}{
                    internal_node.right_child = Some((unsafe{NonNull::new_unchecked(new_leaf_pointer)}, 1));
                }
                // update the parent of the new internal node to point to the new internal node if it exists.
                if let Some(parent) = old_parent_pointer.map(|mut p| unsafe{p.as_mut()}){
                    match parent{
                        TreeNode::Leaf(_) => unreachable!(),
                        TreeNode::Internal(parent_internal_node) =>{
                            if parent_internal_node.left_child.unwrap().0.as_ptr() == tree_node_pointer{
                                parent_internal_node.left_child = parent_internal_node.left_child.map(|(_, size)| (unsafe{NonNull::new_unchecked(new_internal_node_pointer)}, size));
                            } else{
                                parent_internal_node.right_child = parent_internal_node.right_child.map(|(_, size)| (unsafe{NonNull::new_unchecked(new_internal_node_pointer)}, size));
                            }
                        }
                    }
                }else{
                    // The leaf node was the root so we have to update the root pointer to point to the new internal node.
                    self.root = self.root.map(|(_, size)| (new_internal_node_pointer, size));
                }
                return new_leaf_pointer;
            },
            TreeNode::Internal(_) =>{
                // We need to insert the new node in the subtree rooted at the left or right child.
                unreachable!();
            }
        }
    }


}

#[cfg(test)]
mod tests{
    use super::*;
    use std::collections::HashSet;


    #[allow(dead_code)]
    fn compute_actual_total_contribution(data_points: &[DatapointWithCoresetCrossTerm], indices_present: &HashSet<Index>) -> Float{
        (0..data_points.len()).map(|i|{
            match indices_present.contains(&Index(i)){
                true => data_points[i].contribution(),
                false => 0.0
            }
        }).sum::<Float>()
    }
    
    #[allow(dead_code)]
    fn compute_actual_total_smoothed_contribution(data_points: &[DatapointWithCoresetCrossTerm], indices_present: &HashSet<Index>, cost: Float, coreset_star_weight: Float) -> Float{
        (0..data_points.len()).map(|i|{
            match indices_present.contains(&Index(i)){
                true => data_points[i].smoothed_contribution(cost, coreset_star_weight.into()),
                false => 0.0
            }
        }).sum::<Float>()
    }
    
    #[allow(dead_code)]
    fn test_total_contribution(tree: &IncidentTree, data_points: &[DatapointWithCoresetCrossTerm], indices_present: &HashSet<Index>){
        let total_contribution = compute_actual_total_contribution(data_points, indices_present);
        println!("Actual vs tree total contribution: {:.3?} vs {:.3?}", total_contribution, tree.contribution());
        assert!((total_contribution - tree.contribution()).abs() < 1e-6);
    }
    
    #[allow(dead_code)]
    fn test_sampling_probabilities(tree: &IncidentTree, data_points: &[DatapointWithCoresetCrossTerm], indices_present: &HashSet<Index>, pointers: &[*mut TreeNode]){
        let total_contribution = compute_actual_total_contribution(data_points, indices_present);
    
        let target_probs = (0..data_points.len()).map(|i|{
            match indices_present.contains(&i.into()){
                true => data_points[i].contribution()/total_contribution,
                false => 0.0
            }
        }).collect::<Vec<Float>>();
    
        let tree_probs = (0..data_points.len()).map(|i|{
            match indices_present.contains(&i.into()){
                true => tree.compute_sampling_probability(pointers[i]),
                false => 0.0
            }
        }).collect::<Vec<Float>>();
    
        println!("Target probs: {:?}", target_probs);
        println!("Tree probs: {:?}", tree_probs);
    
        for (target_prob, tree_prob) in target_probs.iter().zip(tree_probs.iter()){
            assert!((target_prob - tree_prob).abs() < 1e-6);
        }
    }
    
    #[allow(dead_code)]
    fn test_smoothed_sampling_probabilities(tree: &IncidentTree, data_points: &[DatapointWithCoresetCrossTerm], indices_present: &HashSet<Index>, cost: Float, coreset_star_weight: Float, pointers: &[*mut TreeNode]){
        let total_contribution = compute_actual_total_smoothed_contribution(data_points, indices_present, cost, coreset_star_weight);
    
        let target_probs = (0..data_points.len()).map(|i|{
            match indices_present.contains(&i.into()){
                true => data_points[i].smoothed_contribution(cost, coreset_star_weight.into())/total_contribution,
                false => 0.0
            }
        }).collect::<Vec<Float>>();
    
        let tree_probs = (0..data_points.len()).map(|i|{
            match indices_present.contains(&i.into()){
                true => tree.compute_smoothed_sampling_probability(pointers[i], cost, coreset_star_weight),
                false => 0.0
            }
        }).collect::<Vec<Float>>();
    
        println!("Target probs: {:?}", target_probs);
        println!("Tree probs: {:?}", tree_probs);
    
        for (target_prob, tree_prob) in target_probs.iter().zip(tree_probs.iter()){
            assert!((target_prob - tree_prob).abs() < 1e-6);
        }
    }



    #[test]
    fn test_incident_tree_sample(){
        let mut data_points = vec![
            DatapointWithCoresetCrossTerm{weight: Weight(1.0), self_affinity: SelfAffinity(1.0), coreset_cross_term: CoresetCrossTerm(1.0)},
            DatapointWithCoresetCrossTerm{weight: Weight(2.0), self_affinity: SelfAffinity(2.0), coreset_cross_term: CoresetCrossTerm(2.0)},
            DatapointWithCoresetCrossTerm{weight: Weight(3.0), self_affinity: SelfAffinity(3.0), coreset_cross_term: CoresetCrossTerm(3.0)},
            DatapointWithCoresetCrossTerm{weight: Weight(4.0), self_affinity: SelfAffinity(4.0), coreset_cross_term: CoresetCrossTerm(4.0)},
            DatapointWithCoresetCrossTerm{weight: Weight(5.0), self_affinity: SelfAffinity(5.0), coreset_cross_term: CoresetCrossTerm(5.0)}
        ];
        let mut node_pointers = vec!();

        let mut tree = IncidentTree::new();
        let order_to_add = vec![0,1,2,3,4];
        let order_to_delete = vec![2,3,1,0,4];
    
        let mut indices_present = HashSet::new();
        
        println!("testing insertion");
        for i in order_to_add.iter(){
            let pointer = tree.insert_node(data_points[*i], (*i).into());
            node_pointers.push(pointer);
            println!(" gives {:?}", tree);
            indices_present.insert((*i).into());
            test_total_contribution(&tree, &data_points, &indices_present);
            test_sampling_probabilities(&tree, &data_points, &indices_present, &node_pointers);
            test_smoothed_sampling_probabilities(&tree, &data_points, &indices_present, 1.0, 15.0, &node_pointers);
        }

        println!("testing updates");
        // Now test updating the coreset cross terms
        for i in order_to_add{
            let new_cross_term = CoresetCrossTerm(1.0/((i+1) as Float));
            println!("{:?}", (data_points[i].coreset_cross_term,new_cross_term));
            data_points[i].coreset_cross_term = new_cross_term;

            tree.update_node_coreset_cross_term(node_pointers[i], new_cross_term);
            test_total_contribution(&tree, &data_points, &indices_present);
            test_sampling_probabilities(&tree, &data_points, &indices_present, &node_pointers);
            test_smoothed_sampling_probabilities(&tree, &data_points, &indices_present, 1.0, 15.0, &node_pointers);
        }
        println!("{:?}", &tree);
        println!("testing deletion");
        for i in (0..order_to_delete.len()).into_iter(){
            let to_delete = order_to_delete[i];
            println!("{:?}", "#".repeat(40));
            tree.delete_node(node_pointers[to_delete]);
            indices_present.remove(&to_delete.into());
            test_total_contribution(&tree, &data_points, &indices_present);
            test_sampling_probabilities(&tree, &data_points, &indices_present, &node_pointers);
        }
    }
}
