import math,sys
import numpy as np
import pandas as pd

def optimizer(weight_sets: np.ndarray, pareto_front: np.ndarray) -> np.ndarray:  
    """For each set of objective weights, determines the best solution from the given pareto-front.

    Returns:
        np.ndarray: The best solutions in the pareto-front according to each set of objective weights.
    """    
    linearized_objs = pareto_front @ weight_sets.T
    res = pareto_front[np.argmin(linearized_objs, axis=0)]
    return res

def optimizer_ref_point(pareto_front: np.ndarray, reference_points: np.ndarray) -> np.ndarray:  
    """Determines the best solution(s) of a pareto-front with respect to some reference points.

    Args:
        pareto_front (np.ndarray): A list of objective vectors.
        reference_points (np.ndarray): A list of reference points relative to which the best solution should be determined.

    Returns:
        np.ndarray: The best solutions in the pareto-front according to each set of objective weights.
    """
    
    ideal = np.min(pareto_front, axis=0)
    utopian_point = ideal-0.01
    
    linearized_objs = scalarize_STOM(pareto_front, utopian_point, reference_points)
    res = pareto_front[np.argmin(linearized_objs, axis=0)]
    return res

def scalarize_STOM(obj_vectors: np.ndarray, utopian_point: np.array, ref_points: np.ndarray) -> np.ndarray:
    """Scalarizes a given objective vector according to the STOM [1] scalarizing function.
    [1] Nakayama, H. (1995). Aspiration level approach to interactive multi-objective programming and its applications.
    
    Args:
        obj_vector (np.ndarray): A list of objective vectors of size n*o, where o is the number of objectives. 
        utopian_pt (np.array): A point in objective space (length o) that is better than any achievable objective value in all objectives.
        ref_point (np.ndarray): A list of reference points of size k*o. Each point represents a set of desirable objective values. These values should be larger than the corresponding values in the utopian_pt.
        
    Returns:
        np.ndarray: The scalarized objective value of each objective vector with respect to each reference point. The result has size n*k.
    """
    rho = 10 ** -8
    return np.max((obj_vectors - utopian_point)[:,:,None] / (ref_points - utopian_point).T[None,:,:], axis=1) + rho * np.sum(obj_vectors[:,:,None] / (ref_points - utopian_point).T[None,:,:], axis=1)

def shapley_values(pareto_front: np.ndarray, weight_set: np.array) -> np.ndarray:
    """Calculates Shapley regression values (defined in [1], calculation from [2]) of each objective for the given weight vector.
    The "model" is a black-box optimizer which selects the best solution out of a multi-objective Pareto-front (or its approximation), 
    with respect to a particular objective weight vector, which is regarded as the model input (each weight is one feature).
    Feature subsets are generated by masking out missing weights (i.e. replacing them with 0), which does not require any modification of the optimizer.
    Note that as exact computation of regressive Shapley values requires an evaluation for each subset of objectives, 
    the time required by this implementation increases exponentially with the number of features, making it unsuitable for problems with more than ~20 objectives.

    Args:
        pareto_front (np.ndarray): A set of (near) Pareto-optimal, nondominated solutions. Each row contains the objective vector of one solution.
        weight_set (np.array): A list of objective weights.

    Returns:
        np.ndarray: a 2d array of Shapley values, where result[i, j] indicates the effect of objective j on objective i under the given weights. 
    """    
    
    n_objs = len(weight_set)
    
    subset_sizes = [bin(i).count('1') for i in range(2 ** n_objs)]
    size_factors = np.array([1 / (math.comb(n_objs, s) * (n_objs - s)) for s in subset_sizes[:-1]])
        
    masked_weights = weight_set * np.array([idx_to_subset(i, n_objs) for i in range(2 ** n_objs)])
    opt_solutions = optimizer(masked_weights, pareto_front)
    
    major_obj_values = np.zeros((n_objs, 2 ** n_objs - 1, n_objs))
    for obj in range(n_objs):
        for s in range(2 ** n_objs - 1):
            for i in range(n_objs):
                major_obj_values[obj, s, i] = opt_solutions[s | 2**(n_objs-i-1), obj]
                
    obj_diffs = major_obj_values - opt_solutions.T[:,:-1,None]
            
    shapley_values = np.sum(obj_diffs * size_factors[:,None], axis=1)
    return shapley_values
        
def idx_to_subset(index, n_objs):
    return [int(x) for x in bin(index)[2:].zfill(n_objs)]
        
        
