import numpy as np
import pandas as pd
from skfp.preprocessing import MolFromSmilesTransformer
from sklearn.ensemble import RandomForestClassifier
from sklearn.metrics import roc_auc_score
from sklearn.pipeline import make_pipeline

from mol2vec.mol2vec_fp import Mol2VecFingerprint


def load_data() -> tuple[list[str], list[str], np.ndarray, np.ndarray]:
    df_train = pd.read_csv("mol2vec/other_datasets/apistox_time_train.csv")
    df_test = pd.read_csv("mol2vec/other_datasets/apistox_time_test.csv")

    smiles_train = df_train["SMILES"]
    smiles_test = df_test["SMILES"]

    y_train = df_train["label"]
    y_test = df_test["label"]

    return smiles_train, smiles_test, y_train, y_test


if __name__ == "__main__":
    smiles_train, smiles_test, y_train, y_test = load_data()

    for model_type, model_path in [
        ("original", "outputs/mol2vec_original.pkl"),
        ("retrained", "outputs/mol2vec.model"),
    ]:
        print(model_type)

        pipeline = make_pipeline(
            MolFromSmilesTransformer(n_jobs=-1, suppress_warnings=True),
            Mol2VecFingerprint(model_path, n_jobs=-1),
            RandomForestClassifier(
                n_estimators=500, criterion="entropy", n_jobs=-1, random_state=0
            ),
        )
        pipeline.fit(smiles_train, y_train)

        y_pred = pipeline.predict(smiles_test)
        y_pred_proba = pipeline.predict_proba(smiles_test)[:, 1]

        auroc = roc_auc_score(y_test, y_pred_proba)
        print(f"AUROC: {auroc:.2%}")
        print()
