import gurobipy as gp
from gurobipy import GRB

def solve_cost_aware_matching(weights, init_cap, beta=0.03):
    n, m = weights.shape
    K = sum(init_cap)

    model = gp.Model("CostAwareMatching")
    z = model.addVars(n, m, vtype=GRB.BINARY, name="z")

    c = model.addVars(m, vtype=GRB.INTEGER, lb=0, ub=n, name="c")

    # Absolute deviation  |c_j − init_cap_j|
    dev = model.addVars(m, vtype=GRB.CONTINUOUS, name="dev")
    model.addConstrs(dev[j] >=  c[j] - init_cap[j] for j in range(m))
    model.addConstrs(dev[j] >= -c[j] + init_cap[j] for j in range(m))

    # scaledDev_j  =  β · |Δc_j|
    scaledDev = model.addVars(m, lb=-GRB.INFINITY, name="scaledDev")
    model.addConstrs(scaledDev[j] == beta * dev[j]   for j in range(m))

    penalty = gp.quicksum(scaledDev[j] for j in range(m))
    
    # 1. Matching constraints
    model.addConstrs(z.sum(i, '*') <= 1 for i in range(n))

    # 2. Capacity constraints
    model.addConstrs(z.sum('*', j) <= c[j] for j in range(m))

    # 3. Total capacity constraint
    model.addConstr(c.sum() == K, name='GlobalCap')

    # Objective: maximise total welfare minus penalty
    model.setObjective(gp.quicksum(weights[i, j] * z[i, j]
                                    for i in range(n) for j in range(m)) - penalty,
                        GRB.MAXIMIZE)

    model.optimize()
    
    return model, z, c
