"""mobilenetv2 in pytorch
[1] Mark Sandler, Andrew Howard, Menglong Zhu, Andrey Zhmoginov, Liang-Chieh Chen

    MobileNetV2: Inverted Residuals and Linear Bottlenecks
    https://arxiv.org/abs/1801.04381
"""

import torch
import torch.nn as nn
import torch.nn.functional as F
from .lsq_layer import Conv2dLSQ, LinearLSQ, ActLSQ

def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=False)

def quanconv3x3(in_planes, out_planes, num_bits, stride=1):
    """3x3 convolution with padding"""
    return Conv2dLSQ(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, nbits=num_bits)

def quanconv1x1(in_planes, out_planes, num_bits, stride=1):
    """1x1 convolution"""
    return Conv2dLSQ(in_planes, out_planes, kernel_size=1, stride=stride, padding=0, nbits=num_bits)

class LearnableBias(nn.Module):
    def __init__(self, out_chn):
        super(LearnableBias, self).__init__()
        self.bias = nn.Parameter(torch.zeros(1,out_chn,1,1), requires_grad=True)

    def forward(self, x):
        out = x + self.bias.expand_as(x)
        return out

class LinearBottleNeck(nn.Module):

    def __init__(self, in_channels, out_channels, stride, t=6, class_num=100):
        super().__init__()

        self.residual = nn.Sequential(
            nn.Conv2d(in_channels, in_channels * t, 1),
            nn.BatchNorm2d(in_channels * t),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t, in_channels * t, 3, stride=stride, padding=1, groups=in_channels * t),
            nn.BatchNorm2d(in_channels * t),
            nn.ReLU6(inplace=True),

            nn.Conv2d(in_channels * t, out_channels, 1),
            nn.BatchNorm2d(out_channels)
        )

        self.stride = stride
        self.in_channels = in_channels
        self.out_channels = out_channels

    def forward(self, x):

        residual = self.residual(x)

        if self.stride == 1 and self.in_channels == self.out_channels:
            residual += x

        return residual
    
class QLinearBottleNeck(nn.Module):

    def __init__(self, bitwidth, in_channels, out_channels, stride, t=6, class_num=100):
        super().__init__()

        self.bias11 = LearnableBias(in_channels)
        self.prelu1 = nn.PReLU(in_channels)
        self.bias12 = LearnableBias(in_channels)
        self.quan1 = ActLSQ(nbits = bitwidth)
        self.conv1 = quanconv1x1(in_channels, in_channels * t, bitwidth)
        self.bn1 = nn.BatchNorm2d(in_channels * t)

        self.bias21 = LearnableBias(in_channels * t)
        self.prelu2 = nn.PReLU(in_channels * t)
        self.bias22 = LearnableBias(in_channels * t)
        self.quan2 = ActLSQ(nbits = bitwidth)
        self.conv2 = quanconv3x3(in_channels * t, in_channels * t, bitwidth)
        self.bn2 = nn.BatchNorm2d(in_channels * t)

        self.bias31 = LearnableBias(in_channels * t)
        self.prelu3 = nn.PReLU(in_channels * t)
        self.bias32 = LearnableBias(in_channels * t)
        self.quan3 = ActLSQ(nbits = bitwidth)
        self.conv3 = quanconv1x1(in_channels * t, out_channels, bitwidth)
        self.bn3 = nn.BatchNorm2d(out_channels)

        self.stride = stride
        self.in_channels = in_channels
        self.out_channels = out_channels

    def forward(self, x):

        out = self.bias11(x)
        out = self.prelu1(out)
        out = self.bias12(out)
        out = self.quan1(out)
        out = self.conv1(out)
        out = self.bn1(out)

        out = self.bias21(out)
        out = self.prelu2(out)
        out = self.bias22(out)
        out = self.quan2(out)
        out = self.conv2(out)
        out = self.bn2(out)

        out = self.bias31(out)
        out = self.prelu3(out)
        out = self.bias32(out)
        out = self.quan3(out)
        out = self.conv3(out)
        out = self.bn3(out)

        if self.stride == 1 and self.in_channels == self.out_channels:
            out += x

        return out

class MobileNetV2(nn.Module):
    def __init__(self, size_im = 32, class_num=100):
        super().__init__()

        self.pre = nn.Sequential(
            nn.Conv2d(3, size_im, 1, padding=1),
            nn.BatchNorm2d(size_im),
            nn.ReLU6(inplace=True)
        )

        self.stage1 = LinearBottleNeck(32, 16, 1, 1)
        self.stage2 = self._make_stage(2, 16, 24, 2, 6)
        self.stage3 = self._make_stage(3, 24, 32, 2, 6)
        self.stage4 = self._make_stage(4, 32, 64, 2, 6)
        self.stage5 = self._make_stage(3, 64, 96, 1, 6)
        self.stage6 = self._make_stage(3, 96, 160, 1, 6)
        self.stage7 = LinearBottleNeck(160, 320, 1, 6)

        self.conv1 = nn.Sequential(
            nn.Conv2d(320, 1280, 1),
            nn.BatchNorm2d(1280),
            nn.ReLU6(inplace=True)
        )

        self.conv2 = nn.Conv2d(1280, class_num, 1)

    def forward(self, x):
        x = self.pre(x)
        x = self.stage1(x)
        x = self.stage2(x)
        x = self.stage3(x)
        x = self.stage4(x)
        x = self.stage5(x)
        x = self.stage6(x)
        x = self.stage7(x)
        x = self.conv1(x)
        x = F.adaptive_avg_pool2d(x, 1)
        x = self.conv2(x)
        x = x.view(x.size(0), -1)

        return x

    def _make_stage(self, repeat, in_channels, out_channels, stride, t):

        layers = []
        layers.append(LinearBottleNeck(in_channels, out_channels, stride, t))

        while repeat - 1:
            layers.append(LinearBottleNeck(out_channels, out_channels, 1, t))
            repeat -= 1

        return nn.Sequential(*layers)
    
class QMobileNetV2(nn.Module):
    def __init__(self, bitwidth, size_im = 32, class_num=100):
        super().__init__()

        self.pre = nn.Sequential(
            nn.Conv2d(3, size_im, 1, padding=1),
            nn.BatchNorm2d(size_im),
            nn.ReLU6(inplace=True)
        )
        self.stage1 = QLinearBottleNeck(bitwidth, 32, 16, 1, 1)
        self.stage2 = self._make_stage(bitwidth, 2, 16, 24, 2, 6)
        self.stage3 = self._make_stage(bitwidth, 3, 24, 32, 2, 6)
        self.stage4 = self._make_stage(bitwidth, 4, 32, 64, 2, 6)
        self.stage5 = self._make_stage(bitwidth, 3, 64, 96, 1, 6)
        self.stage6 = self._make_stage(bitwidth, 3, 96, 160, 1, 6)
        self.stage7 = QLinearBottleNeck(bitwidth, 160, 320, 1, 6)
        self.conv1 = nn.Sequential(
            quanconv1x1(320, 1280, bitwidth, 1),
            nn.BatchNorm2d(1280),
            nn.ReLU6(inplace=True)
        )
        self.conv2 = nn.Conv2d(1280, class_num, 1)

    def forward(self, x):
        x = self.pre(x)
        x = self.stage1(x)
        x = self.stage2(x)
        x = self.stage3(x)
        x = self.stage4(x)
        x = self.stage5(x)
        x = self.stage6(x)
        x = self.stage7(x)
        x = self.conv1(x)
        x = F.adaptive_avg_pool2d(x, 1)
        x = self.conv2(x)
        x = x.view(x.size(0), -1)

        return x

    def _make_stage(self, bitwidth, repeat, in_channels, out_channels, stride, t):
        layers = []
        layers.append(QLinearBottleNeck(bitwidth, in_channels, out_channels, stride, t))
        while repeat - 1:
            layers.append(QLinearBottleNeck(bitwidth, out_channels, out_channels, 1, t))
            repeat -= 1
        return nn.Sequential(*layers)

def Mobilenetv2(channel: int, num_classes: int, im_size, record_embedding: bool = False, no_grad: bool = False,
    pretrained: bool = False, **kwargs):
    return MobileNetV2(size_im = im_size[0], class_num = num_classes)

def QMobilenetv2(bitwidth: int, num_classes: int, im_size, record_embedding: bool = False, no_grad: bool = False,
    pretrained: bool = False, **kwargs):
    return QMobileNetV2(bitwidth, size_im = im_size[0], class_num = num_classes)