#!/bin/bash

# Define the base multipliers
N_VALUES=(2 15 37 80 99 130 144 153 162 176 250 319 409 277 417 466 486 698 770 879 968 207 327 340 353 387 468 505 530 587 629 717 779 805 947 950)

# Create an array of indices
SAMPLE_INDICES=()
for n in "${N_VALUES[@]}"; do
    start=$((n * 50))
    end=$(((n+1) * 50))
    for ((i=start; i<end; i++)); do
        SAMPLE_INDICES+=($i)
    done
done

# Define the jobs directory and error log
JOBS_DIR="/data5/users/anonymoususer/causal_path/CausalPathTracing_for_ViT/main/jobs/deit_tiny_patch16_224"
ERROR_LOG="$JOBS_DIR/log_error_samples.txt"
declare -A ERROR_IDX
if [ -f "$ERROR_LOG" ]; then
    while read -r line; do
        idx=$(echo "$line" | awk '{print $2}')
        ERROR_IDX[$idx]=1
    done < "$ERROR_LOG"
fi

# Loop over sample indices
for sample_idx in "${SAMPLE_INDICES[@]}"; do
    # Find all matching sample directories and pick the one with the largest time suffix
    sample_dirs=( $JOBS_DIR/sample_${sample_idx}_* )
    sample_dir=""
    if [ ${#sample_dirs[@]} -gt 0 ]; then
        # Sort by the numeric value after the last underscore and pick the largest
        sample_dir=$(printf "%s\n" "${sample_dirs[@]}" | awk -F'_' '{print $NF, $0}' | sort -n | tail -1 | cut -d' ' -f2-)
    fi
    if [ -z "$sample_dir" ]; then
        echo "Sample dir for $sample_idx not found, will run."
    else
        # Check result file
        result_file="$sample_dir/results/R$(printf "%04d" $sample_idx)/C$(printf "%06d" $sample_idx).json"
        log_txt="$sample_dir/log.txt"
        if [ -f "$result_file" ]; then
            echo "Sample $sample_idx already has result file. Skipping."
            continue
        fi
        if [[ -n "${ERROR_IDX[$sample_idx]}" ]]; then
            echo "Sample $sample_idx found in error log. Skipping."
            continue
        fi
        if [ -f "$log_txt" ] && grep -q "\[Incorrect Answer Pass\]" "$log_txt"; then
            echo "Sample $sample_idx has '[Incorrect Answer Pass]' in log.txt. Skipping."
            continue
        fi
    fi
    echo "Running with sample index $sample_idx"
    python main.py \
        --save_root "jobs/sample_${sample_idx}" \
        --num_noise_sample 100 \
        --model "deit_tiny_patch16_224" \
        --gpu 1ß \
        --noise_type "other" \
        --debug_index 0 \
        --detailed_save \
        --slightly_quiet_mode \
        --efficient_mode \
        --target_sample_idx $sample_idx
done

