"""
 Copyright 2023 [Anonymized]
 
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

      https://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 """

from typing import List

import tensorflow as tf
from tensorflow import Tensor

from tensorflow_similarity.layers import MetricEmbedding
from tensorflow.keras import layers


def build_similarity_output(encoder: Tensor,
                            similarity_dim: int = 256,
                            dense_layers: int = 0,
                            dense_contraction_rate: int = 1,
                            activation: str = 'swish',
                            output_dtype: str = tf.float32,
                           ) -> List[Tensor]:
    """Create output similarity head.

    Args:
        encoder: Encoder layer to build outputs on.

        similarity_dim: Final dimension of similarity output.

        num_dense_layers: Number of dense layers in similarity head.
            Other than the last layer, each dense is followed by
            a batchnorm and an activation layer of 'activation'.

        dense_contraction_rate: Ratio to contract the dim of each
            successful dense layer by.

        activation: Activation to use in dense layers, barring the last layer.

    Returns:
        The similarity output for the RetSim model.
    """
    outputs = []
    similarity_output = encoder

    if similarity_dim:

        layer_count = 0
        for _ in range(dense_layers - 1):
            layer_name = f'pre_similarity_{layer_count}'
            similarity_output = layers.Dense(
                similarity_dim, name=layer_name)(similarity_output)
            similarity_output = layers.Activation(
                activation)(similarity_output)

            layer_count += 1

        similarity_output = MetricEmbedding(
            similarity_dim,
            name='similarity',
            dtype=output_dtype
        )(similarity_output)
        outputs.append(similarity_output)

    return outputs


def dense_output(x: Tensor,
                 dim: int,
                 pre_dim: int,
                 num_layers: int,
                 activation: str,
                 output_activation: str,
                 norm_type: str,
                 name: str,
                 compression_factor: int = 1,
                 epsilon: float = 1e-6) -> Tensor:
    """Convolutional output head. """

    output = x

    if pre_dim:
        for i in range(num_layers):
            layer_name = f"pre_{name}{i}"
            output = layers.Dense(pre_dim,
                                  name=layer_name)(output)

            if norm_type == 'layer':
                output = layers.LayerNormalization(
                    epsilon=epsilon, name=layer_name + '_layernorm')(output)

            elif norm_type == 'batch':
                output = layers.BatchNormalization(
                    name=layer_name + '_batchnorm')(output)

            output = layers.Activation(activation,
                                       name=layer_name + '_act')(output)

            pre_dim //= compression_factor

    output = layers.Dense(dim,
                          activation=output_activation,
                          name=name)(output)
    return output
